-- ============================================================================
-- >>>>>>>>>>>>>>>>>>>>>>>>>> ADA COMPILATION UNIT <<<<<<<<<<<<<<<<<<<<<<<<<<<<
-- ============================================================================
--
-- NAME:        Asis_Error_Handling_Support
--
--              SPECIFICATION
-- 
-- AUTHOR:      Chuck Hobin
--              General Research Corporation
--
--                               CHANGE HISTORY
--
-- MM-DD-YY | Initials | Description
-- ----------------------------------------------------------------------------
-- <include SPR#, if applicable>
-- ============================================================================

with Asis;

package Asis_Error_Handling_Support is

-- Operations for logging ASIS-related errors in a standard format.  


-------------------------------------------------------------------------------
-- Log_Element_Error
--
-- This operation should be called when exceptions Asis_Failed or 
-- Asis_Inappropriate_Element are raised as a result of calling an element
-- query. 
--
-- Our general policy for handling these exceptions is enclose the Asis query
-- in a block statement, provide a local handler that calls this operation and
-- performs any needed error recovery, and then attempt to continue normal 
-- processing following the block.  Syntactic queries on elements of a known 
-- kind are exempt from this policy because they are unlikely to raise an 
-- exception.  All semantic queries, however, should have a local handler!  
-- These are more likely to fail, and we want to provide as much debugging 
-- information as possible when they do.
--
-- This operation logs
--
--   1) an Internal message indicating that an Asis-related error has 
--      occurred.
--   2) a Debug message containing the ASIS diagnosis.
--   3) a Debug message containing information about the Element.
--   4) a Debug message containing the comp unit and program unit names.
--
-- The handler may log additional Debug messages after calling this operation
-- to further explain the error.
--
-- Usage:
--
--      declare
--          An_Element : Asis.Element;
--      begin
--          x := Asis.<element-query> ( An_Element );
--      exception
--          when Asis.Asis_Failed | Asis.Asis_Inappropriate_Element =>
--               Asis_Error_Handling_Support.Log_Element_Error( An_Element, ... );
--
--               <optional Debug messages describing what the query was, the 
--                context, etc.>
--
--               <optional error recovery>
--      end;
--
-------------------------------------------------------------------------------

    procedure Log_Element_Error (The_Element : in Asis.Element; 
				 Compilation_Unit : in String; 
				 Program_Unit : in String);

-------------------------------------------------------------------------------
-- Log_Library_Open_Failure
--
-- This operation should be called when the exception Asis_Failed is raised
-- as a result of calling Asis.Libraries.Open.  The calling subprogram will
-- normally terminate afterwards.
--
-- Several Error_Kinds can be returned by Asis.Libraries.Open.  This
-- operation logs 
--
--   1) an Error message, since failures in this operation are typically
--      caused by the end-user.  The value of the Library_Reference_Name
--      parameter appears in this message; this should be the name
--      of the library as it is known to the end-user.
--   2) a Debug message containing the ASIS diagnosis.
--   3) a Debug message containing the Name and Parameter association
--      of the library.
--
-- Usage:
--
--      begin
--          Asis.Libraries.Open( The_Library );
--      exception
--          when Asis.Asis_Failed =>
--              Asis_Error_Handling_Support.Log_Library_Open_Failure( ... );
--      end;
--
-------------------------------------------------------------------------------

    procedure Log_Library_Open_Failure (Compilation_Unit : in String; 
					Program_Unit : in String; 
					The_Library : in Asis.Library; 
					Library_Reference_Name : in String);

-------------------------------------------------------------------------------
-- Element_Info
--
-- Returns a string of the form:
--
--    Element : Kind => "<element kind>", Line Number => <#>
--
-- If element is nil, the following is returned:
--
--    Element : <no information available>
--
-------------------------------------------------------------------------------

    function Element_Info (The_Element : in Asis.Element) return String;

end Asis_Error_Handling_Support;

