-- ============================================================================
-- >>>>>>>>>>>>>>>>>>>>>>>>>> ADA COMPILATION UNIT <<<<<<<<<<<<<<<<<<<<<<<<<<<<
-- ============================================================================
--
-- NAME:        Asis_Error_Handling_Support
--
--              BODY
-- 
-- AUTHOR:      Chuck Hobin
--              General Research Corporation
--
--                               CHANGE HISTORY
--
-- MM-DD-YY | Initials | Description
-- ----------------------------------------------------------------------------
-- <include SPR#, if applicable>
-- ============================================================================

with Asis_Debug_Support;
with Msg_Log;

package body Asis_Error_Handling_Support is

    package Asis_En renames Asis.Environment;
    package Asis_L renames Asis.Libraries;
    package Asis_Str renames Asis.Strings;

-------------------------------------------------------------------------------
-- LOCAL SUBPROGRAM DECLARATIONS
-------------------------------------------------------------------------------

    function Status_And_Diagnosis return String;

    function Log_Location 
		(Compilation_Unit : in String; Program_Unit : in String) 
		return String;

-------------------------------------------------------------------------------
-- EXPORTED SUBPROGRAM BODIES
-------------------------------------------------------------------------------

    procedure Log_Element_Error (The_Element : in Asis.Element; 
				 Compilation_Unit : in String; 
				 Program_Unit : in String) is
    begin
	Msg_Log.Put_Msg 
	   (Msg_Log.Internal, 
	    "ASIS error occurred - attempting to continue (diagnosis follows)");
	Msg_Log.Put_Msg (Msg_Log.Debug, Status_And_Diagnosis);
	Msg_Log.Put_Msg (Msg_Log.Debug, Element_Info (The_Element));
	Msg_Log.Put_Msg (Msg_Log.Debug, Log_Location 
					   (Compilation_Unit, Program_Unit));
    end Log_Element_Error;


-------------------------------------------------------------------------------

    procedure Log_Library_Open_Failure (Compilation_Unit : in String; 
					Program_Unit : in String; 
					The_Library : in Asis.Library; 
					Library_Reference_Name : in String) is
    begin
	Msg_Log.Put_Msg (Msg_Log.Error, 
			 "Unable to open " & Library_Reference_Name & 
			    " (diagnosis follows)");
	Msg_Log.Put_Msg (Msg_Log.Debug, Status_And_Diagnosis);
	Msg_Log.Put_Msg 
	   (Msg_Log.Debug, 
	    "ASIS library associations : Name => """ & 
	       Asis_Str.To_Standard_String (Asis_L.Name (The_Library)) & 
	       ", Parameters => " & 
	       Asis_Str.To_Standard_String (Asis_L.Parameters (The_Library)) & 
	       '"');
	Msg_Log.Put_Msg (Msg_Log.Debug, Log_Location 
					   (Compilation_Unit, Program_Unit));

    end Log_Library_Open_Failure;

-------------------------------------------------------------------------------    

    function Element_Info (The_Element : in Asis.Element) return String is

    begin
	if Asis.Elements.Is_Nil (The_Element) then
	    return "Element : <no information available>";
	else
	    return "Element : Kind => """ & 
		      Asis_Debug_Support.Element_Image (The_Element) & 
		      """, Line Number =>" & 
		      Asis.Line_Number'Image 
			 (Asis.Text.First_Line_Number (The_Element));
	end if;
    exception
	when Asis.Asis_Failed | Asis.Asis_Inappropriate_Element =>
	    return "Element : <no information available>";
    end Element_Info;

-------------------------------------------------------------------------------
-- LOCAL SUBPROGRAM BODIES
-------------------------------------------------------------------------------

    function Status_And_Diagnosis return String is

	First_Part : constant String := 
	   "ASIS message : Status => """ & 
	      Asis_En.Error_Kinds'Image (Asis_En.Status) & """, Diagnosis => ";
	Asis_Diagnosis : constant String := 
	   Asis_Str.To_Standard_String (Asis_En.Diagnosis);

    begin

	if Asis_Diagnosis = "" then
	    return First_Part & "<no diagnosis available>";
	else
	    return First_Part & '"' & Asis_Diagnosis & '"';
	end if;

    end Status_And_Diagnosis;

-------------------------------------------------------------------------------    

    function Log_Location 
		(Compilation_Unit : in String; Program_Unit : in String) 
		return String is
    begin
	return "Logged from : Compilation Unit => """ & Compilation_Unit & 
		  """, Program Unit => """ & Program_Unit & '"';
    end Log_Location;

end Asis_Error_Handling_Support;
