-- ============================================================================
-- >>>>>>>>>>>>>>>>>>>>>>>>>> ADA COMPILATION UNIT <<<<<<<<<<<<<<<<<<<<<<<<<<<<
-- ============================================================================
--
-- NAME:        Set_Simple_Sequential_Unbounded_Managed_Iterator
--
--              SPECIFICATION
-- 
-- AUTHOR:      Chuck Hobin
--
-- DATE:        16 September 1993
--
--                               CHANGE HISTORY
--
-- MM-DD-YY | Initials | Description
-- ----------------------------------------------------------------------------
--
-- ============================================================================

with Structure_Exceptions;

generic

    type Item is private;

package Set_Simple_Sequential_Unbounded_Managed_Iterator is

-- Based on the set structure presented in Booch, "Software Components
-- with Ada", Benjamin-Cummings, 1987, Chapter 10.

    type Set is limited private;

    procedure Copy (From_The_Set : in Set; To_The_Set : in out Set);

    -- Copy the items from one set to another set.
    --
    -- Raises : Overflow, if the destination set cannot grow large 
    --          enough to hold the source state.

    procedure Clear (The_Set : in out Set);

    -- Remove all items (if any) from the set and make the set empty.

    procedure Add (The_Item : in Item; To_The_Set : in out Set);

    -- Insert an item as a member of the set.
    --
    -- Raises : Item_Is_In_Set, if the item is already a member of
    --          the set.
    --
    -- Raises : Overflow, if the destination set cannot grow large
    --          enough to hold the new item.

    procedure Remove (The_Item : in Item; From_The_Set : in out Set);

    -- Remove an item as a member of the set.
    --
    -- Raises : Item_Is_Not_In_Set, if the item is currently not a
    --          member of the set.

    procedure Union (Of_The_Set : in Set; 
		     And_The_Set : in Set; 
		     To_The_Set : in out Set);

    -- Given two sets, form a set containing the items that are members
    -- of the first set or the second set.
    --
    -- Raises : Overflow, if the destination set cannot grow large
    --          enough to hold the new item.

    procedure Intersection (Of_The_Set : in Set; 
			    And_The_Set : in Set; 
			    To_The_Set : in out Set);

    -- Given two sets, form a set containing the items that are members
    -- of the first set and the second set.
    --
    -- Raises : Overflow, if the destination set cannot grow large
    --          enough to hold the new item.

    procedure Difference (Of_The_Set : in Set; 
			  And_The_Set : in Set; 
			  To_The_Set : in out Set);

    -- Given two sets, form a set containing the items that are members
    -- of the first set and not members of the second set.
    --
    -- Raises : Overflow, if the destination set cannot grow large
    --          enough to hold the new item.

    function Is_Equal (Left : in Set; Right : in Set) return Boolean;

    -- Return True if the two given sets have the same value.

    function Extent_Of (The_Set : in Set) return Natural;

    -- Return the current number of items in the set.

    function Is_Empty (The_Set : in Set) return Boolean;

    -- Return True if the set contains no items.

    function Is_A_Member 
		(The_Item : in Item; Of_The_Set : in Set) return Boolean;

    -- Return True if the given item is a member of the set.

    function Is_A_Subset (Left : in Set; Right : in Set) return Boolean;

    -- Return True if the first set is a subset of the second set.

    function Is_A_Proper_Subset (Left : in Set; Right : in Set) return Boolean;

    -- Return True if the first set is a proper subset of the second set.

    Overflow : exception;
    Item_Is_In_Set : exception;
    Item_Is_Not_In_Set : exception;

------------------------------------------------------------------------------
-- Passive Iterator

    generic

	with procedure Process (The_Item : in Item; Continue : out Boolean);

    procedure Iterate (Over_The_Set : in Set);

    -- Visits each item in the set.  Process is called with each item.
    -- If Process returns False for Continue, the remaining items
    -- in the set are skipped.

------------------------------------------------------------------------------
-- Active Iterator

    type Iterator is limited private;

    procedure Initialize (The_Iterator : in out Iterator; 
			  With_The_Set : in Set);

    -- Associate the iterator with the set object and position it
    -- at the first item in the set.

    function Is_Done (The_Iterator : in Iterator) return Boolean;

    -- Return True if the iterator has visited every item in the set.

    function Value_Of (The_Iterator : in Iterator) return Item;

    -- Return the current item in the set.
    --
    -- Raises : Iterator_Error, if the iterator has already visited 
    --          every item in the set.

    procedure Get_Next (The_Iterator : in out Iterator);

    -- Advance the iterator to the next item in the set.
    --
    -- Raises : Iterator_Error, if the iterator has already visited 
    --          every item in the set.


    Iterator_Error : exception renames Structure_Exceptions.Overflow;

private

    type Node;

    type Set is access Node;

    type Iterator is new Set;

end Set_Simple_Sequential_Unbounded_Managed_Iterator;
