--| +=========================================================================+
--| |                                                                         |
--| | NAMESPACE_VIEW - an Asis abstraction                                    |
--| |                                                                         |
--| | This package provides the operations to construct and manage the        |
--| | namespace view of a declarative region part.                            |
--| | The namespace view is simply the set of all entity definitions in the   |
--| | region along with some limited context in which the definitions         |
--| | occur (namely, the definitions' parent elements).  Definitions          |
--| | occurring in sub-regions of the region part are not included.           |
--| |                                                                         |
--| | All subprograms in this package that accept a namespace view as         |
--| | input interpret a null view as an empty view.                           |
--| |                                                                         |
--| | Greg Janee                                                              |
--| | General Research Corporation                                            |
--| |                                                                         |
--| +=========================================================================+

with Asis;
with Namespace_Scan;
with Region_Support;
with Text_Io;
with Unchecked_Deallocation;

package Namespace_View is

    package Asis_Id_Io renames Asis.Ids.Id_Io;

    type Definition is
	record
	    Element : Asis.Entity_Name_Definition;
	    Parent : Asis.Element;
	end record;

    type Definition_List is array (Integer range <>) of Definition;

    type Namespace_View is access Definition_List;

--| Construct builds a namespace view.  There are two versions:  the first
--| builds a view for an entire compilation unit, the second for a
--| declarative region part.  The resultant view is merged into The_View.
--|
--| Expand_Instantiations determines if instantiations are
--| expanded in place.  Trace determines if the namespace scan is
--| traced and written to Trace_File (if open) or to the default output file
--| (otherwise).

    Trace_File : Text_Io.File_Type renames Namespace_Scan.Trace_File;

    procedure Construct (The_View : in out Namespace_View; 
			 For_Unit : in Asis.Compilation_Unit; 
			 Expand_Instantiations : in Boolean := False; 
			 Trace : in Boolean := False);

    procedure Construct (The_View : in out Namespace_View; 
			 For_Region : in Region_Support.Region; 
			 Expand_Instantiations : in Boolean := False; 
			 Trace : in Boolean := False);

--| Free frees all memory held by a namespace view.  The_View is set to null.

    procedure Free is new Unchecked_Deallocation 
			     (Definition_List, Namespace_View);

--| Dump writes a textual representation of a namespace view to a file for
--| debugging purposes.

    procedure Dump (The_View : in Namespace_View; 
		    To_The_File : in Text_Io.File_Type := 
		       Text_Io.Standard_Output);

--| Merge merges two namespace views.  Definitions are kept in source code
--| order.

    function Merge (The_View : in Namespace_View; 
		    And_The_View : in Namespace_View) return Namespace_View;

--| Output and Input communicate a namespace view to and from a file, thus
--| giving the view persistence.  For Output, To_Control_File and
--| To_Id_File should already have been opened for writing.  Data is
--| written sequentially starting at the current positions in both files.
--| The files are left open.  For Input, From_Control_File and From_Id_File
--| should already have been opened for reading.  From_Control_File is
--| read sequentially starting at the current position.  The input is not
--| checked for errors.  The files are left open.

    procedure Output (The_View : in Namespace_View; 
		      To_Control_File : in Text_Io.File_Type; 
		      To_Id_File : in Asis_Id_Io.File_Type);

    function Input (From_Control_File : in Text_Io.File_Type; 
		    From_Id_File : in Asis_Id_Io.File_Type; 
		    Library : Asis.Library) return Namespace_View;

--| The following exception is raised if an Asis problem is encountered.

    Traversal_Error : exception;

end Namespace_View;
