--| +=========================================================================+
--| |                                                                         |
--| | MERGE_REGION_REFERENCES                                                 |
--| |                                                                         |
--| | Usage: merge_region_references <library> [-l|-s] <unit>                 |
--| |                                                                         |
--| | This program accepts two command line arguments, the name of an ASIS    |
--| | library and the name of a compilation unit contained therein, and it    |
--| | writes a dump of the unit's reference view to standard output.          |
--| | The view is constructed by merging together the individual reference    |
--| | views of the unit's declarative region parts.                           |
--| |                                                                         |
--| | The unit is looked up primarily as a library unit and secondarily       |
--| | as a secondary unit.  To force looking up a library unit, specify the   |
--| | "-l" option.  To force looking up a secondary unit, specify the "-s"    |
--| | option.                                                                 |
--| |                                                                         |
--| | Greg Janee                                                              |
--| | General Research Corporation                                            |
--| |                                                                         |
--| +=========================================================================+

with Asis;
with Command;
with Msg_Log;
with Reference_View;
with Reference_View_Structures;
with Region_Scan;
with Region_Support;

procedure Merge_Region_References is

--| Standard Asis renames...

    package Asis_Cu renames Asis.Compilation_Units;
    package Asis_E renames Asis.Elements;
    package Asis_En renames Asis.Environment;
    package Asis_L renames Asis.Libraries;
    package Asis_Str renames Asis.Strings;

    package Cli renames Command;

    Argument_Count : Integer := Cli.Argc - 1;
    Arguments : constant Cli.String_Ptr_Array := Cli.Arguments;
    Examine_Library_Units_Only : Boolean := False;
    Examine_Secondary_Units_Only : Boolean := False;
    The_Library : Asis.Library;
    The_Unit : Asis.Compilation_Unit;

    Find_Error : exception;
    Usage_Error : exception;

    type State_Information is
	record
	    View : Reference_View_Structures.Reference_View;
	end record;

    The_State : State_Information;

    procedure Pre_Operation (Region : in Region_Support.Region; 
			     Level : in Natural; 
			     Control : out Asis_E.Traverse_Control; 
			     State : in out State_Information) is
    begin
	Reference_View.Construct (The_State.View, Region);
	Control := Asis_E.Continue;
    end Pre_Operation;

    procedure Post_Operation (Region : in Region_Support.Region; 
			      Level : in Natural; 
			      Control : out Asis_E.Traverse_Control; 
			      State : in out State_Information) is
    begin
	Control := Asis_E.Continue;
    end Post_Operation;

    package The_Region_Scan is new Region_Scan (State_Information, 
						Pre_Operation, Post_Operation);

begin

    Msg_Log.Set_Program ("mrr");

    if Argument_Count = 3 then
	if Arguments (2).all = "-l" then
	    Examine_Library_Units_Only := True;
	elsif Arguments (2).all = "-s" then
	    Examine_Secondary_Units_Only := True;
	else
	    raise Usage_Error;
	end if;
    elsif Argument_Count /= 2 then
	raise Usage_Error;
    end if;

    Asis_En.Initialize;
    Asis_L.Associate (The_Library, Asis_Str.To_Asis_String (Arguments (1).all));
    Asis_L.Open (The_Library);

    if Examine_Library_Units_Only then
	The_Unit := Asis_Cu.Library_Unit 
		       (Arguments (Arguments'Last).all, The_Library);
	if Asis_Cu.Is_Nil (The_Unit) then
	    raise Find_Error;
	end if;
    elsif Examine_Secondary_Units_Only then
	The_Unit := Asis_Cu.Secondary_Unit 
		       (Arguments (Arguments'Last).all, The_Library);
	if Asis_Cu.Is_Nil (The_Unit) then
	    raise Find_Error;
	end if;
    else
	The_Unit := Asis_Cu.Library_Unit 
		       (Arguments (Arguments'Last).all, The_Library);
	if Asis_Cu.Is_Nil (The_Unit) then
	    The_Unit := Asis_Cu.Secondary_Unit 
			   (Arguments (Arguments'Last).all, The_Library);
	    if Asis_Cu.Is_Nil (The_Unit) then
		raise Find_Error;
	    end if;
	end if;
    end if;

    The_Region_Scan.Traverse_Unit (The_Unit, The_State);

    Reference_View.Dump (The_State.View);

    Asis_L.Close (The_Library);
    Asis_L.Dissociate (The_Library);
    Asis_En.Finalize;

exception

    when Usage_Error =>
	Msg_Log.Put_Msg 
	   (Msg_Log.Error, 
	    "usage is ""merge_region_references <library> [-l|-s] " & 
	       "<unit>""");

    when Find_Error =>
	Msg_Log.Put_Msg (Msg_Log.Error, "unit not found");

    when Asis.Asis_Inappropriate_Library =>
	Msg_Log.Put_Msg 
	   (Msg_Log.Error, 
	    "exception Asis_Inappropriate_Library raised; status is " & 
	       Asis_En.Error_Kinds'Image (Asis_En.Status) & 
	       "; diagnosis follows");
	Msg_Log.Put_Msg (Msg_Log.Error, 
			 Asis_Str.To_Standard_String (Asis_En.Diagnosis));

    when Asis.Asis_Failed =>
	Msg_Log.Put_Msg (Msg_Log.Error, 
			 "exception Asis_Failed raised; status is " & 
			    Asis_En.Error_Kinds'Image (Asis_En.Status) & 
			    "; diagnosis follows");
	Msg_Log.Put_Msg (Msg_Log.Error, 
			 Asis_Str.To_Standard_String (Asis_En.Diagnosis));

    when The_Region_Scan.Traversal_Error | Reference_View.Traversal_Error =>
	Msg_Log.Put_Msg (Msg_Log.Error, "exception Traversal_Error raised");

end Merge_Region_References;

