--| +=========================================================================+
--| |                                                                         |
--| | REFERENCE_SCAN.SCAN_CONSTRAINT (body)                                   |
--| |                                                                         |
--| | Greg Janee                                                              |
--| | General Research Corporation                                            |
--| |                                                                         |
--| +=========================================================================+

separate (Reference_Scan)
procedure Scan_Constraint (The_Constraint : in Asis.Constraint; 
			   The_Context : in Context) is

    Cuid : constant String := "Reference_Scan.Scan_Constraint";
    Puid : constant String := "Scan_Constraint";

    package Ctx renames Reference_Scan.Context_Support;
    package Error renames Reference_Scan.Error_Handling_Support;
    package Trace renames Reference_Scan.Trace_Support;

--| +-------------------------------------------------------------------------+
--| | SCAN_DISCRIMINANT_CONSTRAINT (local)                                    |
--| +-------------------------------------------------------------------------+

    procedure Scan_Discriminant_Constraint (The_Constraint : in Asis.Constraint; 
					    The_Context : in Context) is

	Puid : constant String := "Scan_Discriminant_Constraint";

    begin

	declare
	    Arguments : constant Asis.Discriminant_Association_List := 
	       Asis_Td.Discriminant_Associations 
		  (The_Constraint, 
		   Reference_Scan.Normalize_Discriminant_Constraint_Components);
	begin

	    if Trace.On then
		Trace.Log (The_Constraint, The_Context);
		Trace.Add_Level (Arguments'Length);
	    end if;

	    Reference_Scan.Scan_Discriminant_Association_List 
	       (Arguments, The_Context);

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Constraint);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Constraint);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Constraint);
	    raise;

    end Scan_Discriminant_Constraint;

--| +-------------------------------------------------------------------------+
--| | SCAN_FIXED_POINT_CONSTRAINT (local)                                     |
--| +-------------------------------------------------------------------------+

    procedure Scan_Fixed_Point_Constraint (The_Constraint : in Asis.Constraint; 
					   The_Context : in Context) is

	Puid : constant String := "Scan_Fixed_Point_Constraint";

    begin

	declare
	    Accuracy : Asis.Expression := 
	       Asis_Td.Fixed_Accuracy_Definition (The_Constraint);
	    The_Range : Asis.Range_Constraint := 
	       Asis_Td.Fixed_Point_Range_Constraint (The_Constraint);
	begin

	    if Trace.On then
		Trace.Log (The_Constraint, The_Context);
		Trace.Add_Level (1 + Reference_Scan.One_If_Present (The_Range));
	    end if;

	    Reference_Scan.Scan_Expression 
	       (Accuracy, Ctx.Set (The_Context, 
				   Basic_Context => Rvs.A_Read, 
				   Weight => 5, 
				   Add_Data_Access_Context => 
				      Rvs.A_Fixed_Accuracy_Definition));

	    if not Asis_E.Is_Nil (The_Range) then
		Reference_Scan.Scan_Constraint (The_Range, The_Context);
	    end if;

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Constraint);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Constraint);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Constraint);
	    raise;

    end Scan_Fixed_Point_Constraint;

--| +-------------------------------------------------------------------------+
--| | SCAN_FLOATING_POINT_CONSTRAINT (local)                                  |
--| +-------------------------------------------------------------------------+

    procedure Scan_Floating_Point_Constraint 
		 (The_Constraint : in Asis.Constraint; 
		  The_Context : in Context) is

	Puid : constant String := "Scan_Floating_Point_Constraint";

    begin

	declare
	    Accuracy : Asis.Expression := 
	       Asis_Td.Floating_Accuracy_Definition (The_Constraint);
	    The_Range : Asis.Range_Constraint := 
	       Asis_Td.Floating_Point_Range_Constraint (The_Constraint);
	begin

	    if Trace.On then
		Trace.Log (The_Constraint, The_Context);
		Trace.Add_Level (1 + Reference_Scan.One_If_Present (The_Range));
	    end if;

	    Reference_Scan.Scan_Expression 
	       (Accuracy, Ctx.Set (The_Context, 
				   Basic_Context => Rvs.A_Read, 
				   Weight => 5, 
				   Add_Data_Access_Context => 
				      Rvs.A_Floating_Accuracy_Definition));

	    if not Asis_E.Is_Nil (The_Range) then
		Reference_Scan.Scan_Constraint (The_Range, The_Context);
	    end if;

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Constraint);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Constraint);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Constraint);
	    raise;

    end Scan_Floating_Point_Constraint;

--| +-------------------------------------------------------------------------+
--| | SCAN_INDEX_CONSTRAINT (local)                                           |
--| +-------------------------------------------------------------------------+

    procedure Scan_Index_Constraint (The_Constraint : in Asis.Constraint; 
				     The_Context : in Context) is

	Puid : constant String := "Scan_Index_Constraint";

    begin

	declare
	    Ranges : constant Asis.Discrete_Range_List := 
	       Asis_Td.Discrete_Ranges (The_Constraint);
	begin

	    if Trace.On then
		Trace.Log (The_Constraint, The_Context);
		Trace.Add_Level (Ranges'Length);
	    end if;

	    Reference_Scan.Scan_Discrete_Range_List 
	       (Ranges, Ctx.Set (The_Context, 
				 Type_Mark_Context => Rvs.An_Index_Constraint));

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Constraint);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Constraint);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Constraint);
	    raise;

    end Scan_Index_Constraint;

--| +-------------------------------------------------------------------------+
--| | SCAN_RANGE_ATTRIBUTE (local)                                            |
--| +-------------------------------------------------------------------------+

    procedure Scan_Range_Attribute (The_Constraint : in Asis.Constraint; 
				    The_Context : in Context) is

	Puid : constant String := "Scan_Range_Attribute";

    begin

	declare
	    Attribute : Asis.Expression := 
	       Asis_Td.Range_Attribute (The_Constraint);
	begin

	    if Trace.On then
		Trace.Log (The_Constraint, The_Context);
		Trace.Add_Level (1);
	    end if;

	    Reference_Scan.Scan_Expression (Attribute, The_Context);

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Constraint);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Constraint);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Constraint);
	    raise;

    end Scan_Range_Attribute;

--| +-------------------------------------------------------------------------+
--| | SCAN_SIMPLE_RANGE (local)                                               |
--| +-------------------------------------------------------------------------+

    procedure Scan_Simple_Range (The_Constraint : in Asis.Constraint; 
				 The_Context : in Context) is

	Puid : constant String := "Scan_Simple_Range";

    begin

	declare
	    Lower_Bound : Asis.Expression := 
	       Asis_Td.Lower_Bound (The_Constraint);
	    Upper_Bound : Asis.Expression := 
	       Asis_Td.Upper_Bound (The_Constraint);
	begin

	    if Trace.On then
		Trace.Log (The_Constraint, The_Context);
		Trace.Add_Level (2);
	    end if;

	    Reference_Scan.Scan_Expression 
	       (Lower_Bound, Ctx.Set (The_Context, 
				      Basic_Context => Rvs.A_Read, 
				      Weight => 95, 
				      New_Weight => 5, 
				      Add_Data_Access_Context => Rvs.A_Range));

	    Reference_Scan.Scan_Expression 
	       (Upper_Bound, Ctx.Set (The_Context, 
				      Basic_Context => Rvs.A_Read, 
				      Weight => 95, 
				      New_Weight => 5, 
				      Add_Data_Access_Context => Rvs.A_Range));

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Constraint);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Constraint);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Constraint);
	    raise;

    end Scan_Simple_Range;

--| +-------------------------------------------------------------------------+
--| | SCAN_CONSTRAINT (exported)                                              |
--| +-------------------------------------------------------------------------+

begin

    case Asis_Td.Constraint_Kind (The_Constraint) is
	when Asis_Td.A_Simple_Range =>
	    Scan_Simple_Range (The_Constraint, The_Context);
	when Asis_Td.A_Range_Attribute =>
	    Scan_Range_Attribute (The_Constraint, The_Context);
	when Asis_Td.A_Floating_Point_Constraint =>
	    Scan_Floating_Point_Constraint (The_Constraint, The_Context);
	when Asis_Td.A_Fixed_Point_Constraint =>
	    Scan_Fixed_Point_Constraint (The_Constraint, The_Context);
	when Asis_Td.An_Index_Constraint =>
	    Scan_Index_Constraint (The_Constraint, The_Context);
	when Asis_Td.A_Discriminant_Constraint =>
	    Scan_Discriminant_Constraint (The_Constraint, The_Context);
	when Asis_Td.Not_A_Constraint =>
	    Error.Log (Error.An_Unhandled_Case, Cuid, Puid, The_Constraint);
	    if Reference_Scan.Raise_Exception_On_Unhandled_Case then
		raise Traversal_Error;
	    else
		if Trace.On then
		    Trace.Log (The_Constraint, The_Context);
		end if;
	    end if;
    end case;

end Scan_Constraint;
