--| +=========================================================================+
--| |                                                                         |
--| | REFERENCE_SCAN.SCAN_REPRESENTATION_CLAUSE (body)                        |
--| |                                                                         |
--| | Greg Janee                                                              |
--| | General Research Corporation                                            |
--| |                                                                         |
--| +=========================================================================+

separate (Reference_Scan)
procedure Scan_Representation_Clause 
	     (The_Clause : in Asis.Representation_Clause; 
	      The_Context : in Context) is

    Cuid : constant String := "Reference_Scan.Scan_Representation_Clause";
    Puid : constant String := "Scan_Representation_Clause";

    package Ctx renames Reference_Scan.Context_Support;
    package Error renames Reference_Scan.Error_Handling_Support;
    package Trace renames Reference_Scan.Trace_Support;

--| +-------------------------------------------------------------------------+
--| | SCAN_ADDRESS_CLAUSE (local)                                             |
--| +-------------------------------------------------------------------------+

    procedure Scan_Address_Clause (The_Clause : in Asis.Representation_Clause; 
				   The_Context : in Context) is

	Puid : constant String := "Scan_Address_Clause";

    begin

	declare
	    Address : Asis.Expression := 
	       Asis_Rc.Address_Clause_Simple_Expression (The_Clause);
	    Object_Name : Asis.Simple_Name := 
	       Asis_Rc.Address_Clause_Simple_Name (The_Clause);
	begin

	    if Trace.On then
		Trace.Log (The_Clause, The_Context);
		Trace.Add_Level (2);
	    end if;

	    Reference_Scan.Scan_Expression 
	       (Object_Name, Ctx.Set (The_Context, 
				      Basic_Context => Rvs.An_Address_Clause, 
				      Weight => 90));

	    Reference_Scan.Scan_Expression 
	       (Address, Ctx.Set 
			    (The_Context, 
			     Basic_Context => Rvs.A_Read, 
			     Weight => 5, 
			     Add_Data_Access_Context => Rvs.An_Address_Clause));

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Clause);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Clause);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Clause);
	    raise;

    end Scan_Address_Clause;

--| +-------------------------------------------------------------------------+
--| | SCAN_ENUMERATION_REPRESENTATION_CLAUSE (local)                          |
--| +-------------------------------------------------------------------------+

    procedure Scan_Enumeration_Representation_Clause 
		 (The_Clause : in Asis.Representation_Clause; 
		  The_Context : in Context) is

	Puid : constant String := "Scan_Enumeration_Representation_Clause";

    begin

	declare
	    Aggregate : Asis.Expression := 
	       Asis_Rc.Enumeration_Representation_Clause_Aggregate (The_Clause);
	    Type_Name : Asis.Simple_Name := 
	       Asis_Rc.Enumeration_Representation_Clause_Type_Simple_Name 
		  (The_Clause);
	begin

	    if Trace.On then
		Trace.Log (The_Clause, The_Context);
		Trace.Add_Level (2);
	    end if;

	    Reference_Scan.Scan_Expression 
	       (Type_Name, Ctx.Set (The_Context, 
				    Basic_Context => 
				       Rvs.An_Enumeration_Representation_Clause, 
				    Weight => 90));

	    Reference_Scan.Scan_Expression 
	       (Aggregate, Ctx.Set (The_Context, 
				    Basic_Context => 
				       Rvs.An_Enumeration_Representation_Clause, 
				    Weight => 90));

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Clause);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Clause);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Clause);
	    raise;

    end Scan_Enumeration_Representation_Clause;

--| +-------------------------------------------------------------------------+
--| | SCAN_LENGTH_CLAUSE (local)                                              |
--| +-------------------------------------------------------------------------+

    procedure Scan_Length_Clause (The_Clause : in Asis.Representation_Clause; 
				  The_Context : in Context) is

	Puid : constant String := "Scan_Length_Clause";

    begin

	declare
	    Attribute : Asis.Expression := 
	       Asis_Rc.Length_Clause_Attribute (The_Clause);
	    Length : Asis.Expression := 
	       Asis_Rc.Length_Clause_Simple_Expression (The_Clause);
	begin

	    if Trace.On then
		Trace.Log (The_Clause, The_Context);
		Trace.Add_Level (2);
	    end if;

	    Reference_Scan.Scan_Expression 
	       (Attribute, Ctx.Set (The_Context, 
				    Basic_Context => Rvs.A_Length_Clause, 
				    Weight => 92));

	    Reference_Scan.Scan_Expression 
	       (Length, Ctx.Set 
			   (The_Context, 
			    Basic_Context => Rvs.A_Read, 
			    Weight => 5, 
			    Add_Data_Access_Context => Rvs.A_Length_Clause));

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Clause);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Clause);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Clause);
	    raise;

    end Scan_Length_Clause;

--| +-------------------------------------------------------------------------+
--| | SCAN_RECORD_REPRESENTATION_CLAUSE (local)                               |
--| +-------------------------------------------------------------------------+

    procedure Scan_Record_Representation_Clause 
		 (The_Clause : in Asis.Representation_Clause; 
		  The_Context : in Context) is

	Puid : constant String := "Scan_Record_Representation_Clause";

    begin

	declare
	    Alignment : Asis.Expression := 
	       Asis_Rc.Record_Representation_Clause_Alignment_Clause_Expression 
		  (The_Clause);
	    Component_Clauses : constant Asis.Component_Clause_List := 
	       Asis_Rc.Component_Clauses 
		  (The_Clause, Reference_Scan.
				  Include_Record_Representation_Clause_Pragmas);
	    Pragmas : constant Asis.Element_List := 
	       Asis_Rc.Record_Representation_Clause_Alignment_Clause_Pragmas 
		  (The_Clause);
	    Type_Name : Asis.Simple_Name := 
	       Asis_Rc.Record_Representation_Clause_Type_Simple_Name 
		  (The_Clause);
	begin

	    if Trace.On then
		Trace.Log (The_Clause, The_Context);
		Trace.Add_Level 
		   (1 + Reference_Scan.One_If_True 
			   (Reference_Scan.
			    Include_Record_Representation_Clause_Pragmas) * 
			Pragmas'Length + 
		    Reference_Scan.One_If_Present (Alignment) + 
		    Component_Clauses'Length);
	    end if;

	    Reference_Scan.Scan_Expression 
	       (Type_Name, Ctx.Set (The_Context, 
				    Basic_Context => 
				       Rvs.A_Record_Representation_Clause, 
				    Weight => 90));

	    if Reference_Scan.Include_Record_Representation_Clause_Pragmas then
		Reference_Scan.Scan_Pragma_List (Pragmas, The_Context);
	    end if;

	    if not Asis_E.Is_Nil (Alignment) then
		Reference_Scan.Scan_Expression 
		   (Alignment, Ctx.Set (The_Context, 
					Basic_Context => Rvs.A_Read, 
					Weight => 5, 
					Add_Data_Access_Context => 
					   Rvs.An_Alignment_Clause));
	    end if;

	    Reference_Scan.Scan_Component_Clause_List 
	       (Component_Clauses, The_Context);

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Clause);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Clause);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Clause);
	    raise;

    end Scan_Record_Representation_Clause;

--| +-------------------------------------------------------------------------+
--| | SCAN_RECORD_REPRESENTATION_CLAUSE_NON_REGION_SUBELEMENTS (local)        |
--| +-------------------------------------------------------------------------+

    procedure Scan_Record_Representation_Clause_Non_Region_Subelements 
		 (The_Clause : in Asis.Representation_Clause; 
		  The_Context : in Context) is

	Puid : constant String := 
	   "Scan_Record_Representation_Clause_Non_Region_Subelements";

    begin

	declare
	    Type_Name : Asis.Simple_Name := 
	       Asis_Rc.Record_Representation_Clause_Type_Simple_Name 
		  (The_Clause);
	begin

	    if Trace.On then
		Trace.Log (The_Clause, The_Context);
		Trace.Add_Level (1);
	    end if;

	    Reference_Scan.Scan_Expression 
	       (Type_Name, Ctx.Set (The_Context, 
				    Basic_Context => 
				       Rvs.A_Record_Representation_Clause, 
				    Weight => 90));

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Clause);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Clause);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Clause);
	    raise;

    end Scan_Record_Representation_Clause_Non_Region_Subelements;

--| +-------------------------------------------------------------------------+
--| | SCAN_REPRESENTATION_CLAUSE (exported)                                   |
--| +-------------------------------------------------------------------------+

begin

    case Asis_Rc.Kind (The_Clause) is
	when Asis_Rc.A_Length_Clause =>
	    Scan_Length_Clause (The_Clause, The_Context);
	when Asis_Rc.An_Enumeration_Representation_Clause =>
	    Scan_Enumeration_Representation_Clause (The_Clause, The_Context);
	when Asis_Rc.A_Record_Representation_Clause =>
	    if Reference_Scan.Obeying_Regions then
		Scan_Record_Representation_Clause_Non_Region_Subelements 
		   (The_Clause, The_Context);
	    else
		Scan_Record_Representation_Clause (The_Clause, The_Context);
	    end if;
	when Asis_Rc.An_Address_Clause =>
	    Scan_Address_Clause (The_Clause, The_Context);
	when Asis_Rc.Not_A_Representation_Clause =>
	    Error.Log (Error.An_Unhandled_Case, Cuid, Puid, The_Clause);
	    if Reference_Scan.Raise_Exception_On_Unhandled_Case then
		raise Traversal_Error;
	    else
		if Trace.On then
		    Trace.Log (The_Clause, The_Context);
		end if;
	    end if;
    end case;

end Scan_Representation_Clause;
