--| +=========================================================================+
--| |                                                                         |
--| | REFERENCE_SCAN.SCAN_TYPE_DEFINITION (body)                              |
--| |                                                                         |
--| | Greg Janee                                                              |
--| | General Research Corporation                                            |
--| |                                                                         |
--| +=========================================================================+

separate (Reference_Scan)
procedure Scan_Type_Definition (The_Definition : in Asis.Type_Definition; 
				The_Context : in Context) is

    Cuid : constant String := "Reference_Scan.Scan_Type_Definition";
    Puid : constant String := "Scan_Type_Definition";

    package Ctx renames Reference_Scan.Context_Support;
    package Error renames Reference_Scan.Error_Handling_Support;
    package Trace renames Reference_Scan.Trace_Support;

--| +-------------------------------------------------------------------------+
--| | SCAN_ACCESS_TYPE_DEFINITION (local)                                     |
--| +-------------------------------------------------------------------------+

    procedure Scan_Access_Type_Definition 
		 (The_Definition : in Asis.Type_Definition; 
		  The_Context : in Context) is

	Puid : constant String := "Scan_Access_Type_Definition";

    begin

	declare
	    The_Type : Asis.Subtype_Indication := 
	       Asis_Td.Access_To (The_Definition);
	begin

	    if Trace.On then
		Trace.Log (The_Definition, The_Context);
		Trace.Add_Level (1);
	    end if;

	    Reference_Scan.Scan_Subtype_Indication 
	       (The_Type, Ctx.Set (The_Context, 
				   Type_Mark_Context => 
				      Rvs.An_Access_Type_Definition));

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Definition);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Definition);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Definition);
	    raise;

    end Scan_Access_Type_Definition;

--| +-------------------------------------------------------------------------+
--| | SCAN_CONSTRAINED_ARRAY_TYPE_DEFINITION (local)                          |
--| +-------------------------------------------------------------------------+

    procedure Scan_Constrained_Array_Type_Definition 
		 (The_Definition : in Asis.Type_Definition; 
		  The_Context : in Context) is

	Puid : constant String := "Scan_Constrained_Array_Type_Definition";

    begin

	declare
	    Component_Type : Asis.Subtype_Indication := 
	       Asis_Td.Component_Subtype_Indication (The_Definition);
	    Constraint : Asis.Constraint := 
	       Asis_Td.Index_Constraint (The_Definition);
	begin

	    if Trace.On then
		Trace.Log (The_Definition, The_Context);
		Trace.Add_Level (2);
	    end if;

	    Reference_Scan.Scan_Constraint (Constraint, The_Context);

	    Reference_Scan.Scan_Subtype_Indication 
	       (Component_Type, Ctx.Set (The_Context, 
					 Type_Mark_Context => 
					    Rvs.An_Array_Definition));

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Definition);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Definition);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Definition);
	    raise;

    end Scan_Constrained_Array_Type_Definition;

--| +-------------------------------------------------------------------------+
--| | SCAN_DERIVED_TYPE_DEFINITION (local)                                    |
--| +-------------------------------------------------------------------------+

    procedure Scan_Derived_Type_Definition 
		 (The_Definition : in Asis.Type_Definition; 
		  The_Context : in Context) is

	Puid : constant String := "Scan_Derived_Type_Definition";

    begin

	declare
	    Parent_Type : Asis.Subtype_Indication := 
	       Asis_Td.Parent_Subtype (The_Definition);
	begin

	    if Trace.On then
		Trace.Log (The_Definition, The_Context);
		Trace.Add_Level (1);
	    end if;

	    Reference_Scan.Scan_Subtype_Indication 
	       (Parent_Type, Ctx.Set (The_Context, 
				      Type_Mark_Context => 
					 Rvs.A_Derived_Type_Definition));

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Definition);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Definition);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Definition);
	    raise;

    end Scan_Derived_Type_Definition;

--| +-------------------------------------------------------------------------+
--| | SCAN_ENUMERATION_TYPE_DEFINITION (local)                                |
--| +-------------------------------------------------------------------------+

    procedure Scan_Enumeration_Type_Definition 
		 (The_Definition : in Asis.Type_Definition; 
		  The_Context : in Context) is

	Puid : constant String := "Scan_Enumeration_Type_Definition";

    begin

	if Trace.On then
	    Trace.Log (The_Definition, The_Context);
	end if;

    end Scan_Enumeration_Type_Definition;

--| +-------------------------------------------------------------------------+
--| | SCAN_FIXED_TYPE_DEFINITION (local)                                      |
--| +-------------------------------------------------------------------------+

    procedure Scan_Fixed_Type_Definition 
		 (The_Definition : in Asis.Type_Definition; 
		  The_Context : in Context) is

	Puid : constant String := "Scan_Fixed_Type_Definition";

    begin

	declare
	    Constraint : Asis.Constraint := 
	       Asis_Td.Real_Type_Constraint (The_Definition);
	begin

	    if Trace.On then
		Trace.Log (The_Definition, The_Context);
		Trace.Add_Level (1);
	    end if;

	    Reference_Scan.Scan_Constraint (Constraint, The_Context);

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Definition);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Definition);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Definition);
	    raise;

    end Scan_Fixed_Type_Definition;

--| +-------------------------------------------------------------------------+
--| | SCAN_FLOAT_TYPE_DEFINITION (local)                                      |
--| +-------------------------------------------------------------------------+

    procedure Scan_Float_Type_Definition 
		 (The_Definition : in Asis.Type_Definition; 
		  The_Context : in Context) is

	Puid : constant String := "Scan_Float_Type_Definition";

    begin

	declare
	    Constraint : Asis.Constraint := 
	       Asis_Td.Real_Type_Constraint (The_Definition);
	begin

	    if Trace.On then
		Trace.Log (The_Definition, The_Context);
		Trace.Add_Level (1);
	    end if;

	    Reference_Scan.Scan_Constraint (Constraint, The_Context);

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Definition);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Definition);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Definition);
	    raise;

    end Scan_Float_Type_Definition;

--| +-------------------------------------------------------------------------+
--| | SCAN_GENERIC_ACCESS_TYPE_DEFINITION (local)                             |
--| +-------------------------------------------------------------------------+

    procedure Scan_Generic_Access_Type_Definition 
		 (The_Definition : in Asis.Type_Definition; 
		  The_Context : in Context) is

	Puid : constant String := "Scan_Generic_Access_Type_Definition";

    begin

	declare
	    The_Type : Asis.Subtype_Indication := 
	       Asis_Td.Access_To (The_Definition);
	begin

	    if Trace.On then
		Trace.Log (The_Definition, The_Context);
		Trace.Add_Level (1);
	    end if;

	    Reference_Scan.Scan_Subtype_Indication 
	       (The_Type, Ctx.Set (The_Context, 
				   Type_Mark_Context => 
				      Rvs.An_Access_Type_Definition));

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Definition);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Definition);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Definition);
	    raise;

    end Scan_Generic_Access_Type_Definition;

--| +-------------------------------------------------------------------------+
--| | SCAN_GENERIC_CONSTRAINED_ARRAY_TYPE_DEFINITION (local)                  |
--| +-------------------------------------------------------------------------+

    procedure Scan_Generic_Constrained_Array_Type_Definition 
		 (The_Definition : in Asis.Type_Definition; 
		  The_Context : in Context) is

	Puid : constant String := 
	   "Scan_Generic_Constrained_Array_Type_Definition";

    begin

	declare
	    Component_Type : Asis.Subtype_Indication := 
	       Asis_Td.Component_Subtype_Indication (The_Definition);
	    Constraint : Asis.Constraint := 
	       Asis_Td.Index_Constraint (The_Definition);
	begin

	    if Trace.On then
		Trace.Log (The_Definition, The_Context);
		Trace.Add_Level (2);
	    end if;

	    Reference_Scan.Scan_Constraint (Constraint, The_Context);

	    Reference_Scan.Scan_Subtype_Indication 
	       (Component_Type, Ctx.Set (The_Context, 
					 Type_Mark_Context => 
					    Rvs.An_Array_Definition));

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Definition);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Definition);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Definition);
	    raise;

    end Scan_Generic_Constrained_Array_Type_Definition;

--| +-------------------------------------------------------------------------+
--| | SCAN_GENERIC_DISCRETE_SUBTYPE_DEFINITION (local)                        |
--| +-------------------------------------------------------------------------+

    procedure Scan_Generic_Discrete_Subtype_Definition 
		 (The_Definition : in Asis.Type_Definition; 
		  The_Context : in Context) is

	Puid : constant String := "Scan_Generic_Discrete_Subtype_Definition";

    begin

	if Trace.On then
	    Trace.Log (The_Definition, The_Context);
	end if;

    end Scan_Generic_Discrete_Subtype_Definition;

--| +-------------------------------------------------------------------------+
--| | SCAN_GENERIC_FIXED_SUBTYPE_DEFINITION (local)                           |
--| +-------------------------------------------------------------------------+

    procedure Scan_Generic_Fixed_Subtype_Definition 
		 (The_Definition : in Asis.Type_Definition; 
		  The_Context : in Context) is

	Puid : constant String := "Scan_Generic_Fixed_Subtype_Definition";

    begin

	if Trace.On then
	    Trace.Log (The_Definition, The_Context);
	end if;

    end Scan_Generic_Fixed_Subtype_Definition;

--| +-------------------------------------------------------------------------+
--| | SCAN_GENERIC_FLOAT_SUBTYPE_DEFINITION (local)                           |
--| +-------------------------------------------------------------------------+

    procedure Scan_Generic_Float_Subtype_Definition 
		 (The_Definition : in Asis.Type_Definition; 
		  The_Context : in Context) is

	Puid : constant String := "Scan_Generic_Float_Subtype_Definition";

    begin

	if Trace.On then
	    Trace.Log (The_Definition, The_Context);
	end if;

    end Scan_Generic_Float_Subtype_Definition;

--| +-------------------------------------------------------------------------+
--| | SCAN_GENERIC_INTEGER_SUBTYPE_DEFINITION (local)                         |
--| +-------------------------------------------------------------------------+

    procedure Scan_Generic_Integer_Subtype_Definition 
		 (The_Definition : in Asis.Type_Definition; 
		  The_Context : in Context) is

	Puid : constant String := "Scan_Generic_Integer_Subtype_Definition";

    begin

	if Trace.On then
	    Trace.Log (The_Definition, The_Context);
	end if;

    end Scan_Generic_Integer_Subtype_Definition;

--| +-------------------------------------------------------------------------+
--| | SCAN_GENERIC_LIMITED_PRIVATE_TYPE_DEFINITION (local)                    |
--| +-------------------------------------------------------------------------+

    procedure Scan_Generic_Limited_Private_Type_Definition 
		 (The_Definition : in Asis.Type_Definition; 
		  The_Context : in Context) is

	Puid : constant String := 
	   "Scan_Generic_Limited_Private_Type_Definition";

    begin

	if Trace.On then
	    Trace.Log (The_Definition, The_Context);
	end if;

    end Scan_Generic_Limited_Private_Type_Definition;

--| +-------------------------------------------------------------------------+
--| | SCAN_GENERIC_PRIVATE_TYPE_DEFINITION (local)                            |
--| +-------------------------------------------------------------------------+

    procedure Scan_Generic_Private_Type_Definition 
		 (The_Definition : in Asis.Type_Definition; 
		  The_Context : in Context) is

	Puid : constant String := "Scan_Generic_Private_Type_Definition";

    begin

	if Trace.On then
	    Trace.Log (The_Definition, The_Context);
	end if;

    end Scan_Generic_Private_Type_Definition;

--| +-------------------------------------------------------------------------+
--| | SCAN_GENERIC_UNCONSTRAINED_ARRAY_TYPE_DEFINITION (local)                |
--| +-------------------------------------------------------------------------+

    procedure Scan_Generic_Unconstrained_Array_Type_Definition 
		 (The_Definition : in Asis.Type_Definition; 
		  The_Context : in Context) is

	Puid : constant String := 
	   "Scan_Generic_Unconstrained_Array_Type_Definition";

    begin

	declare
	    Component_Type : Asis.Subtype_Indication := 
	       Asis_Td.Component_Subtype_Indication (The_Definition);
	    Indexes : constant Asis.Expression_List := 
	       Asis_Td.Index_Subtype_Definitions (The_Definition);
	begin

	    if Trace.On then
		Trace.Log (The_Definition, The_Context);
		Trace.Add_Level (Indexes'Length + 1);
	    end if;

	    Reference_Scan.Scan_Expression_List 
	       (Indexes, Ctx.Set (The_Context, 
				  Basic_Context => Rvs.A_Type_Mark, 
				  Weight => 90, 
				  Type_Mark_Context => 
				     Rvs.An_Index_Subtype_Definition));

	    Reference_Scan.Scan_Subtype_Indication 
	       (Component_Type, Ctx.Set (The_Context, 
					 Type_Mark_Context => 
					    Rvs.An_Array_Definition));

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Definition);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Definition);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Definition);
	    raise;

    end Scan_Generic_Unconstrained_Array_Type_Definition;

--| +-------------------------------------------------------------------------+
--| | SCAN_INTEGER_TYPE_DEFINITION (local)                                    |
--| +-------------------------------------------------------------------------+

    procedure Scan_Integer_Type_Definition 
		 (The_Definition : in Asis.Type_Definition; 
		  The_Context : in Context) is

	Puid : constant String := "Scan_Integer_Type_Definition";

    begin

	declare
	    Constraint : Asis.Range_Constraint := 
	       Asis_Td.Integer_Constraint (The_Definition);
	begin

	    if Trace.On then
		Trace.Log (The_Definition, The_Context);
		Trace.Add_Level (1);
	    end if;

	    Reference_Scan.Scan_Constraint (Constraint, The_Context);

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Definition);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Definition);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Definition);
	    raise;

    end Scan_Integer_Type_Definition;

--| +-------------------------------------------------------------------------+
--| | SCAN_LIMITED_PRIVATE_TYPE_DEFINITION (local)                            |
--| +-------------------------------------------------------------------------+

    procedure Scan_Limited_Private_Type_Definition 
		 (The_Definition : in Asis.Type_Definition; 
		  The_Context : in Context) is

	Puid : constant String := "Scan_Limited_Private_Type_Definition";

    begin

	if Trace.On then
	    Trace.Log (The_Definition, The_Context);
	end if;

    end Scan_Limited_Private_Type_Definition;

--| +-------------------------------------------------------------------------+
--| | SCAN_PRIVATE_TYPE_DEFINITION (local)                                    |
--| +-------------------------------------------------------------------------+

    procedure Scan_Private_Type_Definition 
		 (The_Definition : in Asis.Type_Definition; 
		  The_Context : in Context) is

	Puid : constant String := "Scan_Private_Type_Definition";

    begin

	if Trace.On then
	    Trace.Log (The_Definition, The_Context);
	end if;

    end Scan_Private_Type_Definition;

--| +-------------------------------------------------------------------------+
--| | SCAN_RECORD_TYPE_DEFINITION (local)                                     |
--| +-------------------------------------------------------------------------+

    procedure Scan_Record_Type_Definition 
		 (The_Definition : in Asis.Type_Definition; 
		  The_Context : in Context) is

	Puid : constant String := "Scan_Record_Type_Definition";

    begin

	declare
	    Components : constant Asis.Record_Component_List := 
	       Asis_Td.Record_Components 
		  (The_Definition, Reference_Scan.
				      Include_Record_Type_Definition_Pragmas);
	    Implicit_Components : constant Asis.Record_Component_List := 
	       Asis_Td.Implicit_Components (The_Definition);
	begin

	    if Trace.On then
		Trace.Log (The_Definition, The_Context);
		Trace.Add_Level 
		   (Reference_Scan.One_If_True 
		       (Reference_Scan.Include_Implicit_Record_Components) * 
		    Implicit_Components'Length + Components'Length);
	    end if;

	    if Reference_Scan.Include_Implicit_Record_Components then
		Reference_Scan.Scan_Declaration_List 
		   (Implicit_Components, The_Context);
	    end if;

	    Reference_Scan.Scan_Record_Component_List (Components, The_Context);

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Definition);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Definition);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Definition);
	    raise;

    end Scan_Record_Type_Definition;

--| +-------------------------------------------------------------------------+
--| | SCAN_SUBTYPE_DEFINITION (local)                                         |
--| +-------------------------------------------------------------------------+

    procedure Scan_Subtype_Definition (The_Definition : in Asis.Type_Definition; 
				       The_Context : in Context) is

	Puid : constant String := "Scan_Subtype_Definition";

    begin

	declare
	    The_Subtype : Asis.Subtype_Indication := 
	       Asis_Td.Subtype_Definition_Subtype_Indication (The_Definition);
	begin

	    if Trace.On then
		Trace.Log (The_Definition, The_Context);
		Trace.Add_Level (1);
	    end if;

	    Reference_Scan.Scan_Subtype_Indication (The_Subtype, The_Context);

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Definition);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Definition);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Definition);
	    raise;

    end Scan_Subtype_Definition;

--| +-------------------------------------------------------------------------+
--| | SCAN_TASK_TYPE_DEFINITION (local)                                       |
--| +-------------------------------------------------------------------------+

    procedure Scan_Task_Type_Definition 
		 (The_Definition : in Asis.Type_Definition; 
		  The_Context : in Context) is

	Puid : constant String := "Scan_Task_Type_Definition";

    begin

	declare
	    Declarations : constant Asis.Declarative_Item_List := 
	       Asis_Td.Task_Type_Declarative_Items 
		  (The_Definition, Reference_Scan.
				      Include_Task_Type_Definition_Pragmas);
	begin

	    if Trace.On then
		Trace.Log (The_Definition, The_Context);
		Trace.Add_Level (Declarations'Length);
	    end if;

	    Reference_Scan.Scan_Declarative_Item_List 
	       (Declarations, The_Context);

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Definition);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Definition);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Definition);
	    raise;

    end Scan_Task_Type_Definition;

--| +-------------------------------------------------------------------------+
--| | SCAN_UNCONSTRAINED_ARRAY_TYPE_DEFINITION (local)                        |
--| +-------------------------------------------------------------------------+

    procedure Scan_Unconstrained_Array_Type_Definition 
		 (The_Definition : in Asis.Type_Definition; 
		  The_Context : in Context) is

	Puid : constant String := "Scan_Unconstrained_Array_Type_Definition";

    begin

	declare
	    Component_Type : Asis.Subtype_Indication := 
	       Asis_Td.Component_Subtype_Indication (The_Definition);
	    Indexes : constant Asis.Expression_List := 
	       Asis_Td.Index_Subtype_Definitions (The_Definition);
	begin

	    if Trace.On then
		Trace.Log (The_Definition, The_Context);
		Trace.Add_Level (Indexes'Length + 1);
	    end if;

	    Reference_Scan.Scan_Expression_List 
	       (Indexes, Ctx.Set (The_Context, 
				  Basic_Context => Rvs.A_Type_Mark, 
				  Weight => 90, 
				  Type_Mark_Context => 
				     Rvs.An_Index_Subtype_Definition));

	    Reference_Scan.Scan_Subtype_Indication 
	       (Component_Type, Ctx.Set (The_Context, 
					 Type_Mark_Context => 
					    Rvs.An_Array_Definition));

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Definition);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Definition);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Definition);
	    raise;

    end Scan_Unconstrained_Array_Type_Definition;

--| +-------------------------------------------------------------------------+
--| | SCAN_UNIVERSAL_TYPE_DEFINITION (local)                                  |
--| +-------------------------------------------------------------------------+

    procedure Scan_Universal_Type_Definition 
		 (The_Definition : in Asis.Type_Definition; 
		  The_Context : in Context) is

	Puid : constant String := "Scan_Universal_Type_Definition";

    begin

	if Trace.On then
	    Trace.Log (The_Definition, The_Context);
	end if;

    end Scan_Universal_Type_Definition;

--| +-------------------------------------------------------------------------+
--| | SCAN_TYPE_DEFINITION (exported)                                         |
--| +-------------------------------------------------------------------------+

begin

    case Asis_Td.Kind (The_Definition) is
	when Asis_Td.A_Subtype_Definition =>
	    Scan_Subtype_Definition (The_Definition, The_Context);
	when Asis_Td.An_Enumeration_Type_Definition =>
	    Scan_Enumeration_Type_Definition (The_Definition, The_Context);
	when Asis_Td.An_Integer_Type_Definition =>
	    Scan_Integer_Type_Definition (The_Definition, The_Context);
	when Asis_Td.A_Float_Type_Definition =>
	    Scan_Float_Type_Definition (The_Definition, The_Context);
	when Asis_Td.A_Fixed_Type_Definition =>
	    Scan_Fixed_Type_Definition (The_Definition, The_Context);
	when Asis_Td.An_Array_Type_Definition =>
	    if Asis_Td.Is_Constrained_Array (The_Definition) then
		Scan_Constrained_Array_Type_Definition 
		   (The_Definition, The_Context);
	    else
		Scan_Unconstrained_Array_Type_Definition 
		   (The_Definition, The_Context);
	    end if;
	when Asis_Td.A_Record_Type_Definition =>
	    Scan_Record_Type_Definition (The_Definition, The_Context);
	when Asis_Td.An_Access_Type_Definition =>
	    Scan_Access_Type_Definition (The_Definition, The_Context);
	when Asis_Td.A_Derived_Type_Definition =>
	    Scan_Derived_Type_Definition (The_Definition, The_Context);
	when Asis_Td.A_Task_Type_Definition =>
	    Scan_Task_Type_Definition (The_Definition, The_Context);
	when Asis_Td.A_Private_Type_Definition =>
	    Scan_Private_Type_Definition (The_Definition, The_Context);
	when Asis_Td.A_Limited_Private_Type_Definition =>
	    Scan_Limited_Private_Type_Definition (The_Definition, The_Context);
	when Asis_Td.A_Generic_Discrete_Subtype_Definition =>
	    Scan_Generic_Discrete_Subtype_Definition 
	       (The_Definition, The_Context);
	when Asis_Td.A_Generic_Integer_Subtype_Definition =>
	    Scan_Generic_Integer_Subtype_Definition 
	       (The_Definition, The_Context);
	when Asis_Td.A_Generic_Float_Subtype_Definition =>
	    Scan_Generic_Float_Subtype_Definition (The_Definition, The_Context);
	when Asis_Td.A_Generic_Fixed_Subtype_Definition =>
	    Scan_Generic_Fixed_Subtype_Definition (The_Definition, The_Context);
	when Asis_Td.A_Generic_Array_Type_Definition =>
	    if Asis_Td.Is_Constrained_Array (The_Definition) then
		Scan_Generic_Constrained_Array_Type_Definition 
		   (The_Definition, The_Context);
	    else
		Scan_Generic_Unconstrained_Array_Type_Definition 
		   (The_Definition, The_Context);
	    end if;
	when Asis_Td.A_Generic_Access_Type_Definition =>
	    Scan_Generic_Access_Type_Definition (The_Definition, The_Context);
	when Asis_Td.A_Generic_Private_Type_Definition =>
	    Scan_Generic_Private_Type_Definition (The_Definition, The_Context);
	when Asis_Td.A_Generic_Limited_Private_Type_Definition =>
	    Scan_Generic_Limited_Private_Type_Definition 
	       (The_Definition, The_Context);
	when Asis_Td.A_Universal_Integer_Definition =>
	    Scan_Universal_Type_Definition (The_Definition, The_Context);
	when Asis_Td.A_Universal_Real_Definition =>
	    Scan_Universal_Type_Definition (The_Definition, The_Context);
	when Asis_Td.A_Universal_Fixed_Definition =>
	    Scan_Universal_Type_Definition (The_Definition, The_Context);
	when Asis_Td.Not_A_Type_Definition =>
	    Error.Log (Error.An_Unhandled_Case, Cuid, Puid, The_Definition);
	    if Reference_Scan.Raise_Exception_On_Unhandled_Case then
		raise Traversal_Error;
	    else
		if Trace.On then
		    Trace.Log (The_Definition, The_Context);
		end if;
	    end if;
    end case;

end Scan_Type_Definition;
