--| +=========================================================================+
--| |                                                                         |
--| | REFERENCE_VIEW_STRUCTURES                                               |
--| |                                                                         |
--| | This package defines the data structures that make up the reference     |
--| | view of a declarative region part.                                      |
--| |                                                                         |
--| | Greg Janee                                                              |
--| | General Research Corporation                                            |
--| |                                                                         |
--| +=========================================================================+

with Asis;

package Reference_View_Structures is

--| We start with the fundamental kinds of references.  The keywords below
--| have the following semantics.  Where more than one kind applies, the
--| more specific is used.
--|
--|     A_Pragma
--|         Any kind of reference occurring in a pragma.  (Note:  pragma
--|         names, formal pragma argument names, and pragma keywords
--|         do not generate references.  For example, the statement
--|         "pragma Suppress (Range_Check, On => X);" generates only a
--|         reference to X.)
--|
--|     A_Read
--|         A read of a named object's value.  Note that only a component of
--|         the object may actually be read.
--|
--|     An_Update
--|         An update to a named object's value.  Note that only
--|         a component of the object may actually be updated.
--|
--|     A_Read_And_Update
--|         A combined read and update of a named object.  This occurs when
--|         the object is the actual argument to an in/out parameter in
--|         a procedure call.  Note that only a component of the object
--|         may actually be read and/or updated.
--|
--|     An_Evaluate
--|         A reference to an enumeration literal that yields an
--|         enumeration value.
--|
--|     A_Type_Mark
--|         A type or subtype name.
--|
--|     A_Discriminant_Association
--|         A discriminant name in a named discriminant association.
--|
--|     A_Variant_Part_Discriminant
--|         The discriminant in a variant-defining case statement in a
--|         record type definition.
--|
--|     A_Selected_Component
--|         A discriminant or component of a record object or type.
--|
--|     A_Task_Entry_Selection
--|         A task name used as a prefix in the selection of an entry.
--|
--|     An_Expanded_Name
--|         A prefix in an expanded name.
--|
--|     An_Attribute
--|         A reference to an attribute of a named object or other entity.
--|         Note that the attribute name does not generate a reference.
--|         For example, the expression "X'Range" generates an attribute
--|         reference to X only.
--|
--|     A_Component_Association
--|         The discriminant or component name or enumeration literal in a
--|         named component association.  (An enumeration literal occurs
--|         in a component association in an enumeration representation
--|         clause.)
--|
--|     An_Exit_Loop_Name
--|         A loop name.
--|
--|     A_Goto_Target
--|         A statement label.
--|
--|     A_Procedure_Call
--|         The procedure name in a procedure call.
--|
--|     A_Parameter_Association
--|         The formal parameter name in a named parameter association.
--|
--|     A_Function_Call
--|         The function name in a function call.
--|
--|     A_Use_Clause
--|         A package name in a use clause that is located anywhere but in the
--|         context clause of a library unit.
--|
--|     A_Rename
--|         The target of a rename declaration.
--|
--|     An_Entry_Call
--|         The entry name in a task entry call.
--|
--|     An_Accept
--|         The entry name in an accept statement.
--|
--|     An_Abort
--|         A task name in an abort statement.
--|
--|     A_With_Clause
--|         A library unit name in a with clause.
--|
--|     A_Use_Context_Clause
--|         A package name in a use clause that is located in the context clause
--|         of a library unit.
--|
--|     A_Handle
--|         An exception name among the choice(s) heading an
--|         exception handler.
--|
--|     A_Raise
--|         The exception name in a named raise statement.
--|
--|     A_Generic_Formal_Subprogram_Default
--|         A subprogram, enumeration literal, or task entry used as the
--|         default value of a generic formal subprogram.
--|
--|     An_Instantiation
--|         A generic package or subprogram in an instantiation.
--|
--|     A_Generic_Association
--|         The formal subprogram, type, or object name in a named
--|         generic association.
--|
--|     A_Generic_Actual_Parameter
--|         A named object or other entity used as an actual parameter
--|         in an instantiation (unless the corresponding formal is
--|         an in-mode generic formal object, in which case the reference kind
--|         is A_Read).
--|
--|     A_Length_Clause
--|         The type name in a length representation clause.
--|
--|     An_Enumeration_Representation_Clause
--|         The type name in an enumeration representation clause.
--|
--|     A_Record_Representation_Clause
--|         The type name in a record representation clause.
--|
--|     A_Component_Clause
--|         A discriminant or component in a record representation clause.
--|
--|     An_Address_Clause
--|         A named object or other entity in an address representation
--|         clause.

    type Reference_Kinds_Or_Unknown is 
       (A_Pragma, A_Read, An_Update, A_Read_And_Update, An_Evaluate, 
	A_Type_Mark, A_Discriminant_Association, A_Variant_Part_Discriminant, 
	A_Selected_Component, A_Task_Entry_Selection, An_Expanded_Name, 
	An_Attribute, A_Component_Association, An_Exit_Loop_Name, 
	A_Goto_Target, A_Procedure_Call, A_Parameter_Association, 
	A_Function_Call, A_Use_Clause, A_Rename, An_Entry_Call, An_Accept, 
	An_Abort, A_With_Clause, A_Use_Context_Clause, A_Handle, 
	A_Raise, A_Generic_Formal_Subprogram_Default, An_Instantiation, 
	A_Generic_Association, A_Generic_Actual_Parameter, A_Length_Clause, 
	An_Enumeration_Representation_Clause, A_Record_Representation_Clause, 
	A_Component_Clause, An_Address_Clause, Unknown);

    subtype Reference_Kinds is Reference_Kinds_Or_Unknown 
				  range A_Pragma .. An_Address_Clause;

--| Reference kinds A_Read, An_Update, and A_Read_And_Update are qualified
--| by a Data_Access_Context, which is an array of
--| one or more contexts listed from most specific to least specific.
--| For example, the declaration
--|
--|     X : Integer := Foo (Y);
--|
--| yields a reference to Y of kind A_Read.  The data access context in this
--| case is (1 => An_Actual_Parameter, 2 => An_Object_Declaration).
--| The keywords below have the following semantics:
--|
--|     An_Object_Declaration
--|         The initial value expression in a defaulted object or constant
--|         declaration.
--|
--|     A_Number_Declaration
--|         The expression in a number declaration.
--|
--|     A_Range
--|         Either of the bound expressions in a range.
--|
--|     A_Floating_Accuracy_Definition
--|         The expression in a floating accuracy definition.
--|
--|     A_Fixed_Accuracy_Definition
--|         The expression in a fixed accuracy definition.
--|
--|     A_Component_Declaration
--|         The initial value expression in a defaulted record component
--|         declaration.
--|
--|     A_Discriminant_Specification
--|         The default value expression in a defaulted discriminant
--|         declaration.
--|
--|     A_Discriminant_Association
--|         The expression in a discriminant association.
--|
--|     A_Choice
--|         A choice expression.
--|
--|     An_Indexed_Component
--|         An index expression of an indexed component.
--|
--|     An_Attribute_Designator
--|         The expression in an attribute designator that requires an argument.
--|         (Note:  this is distinct from the actual argument to
--|         an attribute which is a function.)
--|
--|     A_Component_Association
--|         The expression in a component association.
--|
--|     An_Allocator
--|         The qualified expression in an allocator.
--|
--|     An_Assignment_Statement_Name
--|         The left hand side of an assignment statement.
--|
--|     An_Assignment_Statement_Expression
--|         The right hand side of an assignment statement.
--|
--|     An_If_Condition
--|         A condition expression in an if statement.
--|
--|     A_Case_Statement
--|         The branch expression in a case statement.
--|
--|     A_While_Loop_Condition
--|         The condition expression in a while loop.
--|
--|     An_Exit_Condition
--|         The condition expression in an exit statement.
--|
--|     A_Return_Statement
--|         The expression in a return statement.
--|
--|     A_Parameter_Specification
--|         The default expression in a defaulted parameter specification.
--|
--|     An_Actual_Parameter
--|         An actual argument in a subprogram call.
--|
--|     An_Entry_Index
--|         The index expression in an accept statement for an indexed
--|         task entry.
--|
--|     A_Delay_Statement
--|         The expression in a delay statement.
--|
--|     A_Select_Alternative
--|         The guard expression in a guarded select alternative.
--|
--|     A_Generic_Parameter_Declaration
--|         The default expression in a defaulted generic formal object
--|         declaration.
--|
--|     A_Generic_Actual_Parameter
--|         The actual parameter corresponding to an in-mode formal generic
--|         object in an instantiation.
--|
--|     A_Length_Clause
--|         The expression in a length representation clause.
--|
--|     An_Alignment_Clause
--|         The expression in an alignment representation clause.
--|
--|     A_Component_Clause
--|         Either of the expressions in a record component representation
--|         clause.
--|
--|     An_Address_Clause
--|         The expression in an address representation clause.
--|
--|     A_Dereference
--|         A dereference of an access object.

    type Data_Access_Context_Kinds is 
       (An_Object_Declaration, A_Number_Declaration, A_Range, 
	A_Floating_Accuracy_Definition, A_Fixed_Accuracy_Definition, 
	A_Component_Declaration, A_Discriminant_Specification, 
	A_Discriminant_Association, A_Choice, An_Indexed_Component, 
	An_Attribute_Designator, A_Component_Association, An_Allocator, 
	An_Assignment_Statement_Name, An_Assignment_Statement_Expression, 
	An_If_Condition, A_Case_Statement, A_While_Loop_Condition, 
	An_Exit_Condition, A_Return_Statement, A_Parameter_Specification, 
	An_Actual_Parameter, An_Entry_Index, A_Delay_Statement, 
	A_Select_Alternative, A_Generic_Parameter_Declaration, 
	A_Generic_Actual_Parameter, A_Length_Clause, An_Alignment_Clause, 
	A_Component_Clause, An_Address_Clause, A_Dereference);

    type Data_Access_Context is 
       array (Integer range <>) of Data_Access_Context_Kinds;

    type Data_Access_Context_Ptr is access Data_Access_Context;

--| Reference kind A_Type_Mark is qualified by a single
--| Type_Mark_Context_Kinds.  The keywords below have the following semantics:
--|
--|     An_Object_Declaration
--|         The subtype indication in an object or constant declaration.
--|
--|     A_Subtype_Declaration
--|         The subtype indication in a subtype declaration.
--|
--|     A_Derived_Type_Definition
--|         The subtype indication in a derived type declaration.
--|
--|     An_Index_Subtype_Definition
--|         A subtype definition in an unconstrained array type declaration.
--|
--|     An_Array_Definition
--|         The component subtype indication in an array type declaration.
--|
--|     An_Index_Constraint
--|         An index subtype indication in a constrained array type
--|         declaration.
--|
--|     A_Component_Subtype_Definition
--|         The subtype indication in a record component declaration.
--|
--|     A_Discriminant_Specification
--|         The type mark in a discriminant declaration.
--|
--|     A_Choice
--|         The subtype indication in a discrete range occurring in a choice.
--|
--|     An_Access_Type_Definition
--|         The subtype indication in an access type declaration.
--|
--|     A_Slice
--|         The subtype indication in a discrete range occurring in a slice.
--|
--|     A_Membership_Test
--|         The type mark in a subtype membership test.
--|
--|     A_Type_Conversion
--|         The type mark in a type conversion.
--|
--|     A_Qualified_Expression
--|         The type mark in a qualified expression.
--|
--|     An_Allocator
--|         The subtype indication in an allocator that does not use a
--|         qualified expression.
--|
--|     A_Loop_Parameter_Specification
--|         The subtype indication in a discrete range occurring in a
--|         loop parameter specification.
--|
--|     A_Function_Declaration
--|         The return type mark in a function declaration.
--|
--|     A_Parameter_Specification
--|         The type mark in a parameter specification.
--|
--|     A_Function_Body
--|         The return type mark in a function body.
--|
--|     A_Deferred_Constant_Declaration
--|         The type mark in a deferred constant declaration.
--|
--|     A_Renaming_Declaration
--|         The type mark in an object renaming declaration.
--|
--|     A_Function_Rename
--|         The return type mark in a function renaming declaration.
--|
--|     An_Entry_Declaration
--|         The subtype indication in a discrete range occurring in an
--|         entry declaration.
--|
--|     A_Function_Stub
--|         The return type mark in a function body stub.
--|
--|     A_Generic_Function
--|         The return type mark in a generic function declaration.
--|
--|     A_Generic_Parameter_Declaration
--|         The type mark in a generic formal object declaration.
--|
--|     A_Generic_Formal_Function
--|         The return type mark in a generic formal function declaration.
--|
--|     A_Code_Statement
--|         The type mark in a code statement.

    type Type_Mark_Context_Kinds_Or_Unknown is 
       (An_Object_Declaration, A_Subtype_Declaration, A_Derived_Type_Definition, 
	An_Index_Subtype_Definition, An_Array_Definition, 
	An_Index_Constraint, A_Component_Subtype_Definition, 
	A_Discriminant_Specification, A_Choice, An_Access_Type_Definition, 
	A_Slice, A_Membership_Test, A_Type_Conversion, 
	A_Qualified_Expression, An_Allocator, A_Loop_Parameter_Specification, 
	A_Function_Declaration, A_Parameter_Specification, A_Function_Body, 
	A_Deferred_Constant_Declaration, A_Renaming_Declaration, 
	A_Function_Rename, An_Entry_Declaration, A_Function_Stub, 
	A_Generic_Function, A_Generic_Parameter_Declaration, 
	A_Generic_Formal_Function, A_Code_Statement, Unknown);

    subtype Type_Mark_Context_Kinds is 
       Type_Mark_Context_Kinds_Or_Unknown 
	  range An_Object_Declaration .. A_Code_Statement;

--| The full context in which a reference occurs is described by a
--| Reference_Context.

    type Reference_Context (Kind : Reference_Kinds := A_Read) is
	record
	    case Kind is
		when A_Read | An_Update | A_Read_And_Update =>
		    Data_Access_Context : Data_Access_Context_Ptr;
		when A_Type_Mark =>
		    Type_Mark_Context : Type_Mark_Context_Kinds;
		when others =>
		    null;
	    end case;
	end record;

    type Reference_Context_Ptr is access Reference_Context;

--| A single reference to an entity is described by a Reference record.
--| Reference_Element is the Asis element describing the reference.
--| Context is the context in which the reference occurs, as described
--| previously.
--|
--| Note that a dereference of an access object results in two references:
--| a reference to the ground type of the access type, and
--| A_Read reference to the access object itself.  (We think of the first
--| reference as being to the conceptual storage collection associated with
--| the access type.)  These two types of references are distinguished
--| by Is_Collection_Reference as follows.  If Is_Collection_Reference is
--| false, the reference is a direct reference to Reference_Element,
--| and the expression kind of Reference_Element is one of
--| A_Simple_Name, An_Operator_Symbol, A_Character_Literal, or
--| An_Enumeration_Literal.  If Is_Collection_Reference is true, the
--| reference is to the ground type of Reference_Element, which is an
--| arbitrary expression of an access type.

    type Reference is
	record
	    Reference_Element : Asis.Expression;
	    Context : Reference_Context_Ptr;
	    Is_Collection_Reference : Boolean;
	end record;

    type Reference_List is array (Integer range <>) of Reference;

--| All references within a declarative region part to a given entity are
--| gathered together and described by a Collected_References_Set record.
--| Entity_Definition is the Asis element describing the definition
--| of the entity.  References is the list of references to that entity in
--| source code order.
--|
--| There are two types of reference sets.  In the first type,
--| the element kind of Entity_Definition is An_Entity_Name_Definition and
--| all references have Is_Collection_Reference => False.  In the
--| second, the element kind of Entity_Definition is A_Type_Definition
--| (subkind:  An_Access_Type_Definition) and all references have
--| Is_Collection_Reference => True.

    type Collected_References_Set (Number_References : Natural) is
	record
	    Entity_Definition : Asis.Element;
	    References : Reference_List (1 .. Number_References);
	end record;

    type Collected_References_Set_Ptr is access Collected_References_Set;

    type Collected_References_Set_Ptr_List is 
       array (Integer range <>) of Collected_References_Set_Ptr;

--| The set of all references within a declarative region part is described
--| by a Reference_View.  The Collected_Reference_Set's are in no
--| particular order.

    type Reference_View is access Collected_References_Set_Ptr_List;

end Reference_View_Structures;
