--| +=========================================================================+
--| |                                                                         |
--| | REGION_SCAN - generic traversal of declarative region parts             |
--| |                                                                         |
--| | This package defines a generic Asis traversal analogous to that         |
--| | provided by procedure Asis.Elements.Traverse_Element.  But whereas the  |
--| | latter procedure visits each Asis element, this traversal visits        |
--| | each declarative region part.                                           |
--| |                                                                         |
--| | Two interfaces are provided:  Traverse_Unit, which starts at a          |
--| | compilation unit, and Traverse_Element, which starts at an arbitrary    |
--| | Asis element.  The declarative region parts under the given             |
--| | starting point are visited in depth-first, left-to-right order          |
--| | (i.e., source code order).                                              |
--| |                                                                         |
--| | For each region, Pre_Operation is called when the region is first       |
--| | encountered and Post_Operation is called after all child regions of     |
--| | the region have been visited.  The Level argument indicates the         |
--| | level of nesting; the first region encountered is at level 0.  The      |
--| | Control argument is interpreted as follows:                             |
--| |                                                                         |
--| |   Continue                                                              |
--| |      Continue the scan normally.                                        |
--| |                                                                         |
--| |   Abandon_Children                                                      |
--| |      Returned from Pre_Operation:  invoke the corresponding             |
--| |      Post_Operation immediately.  Returned from Post_Operation:         |
--| |      continue the scan normally.                                        |
--| |                                                                         |
--| |   Abandon_Siblings                                                      |
--| |      Invoke the Post_Operation of the enclosing parent region           |
--| |      immediately.                                                       |
--| |                                                                         |
--| |   Terminate_Immediately                                                 |
--| |       Do just that.                                                     |
--| |                                                                         |
--| | See package Region_Support for the definition of a declarative          |
--| | region part.                                                            |
--| |                                                                         |
--| | Greg Janee                                                              |
--| | General Research Corporation                                            |
--| |                                                                         |
--| | Modified 12/7/94 by CWH                                                 |
--| |                                                                         |
--| | Added Include_Instance_Bodies control parameter to the exported         |
--| | operations.                                                             |
--| +=========================================================================+

with Asis;
with Region_Support;

generic

    type State_Information is limited private;

    with procedure Pre_Operation (Region : in Region_Support.Region; 
				  Level : in Natural; 
				  Control : out Asis.Elements.Traverse_Control; 
				  State : in out State_Information);

    with procedure Post_Operation (Region : in Region_Support.Region; 
				   Level : in Natural; 
				   Control : out Asis.Elements.Traverse_Control; 
				   State : in out State_Information);

package Region_Scan is

--| In the following operations, setting Expand_Instantations to True extends
--| the traversal into the implicit spec of each generic instance.
--| Also setting Include_Instance_Bodies to True further extends the traversal
--| into the corresponding implicit body of each instance, if it exists.

    procedure Traverse_Unit (Unit : in Asis.Compilation_Unit; 
			     State : in out State_Information; 
			     Expand_Instantiations : in Boolean := False; 
			     Include_Instance_Bodies : in Boolean := False; 
			     Trace_Scan : in Boolean := False);

    procedure Traverse_Element (Element : in Asis.Element; 
				State : in out State_Information; 
				Expand_Instantiations : in Boolean := False; 
				Include_Instance_Bodies : in Boolean := False; 
				Trace_Scan : in Boolean := False);

--| The following exception is raised if an Asis problem is encountered.

    Traversal_Error : exception;

end Region_Scan;
