--| +=========================================================================+
--| |                                                                         |
--| | REGION_SCAN.REGION_SCAN_INTERNAL.SCAN_DISCRETE_RANGE (body)             |
--| |                                                                         |
--| | Greg Janee                                                              |
--| | General Research Corporation                                            |
--| |                                                                         |
--| +=========================================================================+

separate (Region_Scan.Region_Scan_Internal)
procedure Scan_Discrete_Range (The_Range : in Asis.Discrete_Range; 
			       The_Context : in out Context) is

    Cuid : constant String := 
       "Region_Scan.Region_Scan_Internal.Scan_Discrete_Range";
    Puid : constant String := "Scan_Discrete_Range";

    package Error renames Region_Scan_Internal.Error_Handling_Support;

--| +-------------------------------------------------------------------------+
--| | SCAN_DISCRETE_SUBTYPE_INDICATION (local)                                |
--| +-------------------------------------------------------------------------+

    procedure Scan_Discrete_Subtype_Indication 
		 (The_Range : in Asis.Discrete_Range; 
		  The_Context : in out Context) is

	Puid : constant String := "Scan_Discrete_Subtype_Indication";

    begin

	declare
	    Constraint : Asis.Constraint := 
	       Asis_Td.Subtype_Constraint (The_Range);
	    Type_Mark : Asis.Expression := Asis_Td.Type_Mark (The_Range);
	begin

	    Region_Scan_Internal.Scan_Expression (Type_Mark, The_Context);

	    if not Asis_E.Is_Nil (Constraint) then
		Region_Scan_Internal.Scan_Constraint (Constraint, The_Context);
	    end if;

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Range);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Range);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Range);
	    raise;

    end Scan_Discrete_Subtype_Indication;

--| +-------------------------------------------------------------------------+
--| | SCAN_RANGE_ATTRIBUTE (local)                                            |
--| +-------------------------------------------------------------------------+

    procedure Scan_Range_Attribute (The_Range : in Asis.Discrete_Range; 
				    The_Context : in out Context) is

	Puid : constant String := "Scan_Range_Attribute";

    begin

	declare
	    Attribute : Asis.Expression := Asis_Td.Range_Attribute (The_Range);
	begin

	    Region_Scan_Internal.Scan_Expression (Attribute, The_Context);

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Range);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Range);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Range);
	    raise;

    end Scan_Range_Attribute;

--| +-------------------------------------------------------------------------+
--| | SCAN_SIMPLE_RANGE (local)                                               |
--| +-------------------------------------------------------------------------+

    procedure Scan_Simple_Range (The_Range : in Asis.Discrete_Range; 
				 The_Context : in out Context) is

	Puid : constant String := "Scan_Simple_Range";

    begin

	declare
	    Lower_Bound : Asis.Expression := Asis_Td.Lower_Bound (The_Range);
	    Upper_Bound : Asis.Expression := Asis_Td.Upper_Bound (The_Range);
	begin

	    Region_Scan_Internal.Scan_Expression (Lower_Bound, The_Context);

	    Region_Scan_Internal.Scan_Expression (Upper_Bound, The_Context);

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Range);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Range);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Range);
	    raise;

    end Scan_Simple_Range;

--| +-------------------------------------------------------------------------+
--| | SCAN_DISCRETE_RANGE (exported)                                          |
--| +-------------------------------------------------------------------------+

begin

    if Rgn.Denotes_One_Or_More_Regions (The_Range, 
					Region_Scan.Expand_Instantiations) then
	declare
	    Remaining_Parent_Elements : constant Asis.Element_List := 
	       Rgn.Non_Region_Subelements (The_Range, 
					   Region_Scan.Expand_Instantiations);
	    Regions : constant Rgn.Region_List := 
	       Rgn.Denoted_Regions (The_Range, 
				    Region_Scan.Expand_Instantiations, 
				    Region_Scan.Include_Instance_Bodies);
	begin
	    Region_Scan_Internal.Scan_Any_List 
	       (Remaining_Parent_Elements, The_Context);
	    for I in Regions'Range loop
		Region_Scan.Process_Region (Regions (I), The_Context);
	    end loop;
	end;
	return;
    end if;

    case Asis_Td.Discrete_Range_Kind (The_Range) is
	when Asis_Td.A_Simple_Range =>
	    Scan_Simple_Range (The_Range, The_Context);
	when Asis_Td.A_Range_Attribute =>
	    Scan_Range_Attribute (The_Range, The_Context);
	when Asis_Td.A_Discrete_Subtype_Indication =>
	    Scan_Discrete_Subtype_Indication (The_Range, The_Context);
	when Asis_Td.Not_A_Discrete_Range =>
	    Error.Log (Error.An_Unhandled_Case, Cuid, Puid, The_Range);
	    if Region_Scan_Internal.Raise_Exception_On_Unhandled_Case then
		raise Traversal_Error;
	    end if;
    end case;

end Scan_Discrete_Range;
