--| +=========================================================================+
--| |                                                                         |
--| | SCAN_FOR_REGIONS                                                        |
--| |                                                                         |
--| | Usage: scan_for_regions [-e|-E] <library> [-l|-s] <unit>                |
--| |                                                                         |
--| | This program is a simple driver for the Region_Scan package.  It        |
--| | accepts two command line arguments, the name of an ASIS library and the |
--| | name of a compilation unit contained therein, and it writes a           |
--| | textual representation of the declarative region part tree for          |
--| | that unit to standard output.                                           |
--| |                                                                         |
--| | The unit is looked up primarily as a library unit and secondarily       |
--| | as a secondary unit.  To force looking up a library unit, specify the   |
--| | "-l" option.  To force looking up a secondary unit, specify the "-s"    |
--| | option.                                                                 |
--| |                                                                         |
--| | The "-e" option causes the implicit specifications of generic instances |
--| | to be traversed. The "-E" option causes the corresponding implicit      |
--| | bodies (if they exist) to be traversed, as well.                        |
--| |                                                                         |
--| | Greg Janee                                                              |
--| | General Research Corporation                                            |
--| |                                                                         |
--| +=========================================================================+

with Asis;
with Command;
with Msg_Log;
with Region_Scan;
with Region_Support;

procedure Scan_For_Regions is

--| Standard Asis renames...

    package Asis_Cu renames Asis.Compilation_Units;
    package Asis_E renames Asis.Elements;
    package Asis_En renames Asis.Environment;
    package Asis_L renames Asis.Libraries;
    package Asis_Str renames Asis.Strings;

    package Cli renames Command;

    Argument_Count : Integer := Cli.Argc - 1;
    Arguments : constant Cli.String_Ptr_Array := Cli.Arguments;
    Examine_Library_Units_Only : Boolean := False;
    Examine_Secondary_Units_Only : Boolean := False;
    Expand_Instantiations : Boolean := False;
    Include_Instance_Bodies : Boolean := False;
    Offset : Integer := 0;
    The_Library : Asis.Library;
    The_Unit : Asis.Compilation_Unit;

    Find_Error : exception;
    Usage_Error : exception;

    type State_Information is
	record
	    null;
	end record;

    The_State : State_Information;

    procedure Do_Nothing (Region : in Region_Support.Region; 
			  Level : in Natural; 
			  Control : out Asis_E.Traverse_Control; 
			  State : in out State_Information) is
    begin
	Control := Asis_E.Continue;
    end Do_Nothing;

    package The_Region_Scan is 
       new Region_Scan (State_Information, Do_Nothing, Do_Nothing);

begin

    Msg_Log.Set_Program ("sfr");

    if Argument_Count > 0 then
	declare
	    A : String renames Arguments (1).all;
	begin
	    if A = "-e" then
		Expand_Instantiations := True;
		Argument_Count := Argument_Count - 1;
		Offset := 1;
	    elsif A = "-E" then
		Expand_Instantiations := True;
		Include_Instance_Bodies := True;
		Argument_Count := Argument_Count - 1;
		Offset := 1;
	    end if;
	end;
    end if;

    if Argument_Count = 3 then
	if Arguments (2 + Offset).all = "-l" then
	    Examine_Library_Units_Only := True;
	elsif Arguments (2 + Offset).all = "-s" then
	    Examine_Secondary_Units_Only := True;
	else
	    raise Usage_Error;
	end if;
    elsif Argument_Count /= 2 then
	raise Usage_Error;
    end if;

    Asis_En.Initialize;
    Asis_L.Associate (The_Library, 
		      Asis_Str.To_Asis_String (Arguments (1 + Offset).all));
    Asis_L.Open (The_Library);

    if Examine_Library_Units_Only then
	The_Unit := Asis_Cu.Library_Unit 
		       (Arguments (Arguments'Last).all, The_Library);
	if Asis_Cu.Is_Nil (The_Unit) then
	    raise Find_Error;
	end if;
    elsif Examine_Secondary_Units_Only then
	The_Unit := Asis_Cu.Secondary_Unit 
		       (Arguments (Arguments'Last).all, The_Library);
	if Asis_Cu.Is_Nil (The_Unit) then
	    raise Find_Error;
	end if;
    else
	The_Unit := Asis_Cu.Library_Unit 
		       (Arguments (Arguments'Last).all, The_Library);
	if Asis_Cu.Is_Nil (The_Unit) then
	    The_Unit := Asis_Cu.Secondary_Unit 
			   (Arguments (Arguments'Last).all, The_Library);
	    if Asis_Cu.Is_Nil (The_Unit) then
		raise Find_Error;
	    end if;
	end if;
    end if;

    The_Region_Scan.Traverse_Unit 
       (The_Unit, The_State, 
	Expand_Instantiations => Expand_Instantiations, 
	Include_Instance_Bodies => Include_Instance_Bodies, 
	Trace_Scan => True);

    Asis_L.Close (The_Library);
    Asis_L.Dissociate (The_Library);
    Asis_En.Finalize;

exception

    when Usage_Error =>
	Msg_Log.Put_Msg 
	   (Msg_Log.Error, 
	    "usage is ""scan_for_regions [-e|-E] <library> [-l|-s] <unit>""");

    when Find_Error =>
	Msg_Log.Put_Msg (Msg_Log.Error, "unit not found");

    when Asis.Asis_Inappropriate_Library =>
	Msg_Log.Put_Msg 
	   (Msg_Log.Error, 
	    "exception Asis_Inappropriate_Library raised; status is " & 
	       Asis_En.Error_Kinds'Image (Asis_En.Status) & 
	       "; diagnosis follows");
	Msg_Log.Put_Msg (Msg_Log.Error, 
			 Asis_Str.To_Standard_String (Asis_En.Diagnosis));

    when Asis.Asis_Failed =>
	Msg_Log.Put_Msg (Msg_Log.Error, 
			 "exception Asis_Failed raised; status is " & 
			    Asis_En.Error_Kinds'Image (Asis_En.Status) & 
			    "; diagnosis follows");
	Msg_Log.Put_Msg (Msg_Log.Error, 
			 Asis_Str.To_Standard_String (Asis_En.Diagnosis));

    when The_Region_Scan.Traversal_Error =>
	Msg_Log.Put_Msg (Msg_Log.Error, "exception Traversal_Error raised");

end Scan_For_Regions;
