-- DEC/CMS REPLACEMENT HISTORY, Element MSG_LOG_.ADA
-- *2     6-JUN-1991 16:27:18 ATVS "Added message passthrough"
-- *1    30-MAY-1991 17:29:30 ATVS "Initial CMS Release"
-- DEC/CMS REPLACEMENT HISTORY, Element MSG_LOG_.ADA
-- =====================================================================
-- >>>>>>>>>>>>>>>>>>>>>> ADA COMPILATION UNIT <<<<<<<<<<<<<<<<<<<<<<<<<
-- =====================================================================
--
-- NAME:        Msg_Log
--
--              SPECIFICATION 
--
-- SYSTEM:      DEC VMS Operating System
-- 
-- AUTHOR:      Chuck Hobin
--
-- DATE:        16 March 1990
--
--                            CHANGE HISTORY
--
-- MM-DD-YY | Initials | Description
-- ---------------------------------------------------------------------
-- 06-05-91     CWH     Added PUT_LINE.
-- 03-19-92     CWH     PORT TO ULTRIX.  Remove all dependencies on
--                      Error_Utilities. Add Debug_On constant and
--                      Put_Msg_Debug, Set_Msg_Out_Status,
--                      Get_Msg_Out_Status.
-- 04-09-92     CWH     Added Put_Msg_Debug which accepts CUID and PUID
--                      parameters.
-- 10-05-93     CWH     Changed order of parameters in Put_Msg, 
--                      Put_Msg_Debug, Put_Msg_Internal, and 
--                      Put_Msg_Fatal.  Put_Msg_Internal no longer 
--                      raises an exception. Changed Msg_Out_Status_Enum.
-- =====================================================================
-- =====================================================================

package Msg_Log is

-- OVERVIEW
--
-- This package provides operations for logging diagnostic messages 
-- generated by the using program, ensuring that all messages are 
-- handled in a uniform manner.

-- The following information is logged for each message:
--
--      1) The name of the program.
--      2) The message kind (Msg_Kind), i.e., its severity.
--      3) The message ID, a short mnemonic string that uniquely
--         identifies the message in the system.
--      4) The message text.
--      5) The location in the program (library unit and program unit)
--         where the message was issued.  (Debug, Internal and Fatal errors
--         only.)

-----------------------------------------------------------------------


    type Msg_Kind is (None, Listing, Statistic, Timing, Id, Debug, 
		      Note, Warning, Error, Internal, Fatal);
-- A value of this type must be associated with each message submitted
-- to this package.  The kinds are listed in increasing order of 
-- severity.

    subtype Valid_Msg is Msg_Kind range Listing .. Fatal;

-- 'None' is a special kind that is not meant to be used with a message. 
-- It is used to indicate that the user wants no messages of any severity 
-- to be logged (see the operation Set_Noise_Level).

    type Msg_Out_Status_Enum is 
       (Msg_Out_Is_Standard_Error, Msg_Out_Is_Standard_Output, 
	Msg_Out_Is_Default_Output, Msg_Out_Is_File);
-- Enumeration of possible places that messages can be written.  By
-- default, messages are written to the standard error device.  This can be
-- changed using the operation Set_Msg_Out_Status.
--
-- For Msg_Out_Is_Default_Output, messages are written to the current
-- default output file in package Text_IO.
--
-- For Msg_Out_Is_File, messages are written to the file object declared
-- in package Msg_Log_File.  The file object must
-- be created/opened before this status is selected.

-----------------------------------------------------------------------
--
-- The following operations are provided by this package:
--
-- SET_PROGRAM          Sets the program name that is to be logged 
--                      with each message
--
-- SET_MSG_OUT_STATUS   Establishes where messages are written.
--
-- GET_MSG_OUT_STATUS   Queries where messages are written.
--
-- SET_NOISE_LEVEL      Sets a threshold that limits the display of
--                      logged messages.
--
-- GET_NOISE_LEVEL      Returns the current noise level.
--
-- PUT_MSG              Logs a message with severity up to 'Error'
--
-- PUT_MSG_DEBUG        Logs a 'Debug' message. Has same effect as
--                      PUT_MSG( Debug, ... ), but provides convenient
--                      interface for reporting source location of
--                      the caller.
--
-- PUT_MSG_INTERNAL     Logs an 'Internal' Error message. Has same effect
--                      as PUT_MSG( Internal, ... ), but provides convenient
--                      interface for reporting source location of
--                      the caller.
--
-- PUT_MSG_FATAL        Logs a 'Fatal' Error message, then raises a 
--                      Fatal_Error exception.
--
-- MAX_SEVERITY_SEEN    Function returning the maximum severity of all
--                      messages logged.
--
-- MSG_COUNT            Function returning the number of messages logged
--                      for a given severity
--
-- RESET_MSG_INFO       Resets the maximum severity and message count
--                      accumulators.
--
-- PUT_LINE             Writes a string directly to the log without
--                      interpretation.  Has no effect on the package 
--                      state.
--
-- The normal sequence for using these operations is described by the
-- following expression:
--
--    Set_Program
--    { Set_Noise_Level } ?
--    { 
--      { Put_Msg | Put_Msg_Debug | Put_Msg_Internal | Put_Msg_Fatal } *
--      { Max_Severity_Seen | Msg_Count } *
--      { Reset_Msg_Info } ?
--    } *
--
-- where '*' means indefinite repeat, '?' means 1 or 0 repeats,
-- and '{}' are used for grouping.
--
-- For example, here is a valid order:
--  Set_Program, Set_Noise_Level, Put_Msg, Put_Msg_Internal, 
--   Max_Severity_Seen, Reset_Msg_Info, Put_Msg, Put_Msg,
--   Max_Severity_Seen

-----------------------------------------------------------------------

    procedure Set_Program (Name : in String);

-- Initializes the program name part of the message.  It
-- allows this part of the message to be set just once 
-- instead of including it as a parameter in every message call.

-----------------------------------------------------------------------

    procedure Set_Msg_Out_Status (Status : Msg_Out_Status_Enum);
-- Establishes where messages are written.  

-----------------------------------------------------------------------

    function Get_Msg_Out_Status return Msg_Out_Status_Enum;
-- Returns the current Msg_Out_Status.

-----------------------------------------------------------------------

    procedure Set_Noise_Level (Level : in Msg_Kind);

-- Controls the quantity of logged messages that are displayed to the
-- user.  When the noise level is set to a particular message kind, only
-- messages with severity greater than or equal to that kind are 
-- displayed.  The kind 'None' is a special case: if it is used, no 
-- messages at all will be displayed.
--
-- NOTE: upon elaboration of this package, the Noise_Level is set to
-- 'Listing' (all messages are displayed).

-----------------------------------------------------------------------

    function Get_Noise_Level return Msg_Kind;

-- Returns the current noise level.

-----------------------------------------------------------------------

    subtype Simple_Msg is Valid_Msg range Listing .. Internal;

    procedure Put_Msg 
		 (Kind : in Simple_Msg; Text : in String; Id : in String := "");

-- Logs a message.  The Kind must not be 'Fatal'.  The 
-- message is not displayed if the Kind is lower than the  
-- threshold set by Set_Severity_Level.

-----------------------------------------------------------------------

    procedure Put_Msg_Debug (Text : in String);

-- Logs a message of the Kind 'Debug'. The message is not displayed 
-- if the threshold set by Set_Severity_Level is greater than 
-- 'Debug'.
--
-- No ID is specified since the messages are normally for development
-- testing use only.

-----------------------------------------------------------------------

    procedure Put_Msg_Debug (Lib_Unit : in String; 
			     Prog_Unit : in String; 
			     Text : in String);

-- Similar in function to the simple version of Put_Msg_Debug.  The
-- user must supply the names of the library unit (or subunit) and program
-- unit where the call to this procedure is made.  These names are
-- incorporated into the message.

-----------------------------------------------------------------------

    procedure Put_Msg_Internal (Lib_Unit : in String; 
				Prog_Unit : in String; 
				Text : in String; 
				Id : in String := "");

-- Logs a message of the Kind 'Internal'. The message is not displayed 
-- if the threshold set by Set_Severity_Level is greater than 
-- 'Internal'.
--
-- Prog_Unit is the name of the program unit that contains the call to
-- this procedure.  Lib_Unit is the name of the library unit (or subunit)
-- that contains the Prog_Unit.

-----------------------------------------------------------------------

    Fatal_Error : exception;

    procedure Put_Msg_Fatal (Lib_Unit : in String; 
			     Prog_Unit : in String; 
			     Text : in String; 
			     Id : in String := "");

-- Logs a message of the Kind 'Fatal'. The message is not displayed 
-- if the threshold set by Set_Severity_Level is greater than 
-- 'Fatal'.
--
-- This procedure raises the exception Fatal_Error. (It is raised 
-- even if the message is not displayed.)
--
-- Prog_Unit is the name of the program unit that contains the call to
-- this procedure.  Lib_Unit is the name of the library unit (or subunit)
-- that contains the Prog_Unit.

-----------------------------------------------------------------------

    function Max_Severity_Seen return Msg_Kind;

-- Returns the kind of the most severe message that has been logged.
-- ('None' is returned if no messages at all have been logged.)  Note that 
-- the severity of a message is recorded even if it is below the 
-- threshold set by Set_Noise_Level.

-----------------------------------------------------------------------

    function Msg_Count (Kind : in Valid_Msg) return Natural;

-- Returns the number of messages of this kind that were logged.
-- Note that a message is counted even if its severity is below the 
-- threshold set by Set_Noise_Level.

-----------------------------------------------------------------------

    procedure Reset_Msg_Info;

-- Causes the Msg_Count to be reset to zero and the Max_Severity_Seen to
-- be reset to 'None'.

-----------------------------------------------------------------------

    procedure Put_Line (Item : in String);
-- Direct pass-through to the log text file.  Has no effect on the 
-- package state. 
--
-- Restricted use of this operation is suggested - do not use
-- to circumvent the Put_Msg operations!!

end Msg_Log;
