--| +=========================================================================+
--| |                                                                         |
--| | NAMESPACE_SCAN (body)                                                   |
--| |                                                                         |
--| | Greg Janee                                                              |
--| | General Research Corporation                                            |
--| |                                                                         |
--| +=========================================================================+

with Unchecked_Deallocation;

package body Namespace_Scan is

--| Standard renames...

    package Asis_Cu renames Asis.Compilation_Units;
    package Asis_D renames Asis.Declarations;
    package Asis_E renames Asis.Elements;
    package Asis_En renames Asis.Environment;
    package Asis_Rc renames Asis.Representation_Clauses;
    package Asis_S renames Asis.Statements;
    package Asis_Str renames Asis.Strings;
    package Asis_Td renames Asis.Type_Definitions;
    package Asis_X renames Asis.Expressions;

--| Support for tracing.  Start, Stop, and On implement the corresponding
--| subprograms exported by this package.  Add_Level creates a new trace level
--| having the indicated number of children.  Add_Children adds
--| children to the current trace level.  Log writes a textual
--| representation of an element to the trace file.
--| Every scan procedure in this package follows the same general outline,
--| namely:
--|
--|     <gather children>
--|     if Trace_Support.On then
--|         Trace_Support.Log (<current element>);
--|         Trace_Support.Add_Level (<number of children>);
--|     end if;
--|     <scan children>

    package Trace_Support is
	procedure Start;
	procedure Stop;
	function On return Boolean;
	procedure Add_Level (Number_Children : in Natural);
	procedure Add_Children (Number_Children : in Natural);
	procedure Log (The_Unit : in Asis.Compilation_Unit);
	procedure Log (The_Element : in Asis.Element);
    end Trace_Support;

    package body Trace_Support is separate;

--| Bodies of the exported trace subprograms.

    procedure Start_Trace is
    begin
	Trace_Support.Start;
    end Start_Trace;

    procedure Stop_Trace is
    begin
	Trace_Support.Stop;
    end Stop_Trace;

    function Trace_On return Boolean is
    begin
	return Trace_Support.On;
    end Trace_On;

--| Error handling.  Log writes a message using the Msg_Log package.
--| The error kind A_Previous_Error is used to produce a stack traceback,
--| i.e., every scan procedure in this package has exception handlers
--| of the form:
--|
--|     when Asis_Inappropriate_Element =>
--|         Error_Handling_Support.Log (A_Bad_Element, ...);
--|         raise Traversal_Error;
--|
--|     when Asis_Failed =>
--|         Error_Handling_Support.Log (An_Asis_Failure, ...);
--|         raise Traversal_Error;
--|
--|     when Traversal_Error =>
--|         Error_Handling_Support.Log (A_Previous_Error, ...);
--|         raise;
--|
--| Malformed_Definition_Error notes a malformed definition.

    package Error_Handling_Support is

	type Error_Kinds is (A_Previous_Error, An_Unhandled_Case, 
			     An_Asis_Failure, A_Bad_Element);

	procedure Log (Error_Kind : in Error_Kinds; 
		       Library_Unit : in String; 
		       Program_Unit : in String; 
		       Current_Unit : in Asis.Compilation_Unit);

	procedure Log (Error_Kind : in Error_Kinds; 
		       Library_Unit : in String; 
		       Program_Unit : in String; 
		       Current_Element : in Asis.Element);

	procedure Malformed_Definition_Error 
		     (Current_Element : in Asis.Entity_Name_Definition);

    end Error_Handling_Support;

    package body Error_Handling_Support is separate;

--| Convenience procedures.  One_If_Present returns 1 if the given element
--| is non-nil and 0 otherwise.  It is used to allow arithmetic computations to
--| be affected by the presence of optional syntactic categories.
--| One_If_True is a similar function for boolean flags.

    procedure Expand_All_Instantiations (Yes : in Boolean) is
    begin
	Expand_Procedure_Instantiations := Yes;
	Expand_Function_Instantiations := Yes;
	Expand_Package_Instantiations := Yes;
    end Expand_All_Instantiations;

    function One_If_Present (The_Element : in Asis.Element) return Integer is
    begin
	if Asis_E.Is_Nil (The_Element) then
	    return 0;
	else
	    return 1;
	end if;
    end One_If_Present;

    function One_If_True (Flag : in Boolean) return Integer is
    begin
	if Flag = True then
	    return 1;
	else
	    return 0;
	end if;
    end One_If_True;

--| Memory control procedures.

    procedure Free is new Unchecked_Deallocation 
			     (Definition_Holder, Definition_Holder_Ptr);

    procedure Free_Definition_List is
	H : Definition_Holder_Ptr;
	T : Definition_Holder_Ptr;
    begin
	H := Definition_List;
	while H /= null loop
	    T := H.Next;
	    Free (H);
	    H := T;
	end loop;
	Definition_List := null;
    end Free_Definition_List;

--| Scan procedure bodies...

    procedure Scan_Any (The_Element : in Asis.Element; 
			The_Context : in Context) is separate;

    procedure Scan_Any_List (The_List : in Asis.Element_List; 
			     The_Context : in Context) is separate;

    procedure Scan_Argument_Association (The_Association : in Asis.Association; 
					 The_Context : in Context) is separate;

    procedure Scan_Argument_Association_List 
		 (The_List : in Asis.Argument_Association_List; 
		  The_Context : in Context) is separate;

    procedure Scan_Case_Statement_Alternative 
		 (The_Alternative : in Asis.Case_Statement_Alternative; 
		  The_Context : in Context) is separate;

    procedure Scan_Case_Statement_Alternative_List 
		 (The_List : in Asis.Case_Statement_Alternative_List; 
		  The_Context : in Context) is separate;

    procedure Scan_Choice (The_Choice : in Asis.Choice; 
			   The_Context : in Context) is separate;

    procedure Scan_Choice_List (The_List : in Asis.Choice_List; 
				The_Context : in Context) is separate;

    procedure Scan_Compilation_Unit (The_Unit : in Asis.Compilation_Unit; 
				     The_Context : in Context) is separate;

    procedure Scan_Component_Association 
		 (The_Association : in Asis.Component_Association; 
		  The_Context : in Context) is separate;

    procedure Scan_Component_Association_List 
		 (The_List : in Asis.Component_Association_List; 
		  The_Context : in Context) is separate;

    procedure Scan_Component_Clause (The_Clause : in Asis.Component_Clause; 
				     The_Context : in Context) is separate;

    procedure Scan_Component_Clause_List 
		 (The_List : in Asis.Component_Clause_List; 
		  The_Context : in Context) is separate;

    procedure Scan_Constraint (The_Constraint : in Asis.Constraint; 
			       The_Context : in Context) is separate;

    procedure Scan_Context_Clause_List (The_List : in Asis.Context_Clause_List; 
					The_Context : in Context) is separate;

    procedure Scan_Declaration (The_Declaration : in Asis.Declaration; 
				The_Context : in Context) is separate;

    procedure Scan_Declaration_List (The_List : in Asis.Declaration_List; 
				     The_Context : in Context) is separate;

    procedure Scan_Declarative_Item_List 
		 (The_List : in Asis.Declarative_Item_List; 
		  The_Context : in Context) is separate;

    procedure Scan_Declarative_Region_Part 
		 (The_Region : in Rgn.Region; 
		  The_Context : in Context) is separate;

    procedure Scan_Discrete_Range (The_Range : in Asis.Discrete_Range; 
				   The_Context : in Context) is separate;

    procedure Scan_Discrete_Range_List (The_List : in Asis.Discrete_Range_List; 
					The_Context : in Context) is separate;

    procedure Scan_Discriminant_Association 
		 (The_Association : in Asis.Discriminant_Association; 
		  The_Context : in Context) is separate;

    procedure Scan_Discriminant_Association_List 
		 (The_List : in Asis.Discriminant_Association_List; 
		  The_Context : in Context) is separate;

    procedure Scan_Entity_Name_Definition 
		 (The_Definition : in Asis.Entity_Name_Definition; 
		  The_Context : in Context) is separate;

    procedure Scan_Entity_Name_Definition_List 
		 (The_List : in Asis.Entity_Name_Definition_List; 
		  The_Context : in Context) is separate;

    procedure Scan_Exception_Handler (The_Handler : in Asis.Exception_Handler; 
				      The_Context : in Context) is separate;

    procedure Scan_Exception_Handler_List 
		 (The_List : in Asis.Exception_Handler_List; 
		  The_Context : in Context) is separate;

    procedure Scan_Expression (The_Expression : in Asis.Expression; 
			       The_Context : in Context) is separate;

    procedure Scan_Expression_List (The_List : in Asis.Expression_List; 
				    The_Context : in Context) is separate;

    procedure Scan_If_Statement_Arm (The_Arm : in Asis.If_Statement_Arm; 
				     The_Context : in Context) is separate;

    procedure Scan_If_Statement_Arm_List 
		 (The_List : in Asis.If_Statement_Arm_List; 
		  The_Context : in Context) is separate;

    procedure Scan_Null_Component (The_Component : in Asis.Record_Component; 
				   The_Context : in Context) is separate;

    procedure Scan_Parameter_Association (The_Association : in Asis.Association; 
					  The_Context : in Context) is separate;

    procedure Scan_Parameter_Association_List 
		 (The_List : in Asis.Association_List; 
		  The_Context : in Context) is separate;

    procedure Scan_Pragma (The_Pragma : in Asis.Pragma_Element; 
			   The_Context : in Context) is separate;

    procedure Scan_Pragma_List (The_List : in Asis.Pragma_Element_List; 
				The_Context : in Context) is separate;

    procedure Scan_Record_Component_List 
		 (The_List : in Asis.Record_Component_List; 
		  The_Context : in Context) is separate;

    procedure Scan_Representation_Clause 
		 (The_Clause : in Asis.Representation_Clause; 
		  The_Context : in Context) is separate;

    procedure Scan_Select_Alternative 
		 (The_Alternative : in Asis.Select_Alternative; 
		  The_Context : in Context) is separate;

    procedure Scan_Select_Statement_Arm (The_Arm : in Asis.Select_Statement_Arm; 
					 The_Context : in Context) is separate;

    procedure Scan_Select_Statement_Arm_List 
		 (The_List : in Asis.Select_Statement_Arm_List; 
		  The_Context : in Context) is separate;

    procedure Scan_Statement (The_Statement : in Asis.Statement; 
			      The_Context : in Context) is separate;

    procedure Scan_Statement_List (The_List : in Asis.Statement_List; 
				   The_Context : in Context) is separate;

    procedure Scan_Subtype_Indication 
		 (The_Indication : in Asis.Subtype_Indication; 
		  The_Context : in Context) is separate;

    procedure Scan_Type_Definition (The_Definition : in Asis.Type_Definition; 
				    The_Context : in Context) is separate;

    procedure Scan_Use_Clause (The_Clause : in Asis.Context_Clause; 
			       The_Context : in Context) is separate;

    procedure Scan_Variant (The_Variant : in Asis.Variant; 
			    The_Context : in Context) is separate;

    procedure Scan_Variant_List (The_List : in Asis.Variant_List; 
				 The_Context : in Context) is separate;

    procedure Scan_Variant_Part (The_Part : in Asis.Record_Component; 
				 The_Context : in Context) is separate;

    procedure Scan_With_Clause (The_Clause : in Asis.Context_Clause; 
				The_Context : in Context) is separate;

end Namespace_Scan;
