--| +=========================================================================+
--| |                                                                         |
--| | REFERENCE_SCAN.SCAN_STATEMENT (body)                                    |
--| |                                                                         |
--| | Greg Janee                                                              |
--| | General Research Corporation                                            |
--| |                                                                         |
--| +=========================================================================+

separate (Reference_Scan)
procedure Scan_Statement (The_Statement : in Asis.Statement; 
			  The_Context : in Context) is

    Cuid : constant String := "Reference_Scan.Scan_Statement";
    Puid : constant String := "Scan_Statement";

    package Ctx renames Reference_Scan.Context_Support;
    package Error renames Reference_Scan.Error_Handling_Support;
    package Trace renames Reference_Scan.Trace_Support;

--| +-------------------------------------------------------------------------+
--| | SCAN_ABORT_STATEMENT (local)                                            |
--| +-------------------------------------------------------------------------+

    procedure Scan_Abort_Statement (The_Statement : in Asis.Statement; 
				    The_Context : in Context) is

	Puid : constant String := "Scan_Abort_Statement";

    begin

	declare
	    Tasks : constant Asis.Expression_List := 
	       Asis_S.Aborted_Tasks (The_Statement);
	begin

	    if Trace.On then
		Trace.Log (The_Statement, The_Context);
		Trace.Add_Level (Tasks'Length);
	    end if;

	    Reference_Scan.Scan_Expression_List 
	       (Tasks, Ctx.Set (The_Context, 
				Basic_Context => Rvs.An_Abort, 
				Weight => 90));

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Statement);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Statement);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Statement);
	    raise;

    end Scan_Abort_Statement;

--| +-------------------------------------------------------------------------+
--| | SCAN_ACCEPT_STATEMENT (local)                                           |
--| +-------------------------------------------------------------------------+

    procedure Scan_Accept_Statement (The_Statement : in Asis.Statement; 
				     The_Context : in Context) is

	Puid : constant String := "Scan_Accept_Statement";

    begin

	declare
	    Entry_Name : Asis.Simple_Name := 
	       Asis_S.Accept_Entry_Simple_Name (The_Statement);
	    Index : Asis.Expression := Asis_S.Family_Index (The_Statement);
	    Parameters : constant Asis.Parameter_Specification_List := 
	       Asis_S.Accept_Parameters (The_Statement);
	    Statements : constant Asis.Statement_List := 
	       Asis_S.Accept_Body_Statements 
		  (The_Statement, Reference_Scan.
				     Include_Accept_Statement_Pragmas);
	begin

	    if Trace.On then
		Trace.Log (The_Statement, The_Context);
		Trace.Add_Level (1 + Reference_Scan.One_If_Present (Index) + 
				 Parameters'Length + Statements'Length);
	    end if;

	    Reference_Scan.Scan_Expression 
	       (Entry_Name, Ctx.Set (The_Context, 
				     Basic_Context => Rvs.An_Accept, 
				     Weight => 90));

	    if not Asis_E.Is_Nil (Index) then
		Reference_Scan.Scan_Expression 
		   (Index, Ctx.Set 
			      (The_Context, 
			       Basic_Context => Rvs.A_Read, 
			       Weight => 5, 
			       Add_Data_Access_Context => Rvs.An_Entry_Index));
	    end if;

	    Reference_Scan.Scan_Declaration_List (Parameters, The_Context);

	    Reference_Scan.Scan_Statement_List (Statements, The_Context);

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Statement);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Statement);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Statement);
	    raise;

    end Scan_Accept_Statement;

--| +-------------------------------------------------------------------------+
--| | SCAN_ACCEPT_STATEMENT_NON_REGION_SUBELEMENTS (local)                    |
--| +-------------------------------------------------------------------------+

    procedure Scan_Accept_Statement_Non_Region_Subelements 
		 (The_Statement : in Asis.Statement; 
		  The_Context : in Context) is

	Puid : constant String := 
	   "Scan_Accept_Statement_Non_Region_Subelements";

    begin

	declare
	    Entry_Name : Asis.Simple_Name := 
	       Asis_S.Accept_Entry_Simple_Name (The_Statement);
	    Index : Asis.Expression := Asis_S.Family_Index (The_Statement);
	begin

	    if Trace.On then
		Trace.Log (The_Statement, The_Context);
		Trace.Add_Level (1 + Reference_Scan.One_If_Present (Index));
	    end if;

	    Reference_Scan.Scan_Expression 
	       (Entry_Name, Ctx.Set (The_Context, 
				     Basic_Context => Rvs.An_Accept, 
				     Weight => 90));

	    if not Asis_E.Is_Nil (Index) then
		Reference_Scan.Scan_Expression 
		   (Index, Ctx.Set 
			      (The_Context, 
			       Basic_Context => Rvs.A_Read, 
			       Weight => 5, 
			       Add_Data_Access_Context => Rvs.An_Entry_Index));
	    end if;

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Statement);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Statement);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Statement);
	    raise;

    end Scan_Accept_Statement_Non_Region_Subelements;

--| +-------------------------------------------------------------------------+
--| | SCAN_ASSIGNMENT_STATEMENT (local)                                       |
--| +-------------------------------------------------------------------------+

    procedure Scan_Assignment_Statement (The_Statement : in Asis.Statement; 
					 The_Context : in Context) is

	Puid : constant String := "Scan_Assignment_Statement";

    begin

	declare
	    Left_Side : Asis.Expression := 
	       Asis_S.Object_Assigned_To (The_Statement);
	    Right_Side : Asis.Expression := 
	       Asis_S.Assignment_Expression (The_Statement);
	begin

	    if Trace.On then
		Trace.Log (The_Statement, The_Context);
		Trace.Add_Level (2);
	    end if;

	    Reference_Scan.Scan_Expression 
	       (Left_Side, Ctx.Set (The_Context, 
				    Basic_Context => Rvs.An_Update, 
				    Weight => 5, 
				    Add_Data_Access_Context => 
				       Rvs.An_Assignment_Statement_Name));

	    Reference_Scan.Scan_Expression 
	       (Right_Side, 
		Ctx.Set (The_Context, 
			 Basic_Context => Rvs.A_Read, 
			 Weight => 5, 
			 Add_Data_Access_Context => 
			    Rvs.An_Assignment_Statement_Expression));

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Statement);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Statement);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Statement);
	    raise;

    end Scan_Assignment_Statement;

--| +-------------------------------------------------------------------------+
--| | SCAN_BLOCK_STATEMENT (local)                                            |
--| +-------------------------------------------------------------------------+

    procedure Scan_Block_Statement (The_Statement : in Asis.Statement; 
				    The_Context : in Context) is

	Puid : constant String := "Scan_Block_Statement";

    begin

	declare
	    Declarations : constant Asis.Declarative_Item_List := 
	       Asis_S.Declarative_Items 
		  (The_Statement, Reference_Scan.
				     Include_Block_Statement_Pragmas);
	    Handlers : constant Asis.Exception_Handler_List := 
	       Asis_S.Block_Exception_Handlers (The_Statement);
	    Statements : constant Asis.Statement_List := 
	       Asis_S.Block_Body_Statements 
		  (The_Statement, Reference_Scan.
				     Include_Block_Statement_Pragmas);
	begin

	    if Trace.On then
		Trace.Log (The_Statement, The_Context);
		Trace.Add_Level (Declarations'Length + 
				 Statements'Length + Handlers'Length);
	    end if;

	    Reference_Scan.Scan_Declarative_Item_List 
	       (Declarations, The_Context);

	    Reference_Scan.Scan_Statement_List (Statements, The_Context);

	    Reference_Scan.Scan_Exception_Handler_List (Handlers, The_Context);

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Statement);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Statement);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Statement);
	    raise;

    end Scan_Block_Statement;

--| +-------------------------------------------------------------------------+
--| | SCAN_CASE_STATEMENT (local)                                             |
--| +-------------------------------------------------------------------------+

    procedure Scan_Case_Statement (The_Statement : in Asis.Statement; 
				   The_Context : in Context) is

	Puid : constant String := "Scan_Case_Statement";

    begin

	declare
	    Alternatives : constant Asis.Case_Statement_Alternative_List := 
	       Asis_S.Case_Statement_Alternatives 
		  (The_Statement, Reference_Scan.
				     Include_Case_Statement_Pragmas);
	    Case_Expression : Asis.Expression := 
	       Asis_S.Case_Expression (The_Statement);
	begin

	    if Trace.On then
		Trace.Log (The_Statement, The_Context);
		Trace.Add_Level (1 + Alternatives'Length);
	    end if;

	    Reference_Scan.Scan_Expression 
	       (Case_Expression, 
		Ctx.Set (The_Context, 
			 Basic_Context => Rvs.A_Read, 
			 Weight => 5, 
			 Add_Data_Access_Context => Rvs.A_Case_Statement));

	    Reference_Scan.Scan_Case_Statement_Alternative_List 
	       (Alternatives, The_Context);

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Statement);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Statement);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Statement);
	    raise;

    end Scan_Case_Statement;

--| +-------------------------------------------------------------------------+
--| | SCAN_CODE_STATEMENT (local)                                             |
--| +-------------------------------------------------------------------------+

    procedure Scan_Code_Statement (The_Statement : in Asis.Statement; 
				   The_Context : in Context) is

	Puid : constant String := "Scan_Code_Statement";

    begin

	declare
	    Aggregate : Asis.Expression := 
	       Asis_S.Qualified_Expression (The_Statement);
	begin

	    if Trace.On then
		Trace.Log (The_Statement, The_Context);
		Trace.Add_Level (1);
	    end if;

	    Reference_Scan.Scan_Expression 
	       (Aggregate, Ctx.Set (The_Context, 
				    Type_Mark_Context => Rvs.A_Code_Statement));

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Statement);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Statement);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Statement);
	    raise;

    end Scan_Code_Statement;

--| +-------------------------------------------------------------------------+
--| | SCAN_CONDITIONAL_ENTRY_CALL_STATEMENT (local)                           |
--| +-------------------------------------------------------------------------+

    procedure Scan_Conditional_Entry_Call_Statement 
		 (The_Statement : in Asis.Statement; 
		  The_Context : in Context) is

	Puid : constant String := "Scan_Conditional_Entry_Call_Statement";

    begin

	declare
	    Arms : constant Asis.Select_Statement_Arm_List := 
	       Asis_S.Select_Statement_Arms (The_Statement);
	begin

	    if Trace.On then
		Trace.Log (The_Statement, The_Context);
		Trace.Add_Level (Arms'Length);
	    end if;

	    Reference_Scan.Scan_Select_Statement_Arm_List (Arms, The_Context);

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Statement);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Statement);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Statement);
	    raise;

    end Scan_Conditional_Entry_Call_Statement;

--| +-------------------------------------------------------------------------+
--| | SCAN_DELAY_STATEMENT (local)                                            |
--| +-------------------------------------------------------------------------+

    procedure Scan_Delay_Statement (The_Statement : in Asis.Statement; 
				    The_Context : in Context) is

	Puid : constant String := "Scan_Delay_Statement";

    begin

	declare
	    Amount : Asis.Expression := Asis_S.Delay_Expression (The_Statement);
	begin

	    if Trace.On then
		Trace.Log (The_Statement, The_Context);
		Trace.Add_Level (1);
	    end if;

	    Reference_Scan.Scan_Expression 
	       (Amount, Ctx.Set 
			   (The_Context, 
			    Basic_Context => Rvs.A_Read, 
			    Weight => 5, 
			    Add_Data_Access_Context => Rvs.A_Delay_Statement));

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Statement);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Statement);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Statement);
	    raise;

    end Scan_Delay_Statement;

--| +-------------------------------------------------------------------------+
--| | SCAN_ENTRY_CALL_STATEMENT (local)                                       |
--| +-------------------------------------------------------------------------+

    procedure Scan_Entry_Call_Statement (The_Statement : in Asis.Statement; 
					 The_Context : in Context) is

	Puid : constant String := "Scan_Entry_Call_Statement";

    begin

	declare
	    Arguments : constant Asis.Association_List := 
	       Asis_S.Call_Parameters 
		  (The_Statement, Reference_Scan.
				     Normalize_Entry_Call_Parameters);
	    Name : Asis.Expression := Asis_S.Called_Name (The_Statement);
	begin

	    if Trace.On then
		Trace.Log (The_Statement, The_Context);
		Trace.Add_Level (1 + Arguments'Length);
	    end if;

	    Reference_Scan.Scan_Expression 
	       (Name, Ctx.Set (The_Context, 
			       Basic_Context => Rvs.An_Entry_Call, 
			       Weight => 90));

	    Reference_Scan.Scan_Parameter_Association_List 
	       (Arguments, The_Context);

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Statement);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Statement);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Statement);
	    raise;

    end Scan_Entry_Call_Statement;

--| +-------------------------------------------------------------------------+
--| | SCAN_EXIT_STATEMENT (local)                                             |
--| +-------------------------------------------------------------------------+

    procedure Scan_Exit_Statement (The_Statement : in Asis.Statement; 
				   The_Context : in Context) is

	Puid : constant String := "Scan_Exit_Statement";

    begin

	declare
	    Condition : Asis.Expression := 
	       Asis_S.Exit_Condition (The_Statement);
	    Loop_Name : Asis.Expression := 
	       Asis_S.Exit_Loop_Name (The_Statement);
	begin

	    if Trace.On then
		Trace.Log (The_Statement, The_Context);
		Trace.Add_Level (Reference_Scan.One_If_Present (Loop_Name) + 
				 Reference_Scan.One_If_Present (Condition));
	    end if;

	    if not Asis_E.Is_Nil (Loop_Name) then
		Reference_Scan.Scan_Expression 
		   (Loop_Name, Ctx.Set (The_Context, 
					Basic_Context => Rvs.An_Exit_Loop_Name, 
					Weight => 90));
	    end if;

	    if not Asis_E.Is_Nil (Condition) then
		Reference_Scan.Scan_Expression 
		   (Condition, Ctx.Set (The_Context, 
					Basic_Context => Rvs.A_Read, 
					Weight => 5, 
					Add_Data_Access_Context => 
					   Rvs.An_Exit_Condition));
	    end if;

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Statement);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Statement);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Statement);
	    raise;

    end Scan_Exit_Statement;

--| +-------------------------------------------------------------------------+
--| | SCAN_FOR_LOOP_STATEMENT (local)                                         |
--| +-------------------------------------------------------------------------+

    procedure Scan_For_Loop_Statement (The_Statement : in Asis.Statement; 
				       The_Context : in Context) is

	Puid : constant String := "Scan_For_Loop_Statement";

    begin

	declare
	    Scheme : Asis.Declaration := 
	       Asis_S.For_Loop_Parameter_Specification (The_Statement);
	    Statements : constant Asis.Statement_List := 
	       Asis_S.Loop_Statements 
		  (The_Statement, Reference_Scan.
				     Include_Loop_Statement_Pragmas);
	begin

	    if Trace.On then
		Trace.Log (The_Statement, The_Context);
		Trace.Add_Level (1 + Statements'Length);
	    end if;

	    Reference_Scan.Scan_Declaration (Scheme, The_Context);

	    Reference_Scan.Scan_Statement_List (Statements, The_Context);

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Statement);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Statement);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Statement);
	    raise;

    end Scan_For_Loop_Statement;

--| +-------------------------------------------------------------------------+
--| | SCAN_GOTO_STATEMENT (local)                                             |
--| +-------------------------------------------------------------------------+

    procedure Scan_Goto_Statement (The_Statement : in Asis.Statement; 
				   The_Context : in Context) is

	Puid : constant String := "Scan_Goto_Statement";

    begin

	declare
	    Destination_Label : Asis.Expression := 
	       Asis_S.Goto_Label (The_Statement);
	begin

	    if Trace.On then
		Trace.Log (The_Statement, The_Context);
		Trace.Add_Level (1);
	    end if;

	    Reference_Scan.Scan_Expression 
	       (Destination_Label, Ctx.Set (The_Context, 
					    Basic_Context => Rvs.A_Goto_Target, 
					    Weight => 90));

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Statement);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Statement);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Statement);
	    raise;

    end Scan_Goto_Statement;

--| +-------------------------------------------------------------------------+
--| | SCAN_IF_STATEMENT (local)                                               |
--| +-------------------------------------------------------------------------+

    procedure Scan_If_Statement (The_Statement : in Asis.Statement; 
				 The_Context : in Context) is

	Puid : constant String := "Scan_If_Statement";

    begin

	declare
	    Arms : constant Asis.If_Statement_Arm_List := 
	       Asis_S.If_Statement_Arms (The_Statement);
	begin

	    if Trace.On then
		Trace.Log (The_Statement, The_Context);
		Trace.Add_Level (Arms'Length);
	    end if;

	    Reference_Scan.Scan_If_Statement_Arm_List (Arms, The_Context);

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Statement);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Statement);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Statement);
	    raise;

    end Scan_If_Statement;

--| +-------------------------------------------------------------------------+
--| | SCAN_NULL_STATEMENT (local)                                             |
--| +-------------------------------------------------------------------------+

    procedure Scan_Null_Statement (The_Statement : in Asis.Statement; 
				   The_Context : in Context) is

	Puid : constant String := "Scan_Null_Statement";

    begin

	if Trace.On then
	    Trace.Log (The_Statement, The_Context);
	end if;

    end Scan_Null_Statement;

--| +-------------------------------------------------------------------------+
--| | SCAN_PROCEDURE_CALL_STATEMENT (local)                                   |
--| +-------------------------------------------------------------------------+

    procedure Scan_Procedure_Call_Statement (The_Statement : in Asis.Statement; 
					     The_Context : in Context) is

	Puid : constant String := "Scan_Procedure_Call_Statement";

    begin

	declare
	    Arguments : constant Asis.Association_List := 
	       Asis_S.Call_Parameters 
		  (The_Statement, Reference_Scan.
				     Normalize_Procedure_Call_Parameters);
	    Name : Asis.Expression := Asis_S.Called_Name (The_Statement);
	begin

	    if Trace.On then
		Trace.Log (The_Statement, The_Context);
		Trace.Add_Level (1 + Arguments'Length);
	    end if;

	    Reference_Scan.Scan_Expression 
	       (Name, Ctx.Set (The_Context, 
			       Basic_Context => Rvs.A_Procedure_Call, 
			       Weight => 90));

	    Reference_Scan.Scan_Parameter_Association_List 
	       (Arguments, The_Context);

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Statement);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Statement);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Statement);
	    raise;

    end Scan_Procedure_Call_Statement;

--| +-------------------------------------------------------------------------+
--| | SCAN_RAISE_STATEMENT (local)                                            |
--| +-------------------------------------------------------------------------+

    procedure Scan_Raise_Statement (The_Statement : in Asis.Statement; 
				    The_Context : in Context) is

	Puid : constant String := "Scan_Raise_Statement";

    begin

	declare
	    Exception_Name : Asis.Expression := 
	       Asis_S.Raised_Exception (The_Statement);
	begin

	    if Trace.On then
		Trace.Log (The_Statement, The_Context);
		Trace.Add_Level (Reference_Scan.One_If_Present 
				    (Exception_Name));
	    end if;

	    if not Asis_E.Is_Nil (Exception_Name) then
		Reference_Scan.Scan_Expression 
		   (Exception_Name, Ctx.Set (The_Context, 
					     Basic_Context => Rvs.A_Raise, 
					     Weight => 90));
	    end if;

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Statement);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Statement);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Statement);
	    raise;

    end Scan_Raise_Statement;

--| +-------------------------------------------------------------------------+
--| | SCAN_RETURN_STATEMENT (local)                                           |
--| +-------------------------------------------------------------------------+

    procedure Scan_Return_Statement (The_Statement : in Asis.Statement; 
				     The_Context : in Context) is

	Puid : constant String := "Scan_Return_Statement";

    begin

	declare
	    Return_Expression : Asis.Expression := 
	       Asis_S.Return_Expression (The_Statement);
	begin

	    if Trace.On then
		Trace.Log (The_Statement, The_Context);
		Trace.Add_Level (Reference_Scan.One_If_Present 
				    (Return_Expression));
	    end if;

	    if not Asis_E.Is_Nil (Return_Expression) then
		Reference_Scan.Scan_Expression 
		   (Return_Expression, Ctx.Set (The_Context, 
						Basic_Context => Rvs.A_Read, 
						Weight => 5, 
						Add_Data_Access_Context => 
						   Rvs.A_Return_Statement));
	    end if;

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Statement);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Statement);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Statement);
	    raise;

    end Scan_Return_Statement;

--| +-------------------------------------------------------------------------+
--| | SCAN_SELECTIVE_WAIT_STATEMENT (local)                                   |
--| +-------------------------------------------------------------------------+

    procedure Scan_Selective_Wait_Statement (The_Statement : in Asis.Statement; 
					     The_Context : in Context) is

	Puid : constant String := "Scan_Selective_Wait_Statement";

    begin

	declare
	    Arms : constant Asis.Select_Statement_Arm_List := 
	       Asis_S.Select_Statement_Arms (The_Statement);
	begin

	    if Trace.On then
		Trace.Log (The_Statement, The_Context);
		Trace.Add_Level (Arms'Length);
	    end if;

	    Reference_Scan.Scan_Select_Statement_Arm_List (Arms, The_Context);

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Statement);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Statement);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Statement);
	    raise;

    end Scan_Selective_Wait_Statement;

--| +-------------------------------------------------------------------------+
--| | SCAN_SIMPLE_LOOP_STATEMENT (local)                                      |
--| +-------------------------------------------------------------------------+

    procedure Scan_Simple_Loop_Statement (The_Statement : in Asis.Statement; 
					  The_Context : in Context) is

	Puid : constant String := "Scan_Simple_Loop_Statement";

    begin

	declare
	    Statements : constant Asis.Statement_List := 
	       Asis_S.Loop_Statements 
		  (The_Statement, Reference_Scan.
				     Include_Loop_Statement_Pragmas);
	begin

	    if Trace.On then
		Trace.Log (The_Statement, The_Context);
		Trace.Add_Level (Statements'Length);
	    end if;

	    Reference_Scan.Scan_Statement_List (Statements, The_Context);

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Statement);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Statement);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Statement);
	    raise;

    end Scan_Simple_Loop_Statement;

--| +-------------------------------------------------------------------------+
--| | SCAN_TIMED_ENTRY_CALL_STATEMENT (local)                                 |
--| +-------------------------------------------------------------------------+

    procedure Scan_Timed_Entry_Call_Statement 
		 (The_Statement : in Asis.Statement; 
		  The_Context : in Context) is

	Puid : constant String := "Scan_Timed_Entry_Call_Statement";

    begin

	declare
	    Arms : constant Asis.Select_Statement_Arm_List := 
	       Asis_S.Select_Statement_Arms (The_Statement);
	begin

	    if Trace.On then
		Trace.Log (The_Statement, The_Context);
		Trace.Add_Level (Arms'Length);
	    end if;

	    Reference_Scan.Scan_Select_Statement_Arm_List (Arms, The_Context);

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Statement);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Statement);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Statement);
	    raise;

    end Scan_Timed_Entry_Call_Statement;

--| +-------------------------------------------------------------------------+
--| | SCAN_WHILE_LOOP_STATEMENT (local)                                       |
--| +-------------------------------------------------------------------------+

    procedure Scan_While_Loop_Statement (The_Statement : in Asis.Statement; 
					 The_Context : in Context) is

	Puid : constant String := "Scan_While_Loop_Statement";

    begin

	declare
	    Condition : Asis.Expression := 
	       Asis_S.While_Condition (The_Statement);
	    Statements : constant Asis.Statement_List := 
	       Asis_S.Loop_Statements 
		  (The_Statement, Reference_Scan.
				     Include_Loop_Statement_Pragmas);
	begin

	    if Trace.On then
		Trace.Log (The_Statement, The_Context);
		Trace.Add_Level (1 + Statements'Length);
	    end if;

	    Reference_Scan.Scan_Expression 
	       (Condition, Ctx.Set (The_Context, 
				    Basic_Context => Rvs.A_Read, 
				    Weight => 5, 
				    Add_Data_Access_Context => 
				       Rvs.A_While_Loop_Condition));

	    Reference_Scan.Scan_Statement_List (Statements, The_Context);

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Statement);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Statement);
	    raise Traversal_Error;

	when Rgn.Region_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Statement);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Statement);
	    raise;

    end Scan_While_Loop_Statement;

--| +-------------------------------------------------------------------------+
--| | SCAN_STATEMENT (exported)                                               |
--| +-------------------------------------------------------------------------+

begin

    case Asis_S.Kind (The_Statement) is
	when Asis_S.A_Null_Statement =>
	    Scan_Null_Statement (The_Statement, The_Context);
	when Asis_S.An_Assignment_Statement =>
	    Scan_Assignment_Statement (The_Statement, The_Context);
	when Asis_S.A_Procedure_Call_Statement =>
	    Scan_Procedure_Call_Statement (The_Statement, The_Context);
	when Asis_S.An_Exit_Statement =>
	    Scan_Exit_Statement (The_Statement, The_Context);
	when Asis_S.A_Return_Statement =>
	    Scan_Return_Statement (The_Statement, The_Context);
	when Asis_S.A_Goto_Statement =>
	    Scan_Goto_Statement (The_Statement, The_Context);
	when Asis_S.An_Entry_Call_Statement =>
	    Scan_Entry_Call_Statement (The_Statement, The_Context);
	when Asis_S.A_Delay_Statement =>
	    Scan_Delay_Statement (The_Statement, The_Context);
	when Asis_S.An_Abort_Statement =>
	    Scan_Abort_Statement (The_Statement, The_Context);
	when Asis_S.A_Raise_Statement =>
	    Scan_Raise_Statement (The_Statement, The_Context);
	when Asis_S.A_Code_Statement =>
	    Scan_Code_Statement (The_Statement, The_Context);
	when Asis_S.An_If_Statement =>
	    Scan_If_Statement (The_Statement, The_Context);
	when Asis_S.A_Case_Statement =>
	    Scan_Case_Statement (The_Statement, The_Context);
	when Asis_S.A_Loop_Statement =>
	    if Reference_Scan.Obeying_Regions then
		if Trace.On then
		    Trace.Log (The_Statement, The_Context);
		end if;
	    else
		case Asis_S.Loop_Kind (The_Statement) is
		    when Asis_S.A_For_Loop =>
			Scan_For_Loop_Statement (The_Statement, The_Context);
		    when Asis_S.A_While_Loop =>
			Scan_While_Loop_Statement (The_Statement, The_Context);
		    when Asis_S.A_Simple_Loop =>
			Scan_Simple_Loop_Statement (The_Statement, The_Context);
		    when Asis_S.Not_A_Loop =>
			Error.Log (Error.An_Unhandled_Case, Cuid, 
				   Puid & "/1", The_Statement);
			if Reference_Scan.Raise_Exception_On_Unhandled_Case then
			    raise Traversal_Error;
			else
			    if Trace.On then
				Trace.Log (The_Statement, The_Context);
			    end if;
			end if;
		end case;
	    end if;
	when Asis_S.A_Block_Statement =>
	    if Reference_Scan.Obeying_Regions and then 
	       Rgn.Denotes_One_Or_More_Regions (The_Statement) then
		if Trace.On then
		    Trace.Log (The_Statement, The_Context);
		end if;
	    else
		Scan_Block_Statement (The_Statement, The_Context);
	    end if;
	when Asis_S.An_Accept_Statement =>
	    if Reference_Scan.Obeying_Regions then
		Scan_Accept_Statement_Non_Region_Subelements 
		   (The_Statement, The_Context);
	    else
		Scan_Accept_Statement (The_Statement, The_Context);
	    end if;
	when Asis_S.A_Selective_Wait_Statement =>
	    Scan_Selective_Wait_Statement (The_Statement, The_Context);
	when Asis_S.A_Conditional_Entry_Call_Statement =>
	    Scan_Conditional_Entry_Call_Statement (The_Statement, The_Context);
	when Asis_S.A_Timed_Entry_Call_Statement =>
	    Scan_Timed_Entry_Call_Statement (The_Statement, The_Context);
	when Asis_S.Not_A_Statement =>
	    Error.Log (Error.An_Unhandled_Case, Cuid, 
		       Puid & "/2", The_Statement);
	    if Reference_Scan.Raise_Exception_On_Unhandled_Case then
		raise Traversal_Error;
	    else
		if Trace.On then
		    Trace.Log (The_Statement, The_Context);
		end if;
	    end if;
    end case;

end Scan_Statement;
