--| +=========================================================================+
--| |                                                                         |
--| | SCAN_FOR_REFERENCES                                                     |
--| |                                                                         |
--| | Usage: scan_for_references [-tdopne] <library> [-l|-s] <unit>           |
--| |                                                                         |
--| | This program is a simple driver for the Reference_View package.  It     |
--| | accepts two command line arguments, the name of an ASIS library and     |
--| | the name of a compilation unit contained therein, and it constructs a   |
--| | reference view for the unit.                                            |
--| |                                                                         |
--| | The unit is looked up primarily as a library unit and secondarily       |
--| | as a secondary unit.  To force looking up a library unit, specify the   |
--| | "-l" option.  To force looking up a secondary unit, specify the "-s"    |
--| | option.                                                                 |
--| |                                                                         |
--| | The "-t" option causes a trace of the reference scan to be written to   |
--| | standard output.  The "-d" option causes a textual dump of the view     |
--| | to be written to standard output.  The "-o" option causes the           |
--| | view to be written to the text file "z.ctl" and the direct I/O file     |
--| | "z.ids".                                                                |
--| |                                                                         |
--| | There are some other options as well.  The "-p" option causes the       |
--| | scan to skip over pragmas.  The "-n" option causes the normalized form  |
--| | of all associations to be examined.  The "-e" option causes all         |
--| | instantiations to be expanded.                                          |
--| |                                                                         |
--| | The options can be combined, as in "-dne".                              |
--| |                                                                         |
--| | Greg Janee                                                              |
--| | General Research Corporation                                            |
--| |                                                                         |
--| +=========================================================================+

with Asis;
with Command;
with Msg_Log;
with Reference_View;
with Reference_View_Structures;
with Text_Io;

procedure Scan_For_References is

--| Standard Asis renames...

    package Asis_Cu renames Asis.Compilation_Units;
    package Asis_En renames Asis.Environment;
    package Asis_Id_Io renames Asis.Ids.Id_Io;
    package Asis_L renames Asis.Libraries;
    package Asis_Str renames Asis.Strings;

    package Cli renames Command;

    Argument_Count : Integer := Cli.Argc - 1;
    Arguments : constant Cli.String_Ptr_Array := Cli.Arguments;
    Cfile : Text_Io.File_Type;
    Examine_Library_Units_Only : Boolean := False;
    Examine_Secondary_Units_Only : Boolean := False;
    Expand_Instantiations : Boolean := False;
    Ifile : Asis_Id_Io.File_Type;
    Include_Pragmas : Boolean := True;
    Normalize_Associations : Boolean := False;
    Offset : Integer := 0;
    Output_View : Boolean := False;
    Perform_Trace : Boolean := False;
    Produce_Dump : Boolean := False;
    The_Library : Asis.Library;
    The_Unit : Asis.Compilation_Unit;
    The_View : Reference_View_Structures.Reference_View;

    Find_Error : exception;
    Usage_Error : exception;

begin

    Msg_Log.Set_Program ("sfr");

    if Argument_Count > 0 then
	declare
	    A : String renames Arguments (1).all;
	begin
	    if A'Length > 0 and then A (A'First) = '-' then
		for I in A'First + 1 .. A'Last loop
		    case A (I) is
			when 'd' =>
			    Produce_Dump := True;
			when 'e' =>
			    Expand_Instantiations := True;
			when 'n' =>
			    Normalize_Associations := True;
			when 'o' =>
			    Output_View := True;
			when 'p' =>
			    Include_Pragmas := False;
			when 't' =>
			    Perform_Trace := True;
			when others =>
			    raise Usage_Error;
		    end case;
		end loop;
		Argument_Count := Argument_Count - 1;
		Offset := 1;
	    end if;
	end;
    end if;

    if Argument_Count = 3 then
	if Arguments (2 + Offset).all = "-l" then
	    Examine_Library_Units_Only := True;
	elsif Arguments (2 + Offset).all = "-s" then
	    Examine_Secondary_Units_Only := True;
	else
	    raise Usage_Error;
	end if;
    elsif Argument_Count /= 2 then
	raise Usage_Error;
    end if;

    Asis_En.Initialize;
    Asis_L.Associate (The_Library, 
		      Asis_Str.To_Asis_String (Arguments (1 + Offset).all));
    Asis_L.Open (The_Library);

    if Examine_Library_Units_Only then
	The_Unit := Asis_Cu.Library_Unit 
		       (Arguments (Arguments'Last).all, The_Library);
	if Asis_Cu.Is_Nil (The_Unit) then
	    raise Find_Error;
	end if;
    elsif Examine_Secondary_Units_Only then
	The_Unit := Asis_Cu.Secondary_Unit 
		       (Arguments (Arguments'Last).all, The_Library);
	if Asis_Cu.Is_Nil (The_Unit) then
	    raise Find_Error;
	end if;
    else
	The_Unit := Asis_Cu.Library_Unit 
		       (Arguments (Arguments'Last).all, The_Library);
	if Asis_Cu.Is_Nil (The_Unit) then
	    The_Unit := Asis_Cu.Secondary_Unit 
			   (Arguments (Arguments'Last).all, The_Library);
	    if Asis_Cu.Is_Nil (The_Unit) then
		raise Find_Error;
	    end if;
	end if;
    end if;

    Reference_View.Construct 
       (The_View, The_Unit, Include_Pragmas, Normalize_Associations, 
	Expand_Instantiations, Perform_Trace);

    if Produce_Dump then
	Reference_View.Dump (The_View);
    end if;

    if Output_View then
	begin
	    Text_Io.Open (Cfile, Text_Io.Out_File, "z.ctl");
	exception
	    when Text_Io.Name_Error =>
		Text_Io.Create (Cfile, Text_Io.Out_File, "z.ctl");
	end;
	begin
	    Asis_Id_Io.Open (Ifile, Asis_Id_Io.Out_File, "z.ids");
	exception
	    when Asis_Id_Io.Name_Error =>
		Asis_Id_Io.Create (Ifile, Asis_Id_Io.Out_File, "z.ids");
	end;
	Reference_View.Output (The_View, Cfile, Ifile);
	Text_Io.Close (Cfile);
	Asis_Id_Io.Close (Ifile);
    end if;

    Reference_View.Free (The_View);

    Asis_L.Close (The_Library);
    Asis_L.Dissociate (The_Library);
    Asis_En.Finalize;

exception

    when Usage_Error =>
	Msg_Log.Put_Msg 
	   (Msg_Log.Error, 
	    "usage is ""scan_for_references [-tdopne] <library> [-l|-s] <unit>""");

    when Find_Error =>
	Msg_Log.Put_Msg (Msg_Log.Error, "unit not found");

    when Asis.Asis_Inappropriate_Library =>
	Msg_Log.Put_Msg 
	   (Msg_Log.Error, 
	    "exception Asis_Inappropriate_Library raised; status is " & 
	       Asis_En.Error_Kinds'Image (Asis_En.Status) & 
	       "; diagnosis follows");
	Msg_Log.Put_Msg (Msg_Log.Error, 
			 Asis_Str.To_Standard_String (Asis_En.Diagnosis));

    when Asis.Asis_Failed =>
	Msg_Log.Put_Msg (Msg_Log.Error, 
			 "exception Asis_Failed raised; status is " & 
			    Asis_En.Error_Kinds'Image (Asis_En.Status) & 
			    "; diagnosis follows");
	Msg_Log.Put_Msg (Msg_Log.Error, 
			 Asis_Str.To_Standard_String (Asis_En.Diagnosis));

    when Reference_View.Traversal_Error =>
	Msg_Log.Put_Msg (Msg_Log.Error, "exception Traversal_Error raised");

end Scan_For_References;
