--| +=========================================================================+
--| |                                                                         |
--| | SCAN_UNIT                                                               |
--| |                                                                         |
--| | Usage: scan_unit [-prne] <library> [-l|-s] <unit>                       |
--| |                                                                         |
--| | This program is a simple driver for the Scan package.  It accepts       |
--| | two command line arguments, the name of an ASIS library and the name of |
--| | a compilation unit contained therein, and it writes a textual           |
--| | representation of the Asis tree for that unit to standard output.       |
--| |                                                                         |
--| | The unit is looked up primarily as a library unit and secondarily       |
--| | as a secondary unit.  To force looking up a library unit, specify the   |
--| | "-l" option.  To force looking up a secondary unit, specify the "-s"    |
--| | option.                                                                 |
--| |                                                                         |
--| | There are some other options as well.  The "-p" option causes the       |
--| | scan to skip over pragmas.  The "-r" option causes the scan to include  |
--| | implicit record components (note:  none have been observed so far).     |
--| | The "-n" option causes the normalized form of all associations          |
--| | to be examined.  The "-e" option causes all instantiations to be        |
--| | expanded.  These options can be combined, as in "-re".                  |
--| |                                                                         |
--| | Greg Janee                                                              |
--| | General Research Corporation                                            |
--| |                                                                         |
--| +=========================================================================+

with Asis;
with Command;
with Msg_Log;
with Scan;

procedure Scan_Unit is

--| Standard Asis renames...

    package Asis_Cu renames Asis.Compilation_Units;
    package Asis_En renames Asis.Environment;
    package Asis_L renames Asis.Libraries;
    package Asis_Str renames Asis.Strings;

    package Cli renames Command;

    Argument_Count : Integer := Cli.Argc - 1;
    Arguments : constant Cli.String_Ptr_Array := Cli.Arguments;
    Examine_Library_Units_Only : Boolean := False;
    Examine_Secondary_Units_Only : Boolean := False;
    Offset : Integer := 0;
    The_Context : Scan.Context;
    The_Library : Asis.Library;
    The_Unit : Asis.Compilation_Unit;

    Find_Error : exception;
    Usage_Error : exception;

begin

    Msg_Log.Set_Program ("su");

    if Argument_Count > 0 then
	declare
	    A : String renames Arguments (1).all;
	begin
	    if A'Length > 0 and then A (A'First) = '-' then
		for I in A'First + 1 .. A'Last loop
		    case A (I) is
			when 'p' =>
			    Scan.Include_All_Pragmas (False);
			when 'r' =>
			    Scan.Include_Implicit_Record_Components := True;
			when 'n' =>
			    Scan.Normalize_All_Associations (True);
			when 'e' =>
			    Scan.Expand_All_Instantiations (True);
			when others =>
			    raise Usage_Error;
		    end case;
		end loop;
		Argument_Count := Argument_Count - 1;
		Offset := 1;
	    end if;
	end;
    end if;

    if Argument_Count = 3 then
	if Arguments (2 + Offset).all = "-l" then
	    Examine_Library_Units_Only := True;
	elsif Arguments (2 + Offset).all = "-s" then
	    Examine_Secondary_Units_Only := True;
	else
	    raise Usage_Error;
	end if;
    elsif Argument_Count /= 2 then
	raise Usage_Error;
    end if;

    Asis_En.Initialize;
    Asis_L.Associate (The_Library, 
		      Asis_Str.To_Asis_String (Arguments (1 + Offset).all));
    Asis_L.Open (The_Library);

    if Examine_Library_Units_Only then
	The_Unit := Asis_Cu.Library_Unit 
		       (Arguments (Arguments'Last).all, The_Library);
	if Asis_Cu.Is_Nil (The_Unit) then
	    raise Find_Error;
	end if;
    elsif Examine_Secondary_Units_Only then
	The_Unit := Asis_Cu.Secondary_Unit 
		       (Arguments (Arguments'Last).all, The_Library);
	if Asis_Cu.Is_Nil (The_Unit) then
	    raise Find_Error;
	end if;
    else
	The_Unit := Asis_Cu.Library_Unit 
		       (Arguments (Arguments'Last).all, The_Library);
	if Asis_Cu.Is_Nil (The_Unit) then
	    The_Unit := Asis_Cu.Secondary_Unit 
			   (Arguments (Arguments'Last).all, The_Library);
	    if Asis_Cu.Is_Nil (The_Unit) then
		raise Find_Error;
	    end if;
	end if;
    end if;

    Scan.Start_Trace;
    Scan.Scan_Compilation_Unit (The_Unit, The_Context);
    Scan.Stop_Trace;

    Asis_L.Close (The_Library);
    Asis_L.Dissociate (The_Library);
    Asis_En.Finalize;

exception

    when Usage_Error =>
	Msg_Log.Put_Msg 
	   (Msg_Log.Error, 
	    "usage is ""scan_unit [-prne] <library> [-l|-s] <unit>""");

    when Find_Error =>
	Msg_Log.Put_Msg (Msg_Log.Error, "unit not found");

    when Asis.Asis_Inappropriate_Library =>
	Msg_Log.Put_Msg 
	   (Msg_Log.Error, 
	    "exception Asis_Inappropriate_Library raised; status is " & 
	       Asis_En.Error_Kinds'Image (Asis_En.Status) & 
	       "; diagnosis follows");
	Msg_Log.Put_Msg (Msg_Log.Error, 
			 Asis_Str.To_Standard_String (Asis_En.Diagnosis));

    when Asis.Asis_Failed =>
	Msg_Log.Put_Msg (Msg_Log.Error, 
			 "exception Asis_Failed raised; status is " & 
			    Asis_En.Error_Kinds'Image (Asis_En.Status) & 
			    "; diagnosis follows");
	Msg_Log.Put_Msg (Msg_Log.Error, 
			 Asis_Str.To_Standard_String (Asis_En.Diagnosis));

    when Scan.Traversal_Error =>
	Msg_Log.Put_Msg (Msg_Log.Error, "exception Traversal_Error raised");

end Scan_Unit;
