generic
    Size : Integer;

    type Range_Type is private;
    -- Range_Type is a pure value
    -- no initialization or finalization of values of range_type is
    -- necessary
    -- = and := can be used for equality and copy

    Ignore_Case : Boolean := True;
    pragma Must_Be_Constrained (Yes => Range_Type);

package String_Map_Generic is

    type Map is private;

    function Eval (The_Map : Map; D : String) return Range_Type;

    procedure Find (The_Map : Map; 
		    D : String; 
		    R : in out Range_Type; 
		    Success : out Boolean);

    procedure Define (The_Map : in out Map; 
		      D : String; 
		      R : Range_Type; 
		      Trap_Multiples : Boolean := False);
    procedure Undefine (The_Map : in out Map; D : String);

    procedure Initialize (The_Map : out Map);
    function Is_Empty (The_Map : Map) return Boolean;
    procedure Make_Empty (The_Map : in out Map);

    procedure Copy (Target : in out Map; Source : Map);

    type Iterator is private;

    procedure Init (Iter : out Iterator; The_Map : Map);
    procedure Next (Iter : in out Iterator);
    function Value (Iter : Iterator) return String;
    function Done (Iter : Iterator) return Boolean;

    Undefined : exception;
    -- raised by eval if the domain value in not in the map

    Multiply_Defined : exception;
    -- raised by define if the domain value is already defined and
    -- the trap_multiples flag has been specified (ie. is true)

    function Nil return Map;
    function Is_Nil (The_Map : Map) return Boolean;

    function Cardinality (The_Map : Map) return Natural;

    ------------------------------------------------------
    -- Implementation Notes and Non-Standard Operations --
    ------------------------------------------------------

    -- := and = operate on references
    --   := implies sharing (introduces an alias)
    --   = means is the same set, not the same value of type set
    -- Initializing a map also makes it empty
    -- Accessing an uninitialized map will raise CONSTRAINT_ERROR.

    -- garbage may be generated

private

    subtype Index is Natural range 0 .. Size - 1;

    type Node (Size : Natural);
    type Set is access Node;

    type Table is array (Index) of Set;

    type Map_Data is
	record
	    Bucket : Table;
	    Size : Integer := 0;
	end record;

    type Map is access Map_Data;

    type Iterator is
	record
	    The_Map : Map;
	    Index_Value : Index;
	    Set_Iter : Set;
	    Done : Boolean;
	end record;

    type Node (Size : Natural) is
	record
	    Link : Set;
	    Value : Range_Type;
	    Name : String (1 .. Size);
	end record;

end String_Map_Generic;
