package String_Utilities is

    function Hash_String (S : String) return Integer;

    procedure Upper_Case (C : in out Character);
    procedure Lower_Case (C : in out Character);

    function Upper_Case (C : Character) return Character;
    function Lower_Case (C : Character) return Character;

    function Is_Upper_Case (C : Character) return Boolean;
    function Is_Lower_Case (C : Character) return Boolean;
    -- Based on whether the case translation of C is itself; for most characters
    -- BOTH will return true.

    procedure Upper_Case (S : in out String);
    procedure Lower_Case (S : in out String);

    function Upper_Case (S : String) return String;
    function Lower_Case (S : String) return String;
    -- Result has same 'First and 'Last as S

    function Number_To_String (Value : Integer; 
			       Base : Natural := 10; 
			       Width : Natural := 0; 
			       Leading : Character := ' ') return String;

    function Unsigned_Number_To_String 
		(Value : Integer;  
		 Width : Natural := 0; 
		 Leading : Character := '0';  
		 Include_Leading_Sharp : Boolean := True) return String;

    -- Returns an image in base 16, treating value as a 32-bit unsigned
    -- number.  If Include_Leading_Sharp is true, a '#' will preceed the image
    -- and any leading padding (i.e. it will not occupy the sign position).

    function Cram_Number (N : Integer;  
			  Max_Width : Integer) return String;

    -- Max_Width <= 1 is equivalent to Number_To_String (N).  To get results
    -- that look reasonable, use Max_Width >= 4.  If necessary, will attempt
    -- to remove least significant digits (replacing them by symbols such as
    -- 'k', or 'm') until the result fits into Max_Width; when this is not
    -- possible, returns the whole thing.  Examples:

    --          N       Max_Width     Result
    --   --------------------------------------
    --         1_499        1         1499
    --         1_499        2           1k
    --         1_499        3           1k
    --         1_499        4         1499
    --   721_500_000        2           1g
    --   721_500_000        3           1g
    --   721_500_000        4         722m
    --   721_500_000        5         722m
    --   721_500_000        6         722m
    --   721_500_000        7      721500k
    --   721_500_000        8      721500k
    --   721_500_000        9    721500000

    procedure String_To_Number (Source : String; 
				Target : out Integer; 
				Worked : out Boolean; 
				Base : Natural := 10);

    function Strip_Leading (From : String;  
			    Filler : Character := ' ') return String;
    -- Strips leading Filler characters.

    function Strip_Trailing (From : String;  
			     Filler : Character := ' ') return String;
    -- Strips trailing Filler characters.

    function Strip (From : String;  
		    Filler : Character := ' ') return String;
    -- Strips leading and trailing Filler characters.

    function Locate (Fragment : String; 
		     Within : String; 
		     Ignore_Case : Boolean := False) return Natural;
    function Locate (Fragment : Character; 
		     Within : String; 
		     Ignore_Case : Boolean := False) return Natural;
    -- Returns the index value (in Within) of the first occurrance of
    -- Fragment, or 0 if none found.

    function Reverse_Locate (Fragment : String; 
			     Within : String; 
			     Ignore_Case : Boolean := False) return Natural;
    function Reverse_Locate (Fragment : Character; 
			     Within : String; 
			     Ignore_Case : Boolean := False) return Natural;
    -- Returns the index value (in Within) of the last occurrance of
    -- Fragment, or 0 if none found.

    function Equal (Str1 : String;  
		    Str2 : String; 
		    Ignore_Case : Boolean := False) return Boolean;

    function Less_Than (Str1 : String;  
			Str2 : String; 
			Ignore_Case : Boolean := False) return Boolean;

    function Greater_Than (Str1 : String;  
			   Str2 : String; 
			   Ignore_Case : Boolean := False) return Boolean;

    function Partial_Match (Arg, Val : String) return Boolean;

    -- See if Arg is "=" to Val.  This check allows Arg to be a prefix.
    -- And it allows for abbreviations within the <..> notation.  The
    -- comparison IS case sensitive.

    procedure Capitalize (S : in out String);
    function Capitalize (S : String) return String;
    -- Does the "right thing" with respect to '_' etc.

    function Quote (S : String) return String;
    function Unquote (S : String) return String;
    -- Converts to (from) a quoted string, as per Ada.  For example, the
    -- string containing the 3 characters a"b will be converted to a
    -- string containing the 6 characters "a""b".

    function Printable (Source : String) return String;
    -- Convert control characters in Source to "^C" form; '^' characters
    -- are duplicated; Ascii.Del becomes "^?".

    function Is_Ada_Identifier (S : String) return Boolean;
    -- As defined by the lrm.


    generic
	type Enumeration is (<>); -- can be a numeric type
	with function Image (Value : Enumeration)  
			    return String is Enumeration'Image;
    procedure Get_Unique_Prefix (S : String; 
				 Result : out Enumeration; 
				 Prefix : out Boolean; 
				 Unique : out Boolean);

    -- Prefix is returned true iff the given string is a prefix of one of the
    -- enumeration literals. Unique is returned true iff the given string is
    -- either:
    --   (a) a unique prefix of one of the enumeration literals, or
    --   (b) an exact match.
    -- of the given string.  When Unique is true, Result contains the
    -- 'Value of the identified enumeration literal.

    generic
	Type_Name : String;
	type Discrete_Type is (<>);
	with procedure Put_Line (S : String) is <>;
	with function Image (Value : Discrete_Type)  
			    return String is Discrete_Type'Image;
    procedure Print_Help (Input : String;  
			  Is_Prefix : Boolean; 
			  Line_Length : Positive);

    -- Typical usage: When Get_Unique_Prefix (S) returns Unique => False,
    -- call Print_Help with Prefix => S.  Prints a list of possible
    -- completions.


    generic
	with procedure Visit (Line : String);
    procedure Split_Lines (S : String; 
			   Prefix : String := ""; 
			   Line_Length : Positive := 79);

    -- Breaks S into 0 or more lines (calling Visit for each). Breaks the
    -- lines such that Line'Length is as large as possible, without exceeding
    -- Line_Length, and breaking at good "split points" (control chars and
    -- various punctuation chars).

    -- One can explicitly provide line breaks by embedding Ascii.Lf.  Treats
    -- Ascii.Lf as a line separator.  Consecutive Ascii.Lf's will produce null
    -- lines as output.  The Ascii.Lfs are removed from the output.

    -- If a split point (other than Ascii.Lf) is chosen, all blanks following
    -- the split point are removed.  If the split point itself is a blank, it
    -- and any preceding blanks are also removed.

    -- Other than as described above, no characters are removed.

    -- Prefix is added to continuation lines (but not to the first line);
    -- clearly, Prefix'Length should be considerably smaller than Line_Length.


    generic
	with procedure Visit (Line : String);
    procedure Pretty_Print_Aggregate 
		 (Aggregate : String; 
		  Cram : Boolean := False; 
		  Max_Levels : Natural := Natural'Last; 
		  Indent_Amount : Natural := 4; 
		  Line_Length : Positive := 79; 
		  Propagate_Visit_Exceptions : Boolean := False);

    -- Breaks Aggregate into 0 or more lines (calling Visit for each).  Breaks
    -- the lines such that Line'Length is as large as possible, without
    -- exceeding Line_Length, and breaking at good "split points".  One can
    -- explicitly provide line breaks by embedding Ascii.Lf.  Indent_Amount
    -- specifies the number of leading blanks to be added when the pretty
    -- printer needs to begin a new indenting level.  Clearly, Indent_Amount
    -- should be considerably smaller than Line_Length.

    -- Cram = True causes it to put as much on one line as will possibly
    -- fit.  Makes for compact, but ugly, display.

    -- Assumes that Aggregate conforms to the following BNF:
    --      Image       ::= Value | Aggregate
    --      Aggregate   ::= Left_Paren [ Field [Comma Field]* ]] Right_Paren
    --      Field       ::= Name Arrow Image
    --      Name        ::= Name_Char [ Name_Char * ]
    --      Name_Char   ::= 'a' .. 'z' | 'A' .. 'Z' | '0' .. '9' |
    --                      '#' | '_' | '-' | '.' | '<' | '>' |
    --                      Ascii.Ht | Ascii.Nul | Escape_Seq
    --      Arrow       ::= '=>'
    --      Value       ::= Quoted | Raw
    --      Quoted      ::= a string starting and ending with '"'; embedded
    --                      quotes must be represented with 2 quotes, just as
    --                      in an Ada quoted string.
    --      Raw         ::= a string which does not begin with '"', ' ', '(',
    --                      or ')' and which does not contain ',' or ')'; the
    --                      string may contain Escape_Seq's.
    --      Escape_Seq  ::= Ascii.Esc <Any_Char>

    -- On output, Ascii.Ht is translated to ',' and Ascii.Nul is
    -- translated to ' '.  This translation is applied to names and raw values,
    -- but not to the contents of quoted values.

    -- Note that there is no Escape_Seq processing inside of quoted values.

    -- Exceptions propagated by Visit are handled as follows: The exception
    -- causes the operation to terminate (ie, it stops producing output).  If
    -- Propagate_ is true, the exception is propagated out of this procedure,
    -- else it is swallowed.


    -------------------------------------
    -- String_List Type and Operations --
    -------------------------------------

    type Slice is
	record
	    First : Integer;
	    Last : Integer;
	end record;
    Nil_Slice : constant Slice := (1, 0);
    -- Any slice where first > last is considered nil.

    type Slice_Array is array (Positive range <>) of Slice;
    Nil_Slice_Array : constant Slice_Array := (1 .. 0 => Nil_Slice);

    type String_List (Count : Natural; Buffer_Length : Natural) is
	record  
	    Slices : Slice_Array (1 .. Count);
	    Buffer : String (1 .. Buffer_Length);
	end record;
    Nil_String_List : constant String_List := (0, 0, Nil_Slice_Array, "");

    function Is_Nil (The_Slice : Slice) return Boolean;

    function Get_String (Buffer : String; The_Slice : Slice) return String;
    function Get_String (List : String_List; Index : Positive) return String;
    -- Extract the specified string slice.

    function Get_String_List (List : String_List;  
			      The_Slice : Slice) return String_List;
    -- Extract the portion of List specificed by The_Slice.

    function Append (List : String_List; S : String) return String_List;
    -- Add S to the end of List.

    function Merge (Left, Right : String_List) return String_List;
    -- Combine two lists into one.

    function Unique (List : String_List) return String_List;
    -- Removes duplicates from List.

    function Build (Count : Natural; S : String) return String_List;
    -- Yields a string list that consists of Count occurrences of S.

    generic
	with function Get_String return String;
	with function More_Strings return Boolean;
    function Construct_String_List return String_List;
    -- Constructs a string list based on the sequence of strings returned by
    -- Get_String.  The construction is terminated when not More_Strings.

    generic
	with function Get_String return String;
    function Build_String_List (Count : Natural;  
				Length : Natural) return String_List;
    -- Builds a string list by calling Get_String Count times.  Length is a
    -- guess as to what the buffer length needs to be.  It is ok, but less
    -- efficient, for Length to be smaller than the buffer length required.


end String_Utilities;
