with Bounded_String;
with Sort_Utilities;
with System;
---
package body String_Utilities is

    package Bounded renames Bounded_String;

    type Translate_Table is array (Character) of Character;

    Upper_Ascii : constant Translate_Table := 
       (Ascii.Nul => Ascii.Nul, 
	Ascii.Soh => Ascii.Soh, 
	Ascii.Stx => Ascii.Stx, 
	Ascii.Etx => Ascii.Etx, 
	Ascii.Eot => Ascii.Eot, 
	Ascii.Enq => Ascii.Enq, 
	Ascii.Ack => Ascii.Ack, 
	Ascii.Bel => Ascii.Bel, 
	Ascii.Bs => Ascii.Bs, 
	Ascii.Ht => Ascii.Ht, 
	Ascii.Lf => Ascii.Lf, 
	Ascii.Vt => Ascii.Vt, 
	Ascii.Ff => Ascii.Ff, 
	Ascii.Cr => Ascii.Cr, 
	Ascii.So => Ascii.So, 
	Ascii.Si => Ascii.Si, 
	Ascii.Dle => Ascii.Dle, 
	Ascii.Dc1 => Ascii.Dc1, 
	Ascii.Dc2 => Ascii.Dc2, 
	Ascii.Dc3 => Ascii.Dc3, 
	Ascii.Dc4 => Ascii.Dc4, 
	Ascii.Nak => Ascii.Nak, 
	Ascii.Syn => Ascii.Syn, 
	Ascii.Etb => Ascii.Etb, 
	Ascii.Can => Ascii.Can, 
	Ascii.Em => Ascii.Em, 
	Ascii.Sub => Ascii.Sub, 
	Ascii.Esc => Ascii.Esc, 
	Ascii.Fs => Ascii.Fs, 
	Ascii.Gs => Ascii.Gs, 
	Ascii.Rs => Ascii.Rs, 
	Ascii.Us => Ascii.Us, 
	' ' => ' ', 
	'!' => '!', 
	'"' => '"', 
	'#' => '#', 
	'$' => '$', 
	'%' => '%', 
	'&' => '&', 
	''' => ''', 
	'(' => '(', 
	')' => ')', 
	'*' => '*', 
	'+' => '+', 
	',' => ',', 
	'-' => '-', 
	'.' => '.', 
	'/' => '/', 
	'0' => '0', 
	'1' => '1', 
	'2' => '2', 
	'3' => '3', 
	'4' => '4', 
	'5' => '5', 
	'6' => '6', 
	'7' => '7', 
	'8' => '8', 
	'9' => '9', 
	':' => ':', 
	';' => ';', 
	'<' => '<', 
	'=' => '=', 
	'>' => '>', 
	'?' => '?', 
	'@' => '@', 
	'A' => 'A', 
	'B' => 'B', 
	'C' => 'C', 
	'D' => 'D', 
	'E' => 'E', 
	'F' => 'F', 
	'G' => 'G', 
	'H' => 'H', 
	'I' => 'I', 
	'J' => 'J', 
	'K' => 'K', 
	'L' => 'L', 
	'M' => 'M', 
	'N' => 'N', 
	'O' => 'O', 
	'P' => 'P', 
	'Q' => 'Q', 
	'R' => 'R', 
	'S' => 'S', 
	'T' => 'T', 
	'U' => 'U', 
	'V' => 'V', 
	'W' => 'W', 
	'X' => 'X', 
	'Y' => 'Y', 
	'Z' => 'Z', 
	'[' => '[', 
	'\' => '\', 
	']' => ']', 
	'^' => '^', 
	'_' => '_', 
	'`' => '`', 
	'a' => 'A', 
	'b' => 'B', 
	'c' => 'C', 
	'd' => 'D', 
	'e' => 'E', 
	'f' => 'F', 
	'g' => 'G', 
	'h' => 'H', 
	'i' => 'I', 
	'j' => 'J', 
	'k' => 'K', 
	'l' => 'L', 
	'm' => 'M', 
	'n' => 'N', 
	'o' => 'O', 
	'p' => 'P', 
	'q' => 'Q', 
	'r' => 'R', 
	's' => 'S', 
	't' => 'T', 
	'u' => 'U', 
	'v' => 'V', 
	'w' => 'W', 
	'x' => 'X', 
	'y' => 'Y', 
	'z' => 'Z', 
	'{' => '{', 
	'|' => '|', 
	'}' => '}', 
	'~' => '~', 
	Ascii.Del => Ascii.Del, 
	Character'Val (16#80#) => Character'Val (16#80#), 
	Character'Val (16#81#) => Character'Val (16#81#), 
	Character'Val (16#82#) => Character'Val (16#82#), 
	Character'Val (16#83#) => Character'Val (16#83#), 
	Character'Val (16#84#) => Character'Val (16#84#), 
	Character'Val (16#85#) => Character'Val (16#85#), 
	Character'Val (16#86#) => Character'Val (16#86#), 
	Character'Val (16#87#) => Character'Val (16#87#), 
	Character'Val (16#88#) => Character'Val (16#88#), 
	Character'Val (16#89#) => Character'Val (16#89#), 
	Character'Val (16#8A#) => Character'Val (16#8A#), 
	Character'Val (16#8B#) => Character'Val (16#8B#), 
	Character'Val (16#8C#) => Character'Val (16#8C#), 
	Character'Val (16#8D#) => Character'Val (16#8D#), 
	Character'Val (16#8E#) => Character'Val (16#8E#), 
	Character'Val (16#8F#) => Character'Val (16#8F#), 
	Character'Val (16#90#) => Character'Val (16#90#), 
	Character'Val (16#91#) => Character'Val (16#91#), 
	Character'Val (16#92#) => Character'Val (16#92#), 
	Character'Val (16#93#) => Character'Val (16#93#), 
	Character'Val (16#94#) => Character'Val (16#94#), 
	Character'Val (16#95#) => Character'Val (16#95#), 
	Character'Val (16#96#) => Character'Val (16#96#), 
	Character'Val (16#97#) => Character'Val (16#97#), 
	Character'Val (16#98#) => Character'Val (16#98#), 
	Character'Val (16#99#) => Character'Val (16#99#), 
	Character'Val (16#9A#) => Character'Val (16#9A#), 
	Character'Val (16#9B#) => Character'Val (16#9B#), 
	Character'Val (16#9C#) => Character'Val (16#9C#), 
	Character'Val (16#9D#) => Character'Val (16#9D#), 
	Character'Val (16#9E#) => Character'Val (16#9E#), 
	Character'Val (16#9F#) => Character'Val (16#9F#), 
	Character'Val (16#A0#) => Character'Val (16#A0#), 
	Character'Val (16#A1#) => Character'Val (16#A1#), 
	Character'Val (16#A2#) => Character'Val (16#A2#), 
	Character'Val (16#A3#) => Character'Val (16#A3#), 
	Character'Val (16#A4#) => Character'Val (16#A4#), 
	Character'Val (16#A5#) => Character'Val (16#A5#), 
	Character'Val (16#A6#) => Character'Val (16#A6#), 
	Character'Val (16#A7#) => Character'Val (16#A7#), 
	Character'Val (16#A8#) => Character'Val (16#A8#), 
	Character'Val (16#A9#) => Character'Val (16#A9#), 
	Character'Val (16#AA#) => Character'Val (16#AA#), 
	Character'Val (16#AB#) => Character'Val (16#AB#), 
	Character'Val (16#AC#) => Character'Val (16#AC#), 
	Character'Val (16#AD#) => Character'Val (16#AD#), 
	Character'Val (16#AE#) => Character'Val (16#AE#), 
	Character'Val (16#AF#) => Character'Val (16#AF#), 
	Character'Val (16#B0#) => Character'Val (16#B0#), 
	Character'Val (16#B1#) => Character'Val (16#B1#), 
	Character'Val (16#B2#) => Character'Val (16#B2#), 
	Character'Val (16#B3#) => Character'Val (16#B3#), 
	Character'Val (16#B4#) => Character'Val (16#B4#), 
	Character'Val (16#B5#) => Character'Val (16#B5#), 
	Character'Val (16#B6#) => Character'Val (16#B6#), 
	Character'Val (16#B7#) => Character'Val (16#B7#), 
	Character'Val (16#B8#) => Character'Val (16#B8#), 
	Character'Val (16#B9#) => Character'Val (16#B9#), 
	Character'Val (16#BA#) => Character'Val (16#BA#), 
	Character'Val (16#BB#) => Character'Val (16#BB#), 
	Character'Val (16#BC#) => Character'Val (16#BC#), 
	Character'Val (16#BD#) => Character'Val (16#BD#), 
	Character'Val (16#BE#) => Character'Val (16#BE#), 
	Character'Val (16#BF#) => Character'Val (16#BF#), 
	Character'Val (16#C0#) => Character'Val (16#C0#), 
	Character'Val (16#C1#) => Character'Val (16#C1#), 
	Character'Val (16#C2#) => Character'Val (16#C2#), 
	Character'Val (16#C3#) => Character'Val (16#C3#), 
	Character'Val (16#C4#) => Character'Val (16#C4#), 
	Character'Val (16#C5#) => Character'Val (16#C5#), 
	Character'Val (16#C6#) => Character'Val (16#C6#), 
	Character'Val (16#C7#) => Character'Val (16#C7#), 
	Character'Val (16#C8#) => Character'Val (16#C8#), 
	Character'Val (16#C9#) => Character'Val (16#C9#), 
	Character'Val (16#CA#) => Character'Val (16#CA#), 
	Character'Val (16#CB#) => Character'Val (16#CB#), 
	Character'Val (16#CC#) => Character'Val (16#CC#), 
	Character'Val (16#CD#) => Character'Val (16#CD#), 
	Character'Val (16#CE#) => Character'Val (16#CE#), 
	Character'Val (16#CF#) => Character'Val (16#CF#), 
	Character'Val (16#D0#) => Character'Val (16#D0#), 
	Character'Val (16#D1#) => Character'Val (16#D1#), 
	Character'Val (16#D2#) => Character'Val (16#D2#), 
	Character'Val (16#D3#) => Character'Val (16#D3#), 
	Character'Val (16#D4#) => Character'Val (16#D4#), 
	Character'Val (16#D5#) => Character'Val (16#D5#), 
	Character'Val (16#D6#) => Character'Val (16#D6#), 
	Character'Val (16#D7#) => Character'Val (16#D7#), 
	Character'Val (16#D8#) => Character'Val (16#D8#), 
	Character'Val (16#D9#) => Character'Val (16#D9#), 
	Character'Val (16#DA#) => Character'Val (16#DA#), 
	Character'Val (16#DB#) => Character'Val (16#DB#), 
	Character'Val (16#DC#) => Character'Val (16#DC#), 
	Character'Val (16#DD#) => Character'Val (16#DD#), 
	Character'Val (16#DE#) => Character'Val (16#DE#), 
	Character'Val (16#DF#) => Character'Val (16#DF#), 
	Character'Val (16#E0#) => Character'Val (16#C0#), 
	Character'Val (16#E1#) => Character'Val (16#C1#), 
	Character'Val (16#E2#) => Character'Val (16#C2#), 
	Character'Val (16#E3#) => Character'Val (16#C3#), 
	Character'Val (16#E4#) => Character'Val (16#C4#), 
	Character'Val (16#E5#) => Character'Val (16#C5#), 
	Character'Val (16#E6#) => Character'Val (16#C6#), 
	Character'Val (16#E7#) => Character'Val (16#C7#), 
	Character'Val (16#E8#) => Character'Val (16#C8#), 
	Character'Val (16#E9#) => Character'Val (16#C9#), 
	Character'Val (16#EA#) => Character'Val (16#CA#), 
	Character'Val (16#EB#) => Character'Val (16#CB#), 
	Character'Val (16#EC#) => Character'Val (16#CC#), 
	Character'Val (16#ED#) => Character'Val (16#CD#), 
	Character'Val (16#EE#) => Character'Val (16#CE#), 
	Character'Val (16#EF#) => Character'Val (16#CF#), 
	Character'Val (16#F0#) => Character'Val (16#D0#), 
	Character'Val (16#F1#) => Character'Val (16#D1#), 
	Character'Val (16#F2#) => Character'Val (16#D2#), 
	Character'Val (16#F3#) => Character'Val (16#D3#), 
	Character'Val (16#F4#) => Character'Val (16#D4#), 
	Character'Val (16#F5#) => Character'Val (16#D5#), 
	Character'Val (16#F6#) => Character'Val (16#D6#), 
	Character'Val (16#F7#) => Character'Val (16#D7#), 
	Character'Val (16#F8#) => Character'Val (16#D8#), 
	Character'Val (16#F9#) => Character'Val (16#D9#), 
	Character'Val (16#FA#) => Character'Val (16#DA#), 
	Character'Val (16#FB#) => Character'Val (16#DB#), 
	Character'Val (16#FC#) => Character'Val (16#DC#), 
	Character'Val (16#FD#) => Character'Val (16#DD#), 
	Character'Val (16#FE#) => Character'Val (16#DE#), 
	Character'Val (16#FF#) => Character'Val (16#FF#));

    Lower_Ascii : constant Translate_Table := 
       (Ascii.Nul => Ascii.Nul, 
	Ascii.Soh => Ascii.Soh, 
	Ascii.Stx => Ascii.Stx, 
	Ascii.Etx => Ascii.Etx, 
	Ascii.Eot => Ascii.Eot, 
	Ascii.Enq => Ascii.Enq, 
	Ascii.Ack => Ascii.Ack, 
	Ascii.Bel => Ascii.Bel, 
	Ascii.Bs => Ascii.Bs, 
	Ascii.Ht => Ascii.Ht, 
	Ascii.Lf => Ascii.Lf, 
	Ascii.Vt => Ascii.Vt, 
	Ascii.Ff => Ascii.Ff, 
	Ascii.Cr => Ascii.Cr, 
	Ascii.So => Ascii.So, 
	Ascii.Si => Ascii.Si, 
	Ascii.Dle => Ascii.Dle, 
	Ascii.Dc1 => Ascii.Dc1, 
	Ascii.Dc2 => Ascii.Dc2, 
	Ascii.Dc3 => Ascii.Dc3, 
	Ascii.Dc4 => Ascii.Dc4, 
	Ascii.Nak => Ascii.Nak, 
	Ascii.Syn => Ascii.Syn, 
	Ascii.Etb => Ascii.Etb, 
	Ascii.Can => Ascii.Can, 
	Ascii.Em => Ascii.Em, 
	Ascii.Sub => Ascii.Sub, 
	Ascii.Esc => Ascii.Esc, 
	Ascii.Fs => Ascii.Fs, 
	Ascii.Gs => Ascii.Gs, 
	Ascii.Rs => Ascii.Rs, 
	Ascii.Us => Ascii.Us, 
	' ' => ' ', 
	'!' => '!', 
	'"' => '"', 
	'#' => '#', 
	'$' => '$', 
	'%' => '%', 
	'&' => '&', 
	''' => ''', 
	'(' => '(', 
	')' => ')', 
	'*' => '*', 
	'+' => '+', 
	',' => ',', 
	'-' => '-', 
	'.' => '.', 
	'/' => '/', 
	'0' => '0', 
	'1' => '1', 
	'2' => '2', 
	'3' => '3', 
	'4' => '4', 
	'5' => '5', 
	'6' => '6', 
	'7' => '7', 
	'8' => '8', 
	'9' => '9', 
	':' => ':', 
	';' => ';', 
	'<' => '<', 
	'=' => '=', 
	'>' => '>', 
	'?' => '?', 
	'@' => '@', 
	'A' => 'a', 
	'B' => 'b', 
	'C' => 'c', 
	'D' => 'd', 
	'E' => 'e', 
	'F' => 'f', 
	'G' => 'g', 
	'H' => 'h', 
	'I' => 'i', 
	'J' => 'j', 
	'K' => 'k', 
	'L' => 'l', 
	'M' => 'm', 
	'N' => 'n', 
	'O' => 'o', 
	'P' => 'p', 
	'Q' => 'q', 
	'R' => 'r', 
	'S' => 's', 
	'T' => 't', 
	'U' => 'u', 
	'V' => 'v', 
	'W' => 'w', 
	'X' => 'x', 
	'Y' => 'y', 
	'Z' => 'z', 
	'[' => '[', 
	'\' => '\', 
	']' => ']', 
	'^' => '^', 
	'_' => '_', 
	'`' => '`', 
	'a' => 'a', 
	'b' => 'b', 
	'c' => 'c', 
	'd' => 'd', 
	'e' => 'e', 
	'f' => 'f', 
	'g' => 'g', 
	'h' => 'h', 
	'i' => 'i', 
	'j' => 'j', 
	'k' => 'k', 
	'l' => 'l', 
	'm' => 'm', 
	'n' => 'n', 
	'o' => 'o', 
	'p' => 'p', 
	'q' => 'q', 
	'r' => 'r', 
	's' => 's', 
	't' => 't', 
	'u' => 'u', 
	'v' => 'v', 
	'w' => 'w', 
	'x' => 'x', 
	'y' => 'y', 
	'z' => 'z', 
	'{' => '{', 
	'|' => '|', 
	'}' => '}', 
	'~' => '~', 
	Ascii.Del => Ascii.Del, 
	Character'Val (16#80#) => Character'Val (16#80#), 
	Character'Val (16#81#) => Character'Val (16#81#), 
	Character'Val (16#82#) => Character'Val (16#82#), 
	Character'Val (16#83#) => Character'Val (16#83#), 
	Character'Val (16#84#) => Character'Val (16#84#), 
	Character'Val (16#85#) => Character'Val (16#85#), 
	Character'Val (16#86#) => Character'Val (16#86#), 
	Character'Val (16#87#) => Character'Val (16#87#), 
	Character'Val (16#88#) => Character'Val (16#88#), 
	Character'Val (16#89#) => Character'Val (16#89#), 
	Character'Val (16#8A#) => Character'Val (16#8A#), 
	Character'Val (16#8B#) => Character'Val (16#8B#), 
	Character'Val (16#8C#) => Character'Val (16#8C#), 
	Character'Val (16#8D#) => Character'Val (16#8D#), 
	Character'Val (16#8E#) => Character'Val (16#8E#), 
	Character'Val (16#8F#) => Character'Val (16#8F#), 
	Character'Val (16#90#) => Character'Val (16#90#), 
	Character'Val (16#91#) => Character'Val (16#91#), 
	Character'Val (16#92#) => Character'Val (16#92#), 
	Character'Val (16#93#) => Character'Val (16#93#), 
	Character'Val (16#94#) => Character'Val (16#94#), 
	Character'Val (16#95#) => Character'Val (16#95#), 
	Character'Val (16#96#) => Character'Val (16#96#), 
	Character'Val (16#97#) => Character'Val (16#97#), 
	Character'Val (16#98#) => Character'Val (16#98#), 
	Character'Val (16#99#) => Character'Val (16#99#), 
	Character'Val (16#9A#) => Character'Val (16#9A#), 
	Character'Val (16#9B#) => Character'Val (16#9B#), 
	Character'Val (16#9C#) => Character'Val (16#9C#), 
	Character'Val (16#9D#) => Character'Val (16#9D#), 
	Character'Val (16#9E#) => Character'Val (16#9E#), 
	Character'Val (16#9F#) => Character'Val (16#9F#), 
	Character'Val (16#A0#) => Character'Val (16#A0#), 
	Character'Val (16#A1#) => Character'Val (16#A1#), 
	Character'Val (16#A2#) => Character'Val (16#A2#), 
	Character'Val (16#A3#) => Character'Val (16#A3#), 
	Character'Val (16#A4#) => Character'Val (16#A4#), 
	Character'Val (16#A5#) => Character'Val (16#A5#), 
	Character'Val (16#A6#) => Character'Val (16#A6#), 
	Character'Val (16#A7#) => Character'Val (16#A7#), 
	Character'Val (16#A8#) => Character'Val (16#A8#), 
	Character'Val (16#A9#) => Character'Val (16#A9#), 
	Character'Val (16#AA#) => Character'Val (16#AA#), 
	Character'Val (16#AB#) => Character'Val (16#AB#), 
	Character'Val (16#AC#) => Character'Val (16#AC#), 
	Character'Val (16#AD#) => Character'Val (16#AD#), 
	Character'Val (16#AE#) => Character'Val (16#AE#), 
	Character'Val (16#AF#) => Character'Val (16#AF#), 
	Character'Val (16#B0#) => Character'Val (16#B0#), 
	Character'Val (16#B1#) => Character'Val (16#B1#), 
	Character'Val (16#B2#) => Character'Val (16#B2#), 
	Character'Val (16#B3#) => Character'Val (16#B3#), 
	Character'Val (16#B4#) => Character'Val (16#B4#), 
	Character'Val (16#B5#) => Character'Val (16#B5#), 
	Character'Val (16#B6#) => Character'Val (16#B6#), 
	Character'Val (16#B7#) => Character'Val (16#B7#), 
	Character'Val (16#B8#) => Character'Val (16#B8#), 
	Character'Val (16#B9#) => Character'Val (16#B9#), 
	Character'Val (16#BA#) => Character'Val (16#BA#), 
	Character'Val (16#BB#) => Character'Val (16#BB#), 
	Character'Val (16#BC#) => Character'Val (16#BC#), 
	Character'Val (16#BD#) => Character'Val (16#BD#), 
	Character'Val (16#BE#) => Character'Val (16#BE#), 
	Character'Val (16#BF#) => Character'Val (16#BF#), 
	Character'Val (16#C0#) => Character'Val (16#E0#), 
	Character'Val (16#C1#) => Character'Val (16#E1#), 
	Character'Val (16#C2#) => Character'Val (16#E2#), 
	Character'Val (16#C3#) => Character'Val (16#E3#), 
	Character'Val (16#C4#) => Character'Val (16#E4#), 
	Character'Val (16#C5#) => Character'Val (16#E5#), 
	Character'Val (16#C6#) => Character'Val (16#E6#), 
	Character'Val (16#C7#) => Character'Val (16#E7#), 
	Character'Val (16#C8#) => Character'Val (16#E8#), 
	Character'Val (16#C9#) => Character'Val (16#E9#), 
	Character'Val (16#CA#) => Character'Val (16#EA#), 
	Character'Val (16#CB#) => Character'Val (16#EB#), 
	Character'Val (16#CC#) => Character'Val (16#EC#), 
	Character'Val (16#CD#) => Character'Val (16#ED#), 
	Character'Val (16#CE#) => Character'Val (16#EE#), 
	Character'Val (16#CF#) => Character'Val (16#EF#), 
	Character'Val (16#D0#) => Character'Val (16#F0#), 
	Character'Val (16#D1#) => Character'Val (16#F1#), 
	Character'Val (16#D2#) => Character'Val (16#F2#), 
	Character'Val (16#D3#) => Character'Val (16#F3#), 
	Character'Val (16#D4#) => Character'Val (16#F4#), 
	Character'Val (16#D5#) => Character'Val (16#F5#), 
	Character'Val (16#D6#) => Character'Val (16#F6#), 
	Character'Val (16#D7#) => Character'Val (16#D7#), 
	Character'Val (16#D8#) => Character'Val (16#F8#), 
	Character'Val (16#D9#) => Character'Val (16#F9#), 
	Character'Val (16#DA#) => Character'Val (16#FA#), 
	Character'Val (16#DB#) => Character'Val (16#FB#), 
	Character'Val (16#DC#) => Character'Val (16#FC#), 
	Character'Val (16#DD#) => Character'Val (16#FD#), 
	Character'Val (16#DE#) => Character'Val (16#FE#), 
	Character'Val (16#DF#) => Character'Val (16#DF#), 
	Character'Val (16#E0#) => Character'Val (16#E0#), 
	Character'Val (16#E1#) => Character'Val (16#E1#), 
	Character'Val (16#E2#) => Character'Val (16#E2#), 
	Character'Val (16#E3#) => Character'Val (16#E3#), 
	Character'Val (16#E4#) => Character'Val (16#E4#), 
	Character'Val (16#E5#) => Character'Val (16#E5#), 
	Character'Val (16#E6#) => Character'Val (16#E6#), 
	Character'Val (16#E7#) => Character'Val (16#E7#), 
	Character'Val (16#E8#) => Character'Val (16#E8#), 
	Character'Val (16#E9#) => Character'Val (16#E9#), 
	Character'Val (16#EA#) => Character'Val (16#EA#), 
	Character'Val (16#EB#) => Character'Val (16#EB#), 
	Character'Val (16#EC#) => Character'Val (16#EC#), 
	Character'Val (16#ED#) => Character'Val (16#ED#), 
	Character'Val (16#EE#) => Character'Val (16#EE#), 
	Character'Val (16#EF#) => Character'Val (16#EF#), 
	Character'Val (16#F0#) => Character'Val (16#F0#), 
	Character'Val (16#F1#) => Character'Val (16#F1#), 
	Character'Val (16#F2#) => Character'Val (16#F2#), 
	Character'Val (16#F3#) => Character'Val (16#F3#), 
	Character'Val (16#F4#) => Character'Val (16#F4#), 
	Character'Val (16#F5#) => Character'Val (16#F5#), 
	Character'Val (16#F6#) => Character'Val (16#F6#), 
	Character'Val (16#F7#) => Character'Val (16#F7#), 
	Character'Val (16#F8#) => Character'Val (16#F8#), 
	Character'Val (16#F9#) => Character'Val (16#F9#), 
	Character'Val (16#FA#) => Character'Val (16#FA#), 
	Character'Val (16#FB#) => Character'Val (16#FB#), 
	Character'Val (16#FC#) => Character'Val (16#FC#), 
	Character'Val (16#FD#) => Character'Val (16#FD#), 
	Character'Val (16#FE#) => Character'Val (16#FE#), 
	Character'Val (16#FF#) => Character'Val (16#FF#));

    Base_Characters : array (0 .. 15) of Character := 
       ('0', '1', '2', '3', '4', '5', '6', '7', 
	'8', '9', 'A', 'B', 'C', 'D', 'E', 'F');


    procedure Capitalize (S : in out String) is
	Upper : Boolean := True;
    begin
	for I in S'Range loop
	    declare
		C : Character renames S (I);
	    begin
		if Upper then
		    C := Upper_Ascii (C);
		else
		    C := Lower_Ascii (C);
		end if;
		Upper := Upper_Ascii (C) = Lower_Ascii (C);
	    end;
	end loop;
    end Capitalize;


    function Capitalize (S : String) return String is
	Upper : Boolean := True;  
	New_S : String (S'Range) := S;
    begin
	for I in New_S'Range loop
	    declare
		C : Character renames New_S (I);
	    begin
		if Upper then
		    C := Upper_Ascii (C);
		else
		    C := Lower_Ascii (C);
		end if;
		Upper := Upper_Ascii (C) = Lower_Ascii (C);
	    end;
	end loop;

	return New_S;
    end Capitalize;


    function Quote (S : String) return String is
    begin
	for K in S'Range loop
	    if S (K) = '"' then
		return '"' & S (S'First .. K) & Quote (S (K + 1 .. S'Last));
	    end if;
	end loop;
	return '"' & S & '"';
    end Quote;


    function Unquote (S : String) return String is

	F : constant Integer := S'First;
	Len : constant Integer := S'Length;
	Result : String (1 .. Len);
	Last : Integer := 0;
	Have_Quote : Boolean := False;
    begin
	if Len = 0 or else S (F) /= '"' then
	    return S;
	end if;
	for K in F + 1 .. S'Last loop
	    declare
		C : constant Character := S (K);
	    begin
		if C = '"' then
		    if Have_Quote then
			Last := Last + 1;
			Result (Last) := C;
			Have_Quote := False;
		    else
			Have_Quote := True;
		    end if;
		elsif Have_Quote then
		    exit;
		else
		    Last := Last + 1;
		    Result (Last) := C;
		end if;
	    end;
	end loop;
	return Result (1 .. Last);
    end Unquote;


    function Printable (Source : String) return String is

	Result : String (1 .. Source'Length * 2);
	R : Positive := Result'First;
    begin
	for S in Source'Range loop
	    declare
		Char : constant Character := Source (S);
	    begin
		if Char < ' ' then
		    Result (R) := '^';
		    Result (R + 1) := Character'Val (Character'Pos ('@') + 
						     Character'Pos (Char));
		    R := R + 2;
		elsif Char in Character'Val (16#80#) .. 
				 Character'Val (16#9F#) then
		    Result (R) := '^';
		    Result (R + 1) := Character'Val 
					 (Character'Pos (Char) - 16#20#);
		    R := R + 2;
		elsif Char = '^' then
		    Result (R) := '^';
		    Result (R + 1) := '^';
		    R := R + 2;
		else
		    Result (R) := Char;
		    R := R + 1;
		end if;
	    end;
	end loop;
	return Result (1 .. R - 1);
    end Printable;


    function Upper_Case (C : Character) return Character is
    begin
	return Upper_Ascii (C);
    end Upper_Case;


    function Lower_Case (C : Character) return Character is
    begin
	return Lower_Ascii (C);
    end Lower_Case;

    function Is_Upper_Case (C : Character) return Boolean is
    begin
	return Upper_Ascii (C) = C;
    end Is_Upper_Case;

    function Is_Lower_Case (C : Character) return Boolean is
    begin
	return Upper_Ascii (C) = C;
    end Is_Lower_Case;

    procedure Upper_Case (C : in out Character) is
    begin
	C := Upper_Ascii (C);
    end Upper_Case;


    procedure Lower_Case (C : in out Character) is
    begin
	C := Lower_Ascii (C);
    end Lower_Case;


    procedure Upper_Case (S : in out String) is
    begin
	for I in S'Range loop
	    S (I) := Upper_Ascii (S (I));
	end loop;
    end Upper_Case;


    procedure Lower_Case (S : in out String) is
    begin
	for I in S'Range loop
	    S (I) := Lower_Ascii (S (I));
	end loop;
    end Lower_Case;


    function Upper_Case (S : String) return String is
	New_S : String (S'First .. S'Last);
    begin
	for I in S'Range loop
	    New_S (I) := Upper_Ascii (S (I));
	end loop;
	return New_S;
    end Upper_Case;


    function Lower_Case (S : String) return String is
	New_S : String (S'First .. S'Last);
    begin
	for I in S'Range loop
	    New_S (I) := Lower_Ascii (S (I));
	end loop;
	return New_S;
    end Lower_Case;

    function Hash_String (S : String) return Integer is
	-- This function is no longer case-insensitive.

	A1, A2 : Natural := 0;
	Counter : Natural := 3;
    begin
	for J in S'Range loop
	    A1 := A1 * (Character'Pos (Character'Last) + 1) + 
		     Character'Pos (S (J));
	    Counter := Counter - 1;

	    if Counter = 0 then
		A2 := (211 * A2 + A1) mod 10098131;
		-- The significance of these constants is that
		-- they are both prime, and
		--
		--    211 * 10098130 + 256**3;
		--
		-- is a little less than integer'last.

		A1 := 0;
		Counter := 3;
	    end if;
	end loop;

	return 211 * A2 + A1;
    end Hash_String;

    function Number_To_String (Value : Integer; 
			       Base : Natural := 10; 
			       Width : Natural := 0; 
			       Leading : Character := ' ') return String is

	The_Base : Integer := Base;
	Sign : Boolean := False;
	Result : Bounded.Variable_String (80);
	Ch : Integer;

	procedure N2S (Num : Integer; Width : Integer) is
	begin
	    if Num = 0 then
		-- Handle leading stuff
		for I in 1 .. Width loop
		    Bounded.Append (Result, Leading);
		end loop;
		if Sign then
		    Bounded.Append (Result, '-');
		end if;
		if Value = 0 then
		    if Width > 0 then
			Bounded.Replace (Result, Bounded.Length (Result), '0');
		    else
			Bounded.Append (Result, '0');
		    end if;
		end if;
	    else
		N2S (Num / The_Base, Width - 1);
		if not Sign then
		    Ch := Integer (Num mod The_Base);
		else
		    Ch := Integer (The_Base - (Num mod The_Base));
		    if Ch = The_Base then
			Ch := 0;
		    end if;
		end if;
		Bounded.Append (Result, Base_Characters (Ch));
	    end if;
	end N2S;

    begin
	if Base < 2 then
	    The_Base := 10; -- else this function will call itself forever
	end if;
	Bounded.Set_Length (Result, 0);
	if Value < 0 then
	    Sign := True;
	    N2S (Value, Width - 1);
	else
	    N2S (Value, Width);
	end if;

	return Bounded.Image (Result);
    end Number_To_String;


    function Unsigned_Number_To_String 
		(Value : Integer; 
		 Width : Natural := 0; 
		 Leading : Character := '0';  
		 Include_Leading_Sharp : Boolean := True) return String is

	Digit_Width : Integer := Width;

	function Make_Sharp return String is
	begin
	    if Include_Leading_Sharp then
		Digit_Width := Digit_Width - 1;
		return "#";
	    else
		return "";
	    end if;
	end Make_Sharp;

	function Hex (N : Integer; Width : Integer) return String is  
	begin  
	    if Width > 0 then
		return Number_To_String 
			  (N, Base => 16, Width => Width, Leading => Leading);  
	    else
		return Number_To_String (N, Base => 16);
	    end if;
	end Hex;

	function Split_Image (N : Integer; Width : Integer) return String is

	    type Int_Skin is
		record
		    I : Integer;
		end record;
	    Int : Int_Skin := (I => N);

	    subtype Natural_16 is Integer range 0 .. 2 ** 16 - 1;

	    type Split_Skin is
		record
		    Upper : Natural_16;
		    Lower : Natural_16;
		end record;
	    for Split_Skin use
		record at mod 4;
		    Upper at 0 range 0 .. 15;
		    Lower at 0 range 16 .. 31;
		end record;

	    pragma Assert (Split_Skin'Size = Int_Skin'Size);

	    Split : Split_Skin;
	    for Split use at Int'Address;
	begin
	    -- caller guarantees that Split.Upper /= 0 because N is negative
	    return Hex (Integer (Split.Upper), Width - 4) & 
		      Number_To_String (Integer (Split.Lower), 
					Base => 16, 
					Width => 4, 
					Leading => '0');  
	end Split_Image;

    begin  
	if Value < 0 then
	    return Make_Sharp & Split_Image (Value, Digit_Width);
	else
	    return Make_Sharp & Hex (Value, Digit_Width);
	end if;
    end Unsigned_Number_To_String;


    function Cram_Number (N : Integer;  
			  Max_Width : Integer) return String is

	Result : constant String := Number_To_String (N);
    begin
	if Result'Length <= Max_Width or else Max_Width <= 0 then
	    return Result;
	else
	    declare
		Multiplier : constant String (1 .. 4) := "kmgt";
	    begin
		for D in Multiplier'Range loop
		    declare
			Divisor : constant Integer := 1_000 ** D;
			Rounded : constant Integer := 
			   (N + Divisor / 2) / Divisor;
			Result : constant String := Number_To_String (Rounded);
		    begin
			if Result'Length + 1 <= Max_Width then
			    return Result & Multiplier (D);
			end if;
		    end;
		end loop;
	    end;
	    return Result; -- N is too big for Max_Width
	end if;
    end Cram_Number;


    procedure String_To_Number (Source : String; 
				Target : out Integer; 
				Worked : out Boolean; 
				Base : Natural := 10) is
	Sign : Integer;
	Result : Integer;
	Ch : Character;
	Char_Val : Integer;
	Prefix : Boolean := True;
    begin
	Worked := False;
	Target := 0;
	Result := 0;
	Sign := -1;

	-- Since there is one more negative Integer value then positive
	-- the math in result := result*base + val inside this loop is done
	-- calculating negative numbers, and then the result is multiplied by
	-- sign.  The value of Sign is the inverse of the actual sign of the
	-- number.

	for I in Source'Range loop
	    Ch := Source (I);
	    if Ch = ' ' then
		if not Prefix then
		    return;
		end if;
	    else
		if Prefix and Ch = '-' then
		    Sign := +1;
		else
		    if Ch in '0' .. '9' then
			Char_Val := (Character'Pos (Ch) - 48);
		    else
			Upper_Case (Ch);
			if Ch in 'A' .. 'F' then
			    Char_Val := Character'Pos (Ch) - 
					   Character'Pos ('A') + 10;
			else
			    -- set Char_Val > any legal base
			    Char_Val := 500;
			end if;
		    end if;
		    if Char_Val >= Base then
			return;
		    end if;
		    Result := Result * Integer (Base) - Integer (Char_Val);
		end if;
		Prefix := False;
	    end if;
	end loop;

	if Source'Length /= 0 and then 
	   (Source'Length > 1 or else Sign = -1) then
	    Target := Result * Sign;
	    Worked := True;
	end if;

    exception
	when others =>
	    Worked := False;
    end String_To_Number;


    function Locate (Fragment : Character; 
		     Within : String; 
		     Ignore_Case : Boolean := False) return Natural is
    begin
	if Ignore_Case then
	    for I in Within'Range loop
		if Lower_Ascii (Fragment) = Lower_Ascii (Within (I)) then
		    return I;
		end if;
	    end loop;
	else
	    for I in Within'Range loop
		if Fragment = Within (I) then
		    return I;
		end if;
	    end loop;
	end if;
	return 0;
    end Locate;


    function Locate (Fragment : String; 
		     Within : String; 
		     Ignore_Case : Boolean := False) return Natural is

	Dec_Length : constant Integer := Fragment'Length - 1;
	pragma Suppress (Range_Check, On => Dec_Length);
	First : constant Positive := Fragment'First;
	pragma Suppress (Range_Check, On => First);
	-- Suppress required since null strings can have non-Positive bounds.
	First_Char : Character;
    begin
	if Dec_Length >= 1 then
	    First_Char := Fragment (First);

	    if Ignore_Case then
		for I in Within'First .. Within'Last - Dec_Length loop
		    if Lower_Ascii (Within (I)) = Lower_Ascii (First_Char) then
			for J in reverse 1 .. Dec_Length loop
			    if Lower_Ascii (Fragment (First + J)) /= 
			       Lower_Ascii (Within (I + J)) then
				exit;
			    elsif J = 1 then
				return I;
			    end if;
			end loop;
		    end if;
		end loop;
	    else
		for I in Within'First .. Within'Last - Dec_Length loop
		    if Within (I) = First_Char then
			for J in reverse 1 .. Dec_Length loop
			    if Fragment (First + J) /= Within (I + J) then
				exit;
			    elsif J = 1 then
				return I;
			    end if;
			end loop;
		    end if;
		end loop;
	    end if;

	    return 0;

	elsif Dec_Length = 0 then
	    return Locate (Fragment (First), Within, Ignore_Case);

	else
	    return Within'First;
	end if;
    end Locate;


    function Reverse_Locate (Fragment : Character; 
			     Within : String; 
			     Ignore_Case : Boolean := False) return Natural is
    begin
	if Ignore_Case then
	    for I in reverse Within'Range loop
		if Lower_Ascii (Fragment) = Lower_Ascii (Within (I)) then
		    return I;
		end if;
	    end loop;
	else
	    for I in reverse Within'Range loop
		if Fragment = Within (I) then
		    return I;
		end if;
	    end loop;
	end if;
	return 0;
    end Reverse_Locate;


    function Reverse_Locate (Fragment : String; 
			     Within : String; 
			     Ignore_Case : Boolean := False) return Natural is

	Dec_Length : constant Integer := Fragment'Length - 1;
	pragma Suppress (Range_Check, On => Dec_Length);
	First : constant Positive := Fragment'First;
	pragma Suppress (Range_Check, On => First);
	-- Suppress required since null strings can have non-Positive bounds.
	First_Char : Character;
    begin
	if Dec_Length >= 1 then
	    First_Char := Fragment (First);

	    if Ignore_Case then
		for I in reverse Within'First .. Within'Last - Dec_Length loop
		    if Lower_Ascii (Within (I)) = Lower_Ascii (First_Char) then
			for J in reverse 1 .. Dec_Length loop
			    if Lower_Ascii (Fragment (First + J)) /= 
			       Lower_Ascii (Within (I + J)) then
				exit;
			    elsif J = 1 then
				return I + Dec_Length;
			    end if;
			end loop;
		    end if;
		end loop;
	    else
		for I in reverse Within'First .. Within'Last - Dec_Length loop
		    if Within (I) = First_Char then
			for J in reverse 1 .. Dec_Length loop
			    if Fragment (First + J) /= Within (I + J) then
				exit;
			    elsif J = 1 then
				return I + Dec_Length;
			    end if;
			end loop;
		    end if;
		end loop;
	    end if;

	    return 0;

	elsif Dec_Length = 0 then
	    return Reverse_Locate (Fragment (First), Within, Ignore_Case);

	else
	    return Within'Last;
	end if;
    end Reverse_Locate;


    function Equal (Str1 : String;  
		    Str2 : String; 
		    Ignore_Case : Boolean := False) return Boolean is

	Length : Integer := Str1'Length;
	pragma Suppress (Range_Check, On => Length);
	First1 : Positive := Str1'First;
	pragma Suppress (Range_Check, On => First1);
	First2 : Positive := Str2'First;
	pragma Suppress (Range_Check, On => First2);
    begin
	if Length = Str2'Length then
	    if Ignore_Case then
		for I in 0 .. Length - 1 loop
		    if Lower_Ascii (Str1 (First1 + I)) /= 
		       Lower_Ascii (Str2 (First2 + I)) then
			return False;
		    end if;
		end loop;

		return True;
	    else
		return Str1 = Str2;
	    end if;

	else
	    return False;
	end if;
    end Equal;


    function Less_Than 
		(Str1 : String; Str2 : String; Ignore_Case : Boolean := False) 
		return Boolean is
    begin
	if Ignore_Case then
	    return Lower_Case (Str1) < Lower_Case (Str2);
	else
	    return Str1 < Str2;
	end if;
    end Less_Than;


    function Greater_Than 
		(Str1 : String; Str2 : String; Ignore_Case : Boolean := False) 
		return Boolean is
    begin
	if Ignore_Case then
	    return Lower_Case (Str1) > Lower_Case (Str2);
	else
	    return Str1 > Str2;
	end if;
    end Greater_Than;


    function Partial_Match (Arg, Val : String) return Boolean is

	Val_First : constant Integer := Val'First;
    begin
	return Arg'Length <= Val'Length and then 
		  (Val (Val_First .. Val_First + Arg'Length - 1) = Arg or else 
		   (Arg /= "" and then Arg (Arg'Last) = '>' and then 
		    Arg (Arg'First .. Arg'Last - 1) = 
		       Val (Val_First .. Val_First + Arg'Length - 2)));
    end Partial_Match;


    function Strip_Leading 
		(From : String; Filler : Character := ' ') return String is
    begin
	for I in From'First .. From'Last loop
	    if From (I) /= Filler then
		return From (I .. From'Last);
	    end if;
	end loop;
	return "";
    end Strip_Leading;


    function Strip_Trailing 
		(From : String; Filler : Character := ' ') return String is
    begin
	for I in reverse From'First .. From'Last loop
	    if From (I) /= Filler then
		return From (From'First .. I);
	    end if;
	end loop;
	return "";
    end Strip_Trailing;


    function Strip (From : String; Filler : Character := ' ') return String is
    begin
	return Strip_Leading (Strip_Trailing (From, Filler), Filler);
    end Strip;


    function Is_Ada_Identifier (S : String) return Boolean is
	Have_Dash : Boolean := False;
    begin
	for K in S'Range loop
	    case S (K) is
		when 'a' .. 'z' | 'A' .. 'Z' | '0' .. '9' | 
		     Character'Val (16#C0#) .. Character'Val (16#D6#) | 
		     Character'Val (16#D8#) .. Character'Val (16#F6#) | 
		     Character'Val (16#F8#) .. Character'Val (16#FF#) =>
		    Have_Dash := False;
		when '_' =>
		    if Have_Dash or else K = S'First or else K = S'Last then
			return False;
		    end if;
		    Have_Dash := True;
		when others =>
		    return False;
	    end case;
	end loop;
	return S'Length > 0;
    end Is_Ada_Identifier;

    -------------------------------

    generic
	type Element is private;
	pragma Must_Be_Constrained (Element);

	type Index is (<>);
	type Element_Array is array (Index range <>) of Element;
	with function "<" (Left, Right : Element) return Boolean is <>;

    procedure Pure_Element_Table_Sort_Generic (Table : in out Element_Array);

    procedure Pure_Element_Table_Sort_Generic (Table : in out Element_Array) is
	pragma Suppress (Index_Check);
	pragma Suppress (Range_Check);
	First_Index : Integer := Index'Pos (Table'First);
	Last_Index : Integer := Index'Pos (Table'Last);
	J : Integer;
	Jg : Integer;
	Gap : Integer;
	Temp : Element;
    begin
	Gap := Last_Index - First_Index;
	while Gap > 0 loop
	    for I in Index'Val (Gap + First_Index) .. Table'Last loop
		J := Index'Pos (I) - Gap;
		while J >= First_Index loop
		    Jg := J + Gap;
		    declare
			Op1 : Element renames Table (Index'Val (J));
			Op2 : Element renames Table (Index'Val (Jg));
		    begin
			if Op2 < Op1 then
			    Temp := Op1;
			    Op1 := Op2;
			    Op2 := Temp;
			else
			    exit;
			end if;
		    end;
		    J := J - Gap;
		end loop;
	    end loop;
	    Gap := Gap / 2;
	end loop;
    end Pure_Element_Table_Sort_Generic;

    -------------------------------


    -------------------------------

    procedure Check_Prefix (S, Is_In : String; 
			    Prefix : out Boolean; 
			    Exact : out Boolean) is

	Bound : constant Integer := Is_In'First + S'Length - 1;
	Local_Prefix : Boolean;
    begin
	Local_Prefix := S'Length <= Is_In'Length and then 
			   S = Is_In (Is_In'First .. Bound);
	Exact := Local_Prefix and then (S'Length = Is_In'Length);
	Prefix := Local_Prefix;
    end Check_Prefix;

    -------------------------------

    package Aux_Lookup_Generic_Definitions is
	Not_Found : exception;
    end Aux_Lookup_Generic_Definitions;
    use Aux_Lookup_Generic_Definitions;

    generic
	type Enumeration is (<>);
	with function Image (Value : Enumeration)  
			    return String is Enumeration'Image;
    function Lookup_Generic (S : String; Starting_Position : Enumeration) 
			    return Enumeration;

    function Lookup_Generic (S : String; Starting_Position : Enumeration) 
			    return Enumeration is

	Up_S : constant String := Upper_Case (S);
	Prefix : Boolean;
	Exact : Boolean;
    begin
	for Enum in Starting_Position .. Enumeration'Last loop
	    Check_Prefix (Up_S, Image (Enum), Prefix, Exact);
	    if Prefix then
		return Enum;
	    end if;
	end loop;
	raise Not_Found;
    end Lookup_Generic;

    -------------------------------

    procedure Get_Unique_Prefix (S : String; 
				 Result : out Enumeration; 
				 Prefix : out Boolean; 
				 Unique : out Boolean) is

	Already_Matched : Boolean := False;
	Exact : Boolean;
	Match : Boolean;
	Up_S : constant String := Upper_Case (S);
    begin
	Prefix := False;
	Unique := False;
	Result := Enumeration'Last;
	for Enum in Enumeration loop
	    Check_Prefix (Up_S, Strip_Leading (Image (Enum)), Match, Exact);
	    if Exact then
		Result := Enum;
		Unique := True;
		Prefix := False;
		return;
	    elsif Match then
		Unique := not Already_Matched;
		Already_Matched := True;
		Prefix := True;
		Result := Enum;
	    end if;
	end loop;
    end Get_Unique_Prefix;

    -------------------------------

    generic
	Type_Name : String;
	type Enumeration is (<>);
	with function Image (Value : Enumeration)  
			    return String is Enumeration'Image;
	with procedure Put_Line (Line : String) is <>;
    procedure Display_Legal_Enumeration_Values_Generic (For_Input : String; 
							Is_Prefix : Boolean; 
							Line_Length : Positive);

    procedure Display_Legal_Enumeration_Values_Generic 
		 (For_Input : String; 
		  Is_Prefix : Boolean; 
		  Line_Length : Positive) is

	Stripped : constant String := Strip (For_Input);
	Has_Question_Mark : Boolean := False;
	Include_All_Values : Boolean := not Is_Prefix;

	function Max_Items return Integer is
	begin
	    return Enumeration'Pos (Enumeration'Last) - 
		      Enumeration'Pos (Enumeration'First) + 1;
	end Max_Items;

	function Lookup is new Lookup_Generic (Enumeration, Image);

	function Is_A_Prefix (S : String) return Boolean is
	    Dummy : Enumeration;
	begin
	    Dummy := Lookup (S, Enumeration'First);
	    return True;
	exception
	    when Not_Found =>
		return False;
	end Is_A_Prefix;

	function Normalized_Input return String is

	    First : constant Integer := Stripped'First;
	    Last : constant Integer := Stripped'Last;
	begin
	    if Stripped'Length > 0 and then Stripped (Last) = '?' then
		Has_Question_Mark := True;
		if First = Last then
		    Include_All_Values := True;
		else
		    declare
			To_Match : constant String := 
			   Stripped (First .. Last - 1);
		    begin
			Include_All_Values := not Is_A_Prefix (To_Match);
			return To_Match;
		    end;
		end if;
	    end if;
	    return Stripped;
	end Normalized_Input;

	package Aux_Name_Sort_Definitions is
	    The_Input : constant String := Normalized_Input;

	    subtype Name_String is Bounded.Variable_String (80);
	    type Name_Table is array (Natural range <>) of Name_String;
	end Aux_Name_Sort_Definitions;
	use Aux_Name_Sort_Definitions;


	function "<" (Left, Right : Bounded.Variable_String) return Boolean is
	begin
	    return Bounded.Image (Left) < Bounded.Image (Right);
	end "<";

	procedure Name_Sort is 
	   new Pure_Element_Table_Sort_Generic (Element => Name_String, 
						Index => Natural, 
						Element_Array => Name_Table);

	package Sort_The_Name_Table_Aux_Definitions is
	    Items : Natural := 0;
	    Table : Name_Table (1 .. Enumeration'Pos (Enumeration'Last) + 1);
	end Sort_The_Name_Table_Aux_Definitions;
	use Sort_The_Name_Table_Aux_Definitions;

	procedure Sort_The_Name_Table is
	    Enum : Enumeration := Enumeration'First;
	begin
	    loop
		begin
		    if not Include_All_Values then
			Enum := Lookup (The_Input, Enum);-- may raise Not_Found
		    end if;
		    Items := Items + 1;
		    Bounded.Copy (Table (Items), Image (Enum));
		    exit when Enum = Enumeration'Last;
		    Enum := Enumeration'Succ (Enum);
		exception
		    when Not_Found =>
			exit;
		end;
	    end loop;
	    Name_Sort (Table (1 .. Items));
	end Sort_The_Name_Table;

	package Print_Possible_Completions_Aux_Definitions is
	    Name_Width : constant Integer := Enumeration'Width;
	    Column_Width : constant Integer := Name_Width + 2;
	    Line_Buffer : Bounded.Variable_String (Line_Length + Column_Width);

	end Print_Possible_Completions_Aux_Definitions;
	use Print_Possible_Completions_Aux_Definitions;

	procedure Append (S : in out Bounded.Variable_String) is
	begin
	    Bounded.Set_Length (S, Column_Width);
	    Bounded.Append (Line_Buffer, S);
	end Append;

	procedure Print_Possible_Completions is

	    -- guaranteed to be positive:
	    Items_Per_Row : constant Positive := 
	       ((Line_Length - Column_Width) / Column_Width) + 1;

	    Lines : constant Natural := 
	       (Items + Items_Per_Row - 1) / Items_Per_Row;

	    Current_Item : Integer;
	begin
	    for L in 1 .. Lines loop
		Bounded.Free (Line_Buffer);
		for I in 1 .. Items_Per_Row loop
		    Current_Item := L + Lines * (I - 1);
		    exit when Current_Item > Items;
		    Append (Table (Current_Item));
		end loop;
		Put_Line (Bounded.Image (Line_Buffer));
	    end loop;
	end Print_Possible_Completions;
	--/Inline pragma Inline (Print_Possible_Completions);

	package Display_Legal_Enumeration_Values_Aux_Definitions is
	    Discrete_Type_Name : constant String := Capitalize (Type_Name);
	end Display_Legal_Enumeration_Values_Aux_Definitions;
	use Display_Legal_Enumeration_Values_Aux_Definitions;

    begin
	Sort_The_Name_Table;
	if Items > 9 and then not Has_Question_Mark then
	    if Items = Max_Items then
		Put_Line (Stripped & " is not a prefix of any of the " & 
			  Number_To_String (Items) & " possible values for " & 
			  Discrete_Type_Name & ".  Type '?' to see them.");
	    else
		Put_Line (Number_To_String (Items) & 
			  " possible completions for " & 
			  Discrete_Type_Name & ".  Type '" & 
			  Stripped & "?' to see them.");
	    end if;
	else
	    Put_Line ("Possible completions for " & Discrete_Type_Name);
	    Print_Possible_Completions;
	end if;
    end Display_Legal_Enumeration_Values_Generic;

    -------------------------------

    procedure Print_Help (Input : String; 
			  Is_Prefix : Boolean; 
			  Line_Length : Positive) is

	procedure Enum_Help is new Display_Legal_Enumeration_Values_Generic 
				      (Type_Name, Discrete_Type, Image);
    begin
	Enum_Help (Input, Is_Prefix, Line_Length);
    end Print_Help;


    -------------------------------

    procedure Split_Lines (S : String; 
			   Prefix : String := ""; 
			   Line_Length : Positive := 79) is

	Continuation : Boolean := False;

	First : Natural := S'First; -- bounds of arg to Visit
	Last : Natural;
	Next_First : Natural; -- First of the next line

	function Effective_Line_Len return Natural is
	begin
	    if Continuation and then Line_Length > Prefix'Length then
		return Line_Length - Prefix'Length;
	    else
		return Line_Length;
	    end if;
	end Effective_Line_Len;
	--/Inline pragma Inline (Effective_Line_Len);

	function Good_Split_Char (C : Character) return Boolean is
	begin
	    case C is
		when Ascii.Nul .. Ascii.Us |        -- control characters
		     Ascii.Del | ' ' | ',' | ';' | '/' | '-' | '.' =>
		    return True;
		when others =>
		    return False;
	    end case;
	end Good_Split_Char;
	--/Inline pragma Inline (Good_Split_Char);

	function Current_Line_Limit return Natural is
	    Limit : constant Integer := First + Effective_Line_Len - 1;
	begin
	    if Limit > S'Last then
		return S'Last;
	    else
		return Limit;
	    end if;
	end Current_Line_Limit;
	--/Inline pragma Inline (Current_Line_Limit);

	procedure Find_Split_Point is

	    -- Sets First and Last such that S(First..Last) should be given
	    -- to Visit.  Sets Next_First to reference the first character
	    -- to be considered for the subsequent line.

	    procedure Remove_Trailing_Blanks (Split_Pt : Integer) is
	    begin
		Last := First;
		for K in reverse First .. Split_Pt loop
		    if S (K) /= ' ' then
			Last := K;
			exit;
		    end if;
		end loop;
	    end Remove_Trailing_Blanks;

	    procedure Remove_Leading_Blanks_From_Next_Line 
			 (Split_Pt : Integer) is
	    begin
		Next_First := Split_Pt + 1;
		for K in Split_Pt + 1 .. S'Last loop
		    if S (K) /= ' ' then
			Next_First := K;
			exit;
		    end if;
		end loop;
	    end Remove_Leading_Blanks_From_Next_Line;

	    package Find_Split_Point_Aux_Definitions is
		Limit : constant Integer := Current_Line_Limit;
	    end Find_Split_Point_Aux_Definitions;
	    use Find_Split_Point_Aux_Definitions;

	begin
	    for K in First .. Limit loop
		if S (K) = Ascii.Lf then
		    -- explicit break on Ascii.Lf (which is then stripped)
		    Last := K - 1;
		    Next_First := K + 1;
		    return;
		end if;
	    end loop;

	    if Limit = S'Last then
		-- no break required
		Last := Limit;
		Next_First := Last + 1;
		return;
	    end if;

	    for K in reverse First .. Limit loop
		if Good_Split_Char (S (K)) then

		    Remove_Trailing_Blanks (Split_Pt => K);
		    Remove_Leading_Blanks_From_Next_Line (Split_Pt => K);
		    return;
		end if;
	    end loop;

	    -- all else failed; just break at Line_Length boundary:
	    Remove_Trailing_Blanks (Split_Pt => Limit);
	    Remove_Leading_Blanks_From_Next_Line (Split_Pt => Limit);
	end Find_Split_Point;
	--/Inline pragma Inline (Split_Point);

	function Current_Line return String is
	    -- includes the "delimiter"
	begin
	    if Continuation then
		return Prefix & S (First .. Last);
	    else
		return S (First .. Last);
	    end if;
	end Current_Line;
	--/Inline pragma Inline (Current_Line);

    begin
	if S'Length = 0 then
	    Visit (S);
	else
	    loop
		Find_Split_Point;
		exit when First > S'Last;
		Visit (Current_Line);
		Continuation := True;
		First := Next_First;
	    end loop;
	end if;
    end Split_Lines;

    -----------------------------

    procedure Pretty_Print_Aggregate 
		 (Aggregate : String; 
		  Cram : Boolean := False; 
		  Max_Levels : Natural := Natural'Last; 
		  Indent_Amount : Natural := 4; 
		  Line_Length : Positive := 79; 
		  Propagate_Visit_Exceptions : Boolean := False) is

	type Death_Kinds is (Unhandled, Put_Line_Error, 
			     Parser_Error, Pretty_Printer_Error);
	Death_Kind : Death_Kinds := Unhandled;

	type String_Pointer is
	    record
		First : Natural;
		Last : Natural;
	    end record;
	Null_String : constant String_Pointer := (1, 0);


	procedure Put_Line (Line : String) is
	begin
	    Visit (Line);
	exception
	    when others =>
		Death_Kind := Put_Line_Error;
		raise;
	end Put_Line;


	package Node_Aux_Definitions is
	    type Node_Kinds is (Value, Simple_Field, Aggregate_Field);
	    type Node (Kind : Node_Kinds);
	    type Node_Pointer is access Node;
	    type Node (Kind : Node_Kinds) is
		record
		    Sibling_Link : Node_Pointer;
		    case Kind is
			when Value =>
			    Unnamed_Value : String_Pointer;
			when Simple_Field =>
			    Field_Name : String_Pointer;
			    Value : String_Pointer;
			when Aggregate_Field =>
			    Aggregate_Name : String_Pointer;
			    Child_Link : Node_Pointer;
		    end case;
		end record;
	end Node_Aux_Definitions;
	use Node_Aux_Definitions;

	---- semantic routines called by parser/scanner ----

	function Begin_Aggregate return Node_Pointer is
	begin
	    return new Node'(Kind => Aggregate_Field, 
			     Aggregate_Name => Null_String, 
			     Child_Link => null, 
			     Sibling_Link => null);
	end Begin_Aggregate;
	--/Inline pragma Inline (Begin_Aggregate);

	procedure Have_Value (Result : in out Node_Pointer; 
			      Tail : in out Node_Pointer; 
			      Name_Ref : String_Pointer; 
			      Value_Ref : String_Pointer) is

	    New_Field : constant Node_Pointer := 
	       new Node'(Kind => Simple_Field, 
			 Field_Name => Name_Ref, 
			 Value => Value_Ref, 
			 Sibling_Link => null);
	begin
	    if Result.Child_Link = null then
		Result.Child_Link := New_Field;
	    else
		Tail.Sibling_Link := New_Field;
	    end if;
	    Tail := New_Field;
	end Have_Value;
	--/Inline pragma Inline (Have_Value);

	procedure Have_Aggregate_Value (Result : in out Node_Pointer; 
					Tail : in out Node_Pointer; 
					Name_Ref : String_Pointer;  
					Value_Node : Node_Pointer) is
	begin
	    Value_Node.Aggregate_Name := Name_Ref;
	    if Result.Child_Link = null then
		Result.Child_Link := Value_Node;
	    else
		Tail.Sibling_Link := Value_Node;
	    end if;
	    Tail := Value_Node;
	end Have_Aggregate_Value;

	---- combined parser/scanner ----

	package Parser_Scanner_Aux_Definitions is
	    Next : Integer := Aggregate'First; -- next char to look at

	    type State_Number is new Integer range 1 .. 12;

	end Parser_Scanner_Aux_Definitions;
	use Parser_Scanner_Aux_Definitions;

	function Blanks (Len : Natural) return String is
	    Result : String (1 .. Len) := (others => ' ');
	begin
	    return Result;
	end Blanks;

	procedure Display_Current_Location is
	    First : Integer;
	    Last : Integer;
	begin
	    First := Next - 40;
	    if First < Aggregate'First then
		First := Aggregate'First;
	    end if;
	    Last := First + Line_Length - 2;
	    if Last > Aggregate'Last then
		Last := Aggregate'Last;
	    end if;
	    Put_Line (Aggregate (First .. Last));

	    Put_Line (Blanks (Next - First) & "^");
	end Display_Current_Location;
	--/Inline pragma Inline (Display_Current_Location);

	function Received return String is
	begin
	    if Next in Aggregate'Range then
		return String'(1 => ''', 2 => Aggregate (Next), 3 => ''');
	    else
		return "end";
	    end if;
	end Received;
	--/Inline pragma Inline (Received);

	function Expected (In_State : State_Number) return String is
	begin
	    case In_State is
		when 1 =>
		    return "(";
		when 2 =>
		    return "the first character of a name, or ')'; " & 
			      "specifically, a character from the set " & 
			      "{ 'a'..'z', 'A'..'Z', '0'..'9', " & 
			      "'#', '_', '-', '.', '<', '>', Nul, Ht }";
		when 3 =>
		    return "a character appropriate for name, or ""=>""; " & 
			      "specifically, a character from the set " & 
			      "{ 'a'..'z', 'A'..'Z', '0'..'9', " & 
			      "'#', '_', '-', '.', '<', '>', Nul, Ht, '=' }";
		when 4 =>
		    return "an ""=>""";
		when 5 =>
		    return "the 2nd character from the ""=>"" symbol";
		when 6 =>
		    return 
		       "the 1st character from a value or quoted string; " & 
			  "specifically, a character that does not come from " & 
			  "the set { '(', ')', ',' }";
		when 7 =>
		    return "a character from the set { ',', ')', '""'.";
		when 8 =>
		    return "a character appropriate for a quoted string.";
		when 9 =>
		    return "a character from the set { ',', ')' }";
		when 10 =>
		    return 
		       "a character appropriate for a raw value; " & 
			  "specifically, a character that does not come from " & 
			  "the set { ',', ')' }";
		when 11 | 12 =>
		    return "how did we get here?";
	    end case;
	end Expected;
	--/Inline pragma Inline (Expected);

	procedure Fail (In_State : State_Number) is
	begin
	    Display_Current_Location;
	    Put_Line ("Parsing error: received " & Received & 
		      " (at character position" & Integer'Image (Next) & ")" &  
		      ", expected " & Expected (In_State));
	    Death_Kind := Parser_Error;
	    raise Constraint_Error;
	end Fail;

	function Parse_Aggregate return Node_Pointer is

	    Result : Node_Pointer := Begin_Aggregate;
	    Tail : Node_Pointer;

	    Token_First : Integer := -1; -- first char in name/value
	    Name_Ref : String_Pointer := Null_String;
	    State : State_Number := 1;

	    procedure Save_Name is
	    begin
		Name_Ref := (First => Token_First, Last => Next - 1);
	    end Save_Name;
	    --/Inline pragma Inline (Save_Name);

	    function Value_Ref return String_Pointer is
		Last : Integer := Next - 1;
	    begin
		loop -- to strip trailing blanks from the result
		    exit when Last < Token_First;
		    exit when Aggregate (Last) /= ' ';
		    Last := Last - 1;
		end loop;
		return (First => Token_First, Last => Last);
	    end Value_Ref;
	    --/Inline pragma Inline (Save_Value);

	begin
	    loop
		if Next > Aggregate'Last then
		    Fail (State);
		end if;

		case Aggregate (Next) is

		    when '(' =>
			case State is
			    when 1 =>
				State := 2;
			    when 6 =>
				State := 9;
				Have_Aggregate_Value 
				   (Result, Tail, Name_Ref, Parse_Aggregate);
			    when 8 | 10 =>
				null;
			    when 11 =>
				State := 3;
			    when 12 =>
				State := 10;
			    when others =>
				Fail (State);
			end case;

		    when ')' =>
			case State is
			    when 2 | 9 =>
				exit;
			    when 6 =>
				Have_Value (Result, Tail, Name_Ref, 
					    Null_String);
				exit;
			    when 7 =>
				Have_Value (Result, Tail, Name_Ref, 
					    (First => Token_First, 
					     Last => Next - 1));
				exit;

			    when 8 =>
				null;
			    when 10 =>
				Have_Value (Result, Tail, Name_Ref, Value_Ref);
				exit;
			    when 11 =>
				State := 3;
			    when 12 =>
				State := 10;
			    when others =>
				Fail (State);
			end case;

		    when '=' =>
			case State is
			    when 3 =>
				State := 5;
				Save_Name;
			    when 4 =>
				State := 5;
			    when 6 =>
				State := 10;
				Token_First := Next;
			    when 8 | 10 =>
				null;
			    when 11 =>
				State := 3;
			    when 12 =>
				State := 10;
			    when others =>
				Fail (State);
			end case;

		    when '>' =>
			case State is
			    when 2 =>
				State := 3;
				Token_First := Next;
			    when 5 =>
				State := 6;
			    when 6 =>
				State := 10;
				Token_First := Next;
			    when 3 | 8 | 10 =>
				null;
			    when 11 =>
				State := 3;
			    when 12 =>
				State := 10;
			    when others =>
				Fail (State);
			end case;

		    when ',' =>  
			case State is
			    when 6 =>
				State := 2;
				Have_Value (Result, Tail, Name_Ref, 
					    Null_String);
			    when 7 =>
				State := 2;
				Have_Value (Result, Tail, Name_Ref, 
					    (First => Token_First, 
					     Last => Next - 1));
			    when 8 =>
				null;
			    when 9 =>
				State := 2;
			    when 10 =>
				State := 2;
				Have_Value (Result, Tail, Name_Ref, Value_Ref);
			    when 11 =>
				State := 3;
			    when 12 =>
				State := 10;
			    when others =>
				Fail (State);
			end case;

		    when ' ' =>
			case State is
			    when 1 | 2 | 4 | 6 | 8 | 9 | 10 =>
				null;
			    when 3 =>
				State := 4;
				Save_Name;
			    when 7 =>
				State := 9;
				Have_Value (Result, Tail, Name_Ref, 
					    (First => Token_First, 
					     Last => Next - 1));
			    when 11 =>
				State := 3;
			    when 12 =>
				State := 10;
			    when others =>
				Fail (State);
			end case;

		    when '"' =>  
			case State is
			    when 6 =>
				State := 8;
				Token_First := Next;
			    when 7 =>
				State := 8;
			    when 8 =>
				State := 7;
			    when 10 =>
				null;
			    when 11 =>
				State := 3;
			    when 12 =>
				State := 10;
			    when others =>
				Fail (State);
			end case;

		    when 'a' .. 'z' | 'A' .. 'Z' | '0' .. '9' | '#' | 
			 '_' | '-' | '.' | '<' | Ascii.Nul | Ascii.Ht =>
			case State is
			    when 2 =>
				State := 3;
				Token_First := Next;
			    when 3 | 8 | 10 =>
				null;
			    when 6 =>
				State := 10;
				Token_First := Next;
			    when 11 =>
				State := 3;
			    when 12 =>
				State := 10;
			    when others =>
				Fail (State);
			end case;

		    when Ascii.Esc =>
			case State is
			    when 2 =>
				State := 11;
				Token_First := Next;
			    when 3 =>
				State := 11;
			    when 6 =>
				State := 12;
				Token_First := Next;
			    when 8 =>
				null;
			    when 10 =>
				State := 12;
			    when 11 =>
				State := 3;
			    when 12 =>
				State := 10;
			    when others =>
				Fail (State);
			end case;

		    when others =>
			case State is
			    when 6 =>
				State := 10;
				Token_First := Next;
			    when 8 | 10 =>
				null;
			    when 11 =>
				State := 3;
			    when 12 =>
				State := 10;
			    when others =>
				Fail (State);
			end case;
		end case;
		Next := Next + 1;
	    end loop;

	    return Result;
	end Parse_Aggregate;

	function Parse return Node_Pointer is
	begin
	    if Next > Aggregate'Last then
		return new Node'(Kind => Value, 
				 Unnamed_Value => Null_String, 
				 Sibling_Link => null);
	    end if;

	    if Aggregate (Next) = '(' then
		declare
		    Root : constant Node_Pointer := Parse_Aggregate;
		begin
		    if Next < Aggregate'Last then
			Put_Line 
			   ("warning: garbage characters beyond aggregate end");
		    end if;
		    return Root;
		end;
	    else
		-- just a simple value
		return new Node'(Kind => Value, 
				 Unnamed_Value => (First => Next, 
						   Last => Aggregate'Last), 
				 Sibling_Link => null);
	    end if;
	exception
	    when others =>
		if Death_Kind = Unhandled then
		    Put_Line ("*** unhandled parser exception at position" & 
			      Integer'Image (Next));
		end if;
		raise;
	end Parse;

	---- pretty printer ----

	function Size (S : String_Pointer) return Natural is
	begin
	    return S.Last - S.First + 1;
	end Size;

	package Pretty_Printer_Aux_Definitions is
	    Result : Bounded.Variable_String (5000);
	end Pretty_Printer_Aux_Definitions;
	use Pretty_Printer_Aux_Definitions;

	procedure Output_One_Line is new Split_Lines (Put_Line);

	procedure New_Line is
	begin
	    Output_One_Line (S => Bounded.Image (Result), 
			     Prefix => "... ", 
			     Line_Length => Line_Length);
	    Bounded.Free (Result);
	end New_Line;

	procedure Append (S : String) is
	    Fits : constant Boolean := (Result.Maximum_Length - 
					Bounded.Length (Result)) >= S'Length;
	begin
	    if Fits then
		Bounded.Append (Result, S);
	    else
		New_Line;
		Output_One_Line (S);
	    end if;
	end Append;

	function Maybe_Capitalize (S : String) return String is
	begin
	    if Locate ('#', Within => S) = 0 then
		return Capitalize (S);
	    else -- has embedded '#':
		return S;
	    end if;
	end Maybe_Capitalize;

	function Process_Escape_Characters (S : String_Pointer) return String is

	    Len : constant Integer := Size (S);
	    Copy : String (1 .. Len) := Aggregate (S.First .. S.Last);
	    Next : Integer := 1;
	    Last : Integer := Len;
	begin
	    loop
		if Copy (Next) = Ascii.Ht then
		    Copy (Next) := ',';
		end if;
		if Copy (Next) = Ascii.Nul then
		    Copy (Next) := ' ';
		end if;
		if Copy (Next) = Ascii.Esc then
		    Copy (Next .. Last - 1) := Copy (Next + 1 .. Last);
		    Last := Last - 1;
		end if;
		Next := Next + 1;
		exit when Next > Last;
	    end loop;
	    return Copy (1 .. Last);
	end Process_Escape_Characters;

	procedure Append_Name (Name : String_Pointer) is
	begin
	    Append (Maybe_Capitalize (Process_Escape_Characters (Name)));
	end Append_Name;

	function Process_Quoted (Value : String_Pointer) return String is

	    Len : constant Integer := Size (Value);
	    Copy : String (1 .. Len) := Aggregate (Value.First .. Value.Last);
	    Next : Integer := 2; -- keep leading '"'
	    Last : Integer := Len;
	begin
	    loop
		exit when Next >= Last; -- keep trailing '"'
		if Copy (Next) = '"' then
		    Copy (Next .. Last - 1) := Copy (Next + 1 .. Last);
		    Last := Last - 1;
		end if;
		Next := Next + 1;
	    end loop;
	    return Copy (1 .. Last);
	end Process_Quoted;

	procedure Append_Value (Value : String_Pointer) is
	begin
	    if Size (Value) > 0 and then Aggregate (Value.First) = '"' then
		Append (Process_Quoted (Value));
	    else
		Append (Process_Escape_Characters (Value));
	    end if;
	end Append_Value;

	function Indent_Size 
		    (Level : Natural; Before_Left_Paren : Boolean := False) 
		    return Integer is
	    Result : Integer := 1 + Level * Indent_Amount;
	begin
	    if Before_Left_Paren then
		Result := Result - 1;
	    end if;
	    return Result;
	end Indent_Size;
	--/Inline pragma Inline (Indent_Size);

	procedure Indent (Level : Natural; 
			  Before_Left_Paren : Boolean := False) is

	    Size : constant Integer := Indent_Size (Level, Before_Left_Paren);
	    Prefix : String (1 .. Size) := (others => ' ');
	begin
	    Append (Prefix);
	end Indent;
	--/Inline pragma Inline (Indent);

	function Fits_On_Rest_Of_Line 
		    (Tree : Node_Pointer; Level : Natural) return Boolean is

	    Next : Node_Pointer := Tree.Child_Link;
	    Pos : Integer := Bounded.Length (Result) + 1; -- for left paren
	begin
	    if Next /= null then
		loop
		    if Next.Kind /= Simple_Field then
			return False;
		    end if;
		    Pos := Pos + Size (Next.Field_Name);
		    Pos := Pos + 4; -- for arrow
		    Pos := Pos + Size (Next.Value);
		    Next := Next.Sibling_Link;
		    exit when Next = null;
		    Pos := Pos + 2; -- for comma
		end loop;
	    end if;
	    Pos := Pos + Level + 1; -- max # of right parens
	    return Pos <= Line_Length;
	end Fits_On_Rest_Of_Line;
	--/Inline pragma Inline (Fits_On_Rest_Of_Line);

	procedure Pretty_Print (Tree : Node_Pointer; Level : Natural) is

	    Cramming : Boolean := False;

	    procedure Print_List is
		Next : Node_Pointer := Tree.Child_Link;
	    begin
		if Next /= null then
		    loop
			Pretty_Print (Next, Level + 1);
			Next := Next.Sibling_Link;
			exit when Next = null;
			Append (", ");
			if not Cramming then
			    New_Line;
			    Indent (Level);
			end if;
		    end loop;
		end if;
		Append (")");
	    end Print_List;
	    --/Inline pragma Inline (Print_List);
	begin
	    case Tree.Kind is
		when Value =>
		    Append_Value (Tree.Unnamed_Value);
		when Simple_Field =>
		    Append_Name (Tree.Field_Name);
		    Append (" => ");
		    Append_Value (Tree.Value);
		when Aggregate_Field =>
		    if Level > 0 then
			Append_Name (Tree.Aggregate_Name);
			Append (" => ");
			Cramming := Fits_On_Rest_Of_Line (Tree, Level);
			if not Cramming and then Level < Max_Levels then
			    New_Line;
			    Indent (Level, Before_Left_Paren => True);
			end if;
		    end if;
		    if Level < Max_Levels then
			Append ("(");
			if not Cramming then
			    Cramming := Fits_On_Rest_Of_Line (Tree, Level);
			end if;
			Print_List;
		    else
			Append ("...");
		    end if;
	    end case;
	end Pretty_Print;

	procedure Start_Pretty_Print (Tree : Node_Pointer; Level : Natural) is
	begin
	    Pretty_Print (Tree, Level);
	exception
	    when others =>
		if Death_Kind = Unhandled then
		    Put_Line ("*** unhandled pretty printer exception");
		    --[ add exception information ]
		end if;
		raise;
	end Start_Pretty_Print;

    begin
	if Cram then
	    Output_One_Line (Aggregate);
	else
	    Start_Pretty_Print (Parse, Level => 0);
	    New_Line;
	end if;

    exception
	when others =>
	    if Death_Kind = Put_Line_Error then
		if Propagate_Visit_Exceptions then
		    raise;
		else
		    null; -- swallow;
		end if;
	    else
		Output_One_Line (Aggregate);
	    end if;
    end Pretty_Print_Aggregate;

    -------------------------------------
    -- String_List Type and Operations --
    -------------------------------------

    function Is_Nil (The_Slice : Slice) return Boolean is
    begin
	return The_Slice.First > The_Slice.Last;
    end Is_Nil;


    function Get_String (Buffer : String; The_Slice : Slice) return String is
    begin
	return Buffer (The_Slice.First .. The_Slice.Last);
    end Get_String;


    function Get_String (List : String_List; Index : Positive) return String is
    begin
	return Get_String (List.Buffer, List.Slices (Index));
    end Get_String;


    function Build_String_List (Count : Natural;  
				Length : Natural) return String_List is

	type String_Pointer is access String;
	type List_Node;
	type Linked_List is access List_Node;
	type List_Node is
	    record
		Text : String_Pointer;
		Next : Linked_List;
	    end record;

	Excess : Linked_List;
	Tail : Linked_List := null;
	Excess_Index : Positive;

	Slices : Slice_Array (1 .. Count);
	Buffer : String (1 .. Length);
	Char_Count : Natural := 0;

	function Is_Last_String (Index : Positive; S : String) return Boolean is
	begin  
	    if Tail /= null then
		return S = Tail.Text.all;
	    else
		return Index /= 1 and then 
			  S = Get_String (Buffer, Slices (Index - 1));
	    end if;
	end Is_Last_String;

	procedure Save_String (Index : Positive; S : String) is
	    New_Char_Count : constant Natural := Char_Count + S'Length;
	    Node : Linked_List;
	begin
	    if Is_Last_String (Index, S) then
		Slices (Index) := Slices (Index - 1);
	    else
		Slices (Index) := (Char_Count + 1, New_Char_Count);

		if New_Char_Count <= Length then
		    Buffer (Char_Count + 1 .. New_Char_Count) := S;
		else
		    Node := new List_Node'
				   (Text => new String'(S), Next => null);
		    if Tail = null then
			Excess := Node;
			Excess_Index := Index;
		    else
			Tail.Next := Node;
		    end if;
		    Tail := Node;
		end if;

		Char_Count := New_Char_Count;
	    end if;
	end Save_String;
    begin
	for I in 1 .. Count loop
	    Save_String (I, Get_String);
	end loop;

	if Excess = null then
	    return (Count, Char_Count, Slices, Buffer (1 .. Char_Count));
	end if;

	declare
	    Result : String_List (Count, Char_Count);
	    Last_Pos : Natural := Slices (Excess_Index).First - 1;

	    procedure Add_String (S : String) is
		First : Positive := Last_Pos + 1;
	    begin
		Last_Pos := Last_Pos + S'Length;
		Result.Buffer (First .. Last_Pos) := S;
	    end Add_String;
	begin
	    Result.Slices := Slices;
	    Result.Buffer (1 .. Last_Pos) := Buffer (1 .. Last_Pos);

	    while Excess /= null loop
		Add_String (Excess.Text.all);
		Excess := Excess.Next;
	    end loop;

	    pragma Assert (Last_Pos = Char_Count);
	    return Result;
	end;
    end Build_String_List;



    function Construct_String_List return String_List is

	type String_Pointer is access String;
	type List_Node;
	type Linked_List is access List_Node;
	type List_Node is
	    record
		Text : String_Pointer;
		Next : Linked_List;
	    end record;

	List : Linked_List;
	Tail : Linked_List := null;
	String_Count : Natural := 0;
	Char_Count : Natural := 0;

	procedure Save_String (S : String) is
	    Node : constant Linked_List := 
	       new List_Node'(Text => new String'(S), Next => null);
	begin  
	    String_Count := String_Count + 1;
	    Char_Count := Char_Count + S'Length;

	    if Tail = null then
		List := Node;
	    else
		Tail.Next := Node;
	    end if;
	    Tail := Node;
	end Save_String;

	function Get_Saved_String return String is
	    Node : constant Linked_List := List;
	begin
	    List := List.Next;
	    return Node.Text.all;
	end Get_Saved_String;

	function Build_List is new Build_String_List (Get_Saved_String);
    begin
	while More_Strings loop
	    Save_String (Get_String);
	end loop;

	return Build_List (String_Count, Char_Count);
    end Construct_String_List;



    function Get_String_List (List : String_List;  
			      The_Slice : Slice) return String_List is

	Next : Integer := The_Slice.First;

	function More_Strings return Boolean is
	begin
	    return Next <= The_Slice.Last;
	end More_Strings;

	function Get_String return String is
	begin  
	    Next := Next + 1;
	    return Get_String (List, Next - 1);
	end Get_String;

	function Construct is new Construct_String_List 
				     (More_Strings => More_Strings, 
				      Get_String => Get_String);
    begin
	if Is_Nil (The_Slice) then
	    return Nil_String_List;
	elsif The_Slice.First = The_Slice.Last then
	    return Build (1, Get_String (List, The_Slice.First));
	else
	    return Construct;
	end if;
    end Get_String_List;


    function Append (List : String_List; S : String) return String_List is
    begin
	if List.Count = 0 then
	    return Build (1, S);
	elsif Get_String (List, List.Count) = S then
	    return (Count => List.Count + 1, 
		    Buffer_Length => List.Buffer_Length, 
		    Slices => List.Slices & List.Slices (List.Count), 
		    Buffer => List.Buffer);
	else
	    return (Count => List.Count + 1, 
		    Buffer_Length => List.Buffer_Length + S'Length, 
		    Slices => List.Slices & Slice'
					       (List.Buffer_Length + 1, 
						List.Buffer_Length + S'Length), 
		    Buffer => List.Buffer & S);
	end if;
    end Append;


    function Merge (Left, Right : String_List) return String_List is

	Buffer : String (1 .. Left.Buffer_Length + Right.Buffer_Length);
	Slices : Slice_Array (1 .. Left.Count + Right.Count);
	Length : Natural := 0;

	procedure Add_String (S : String; I : Positive) is
	    New_Length : Natural;
	begin
	    if I = 1 or else S /= Get_String (Buffer, Slices (I - 1)) then
		New_Length := Length + S'Length;
		Buffer (Length + 1 .. New_Length) := S;
		Slices (I) := (Length + 1, New_Length);
		Length := New_Length;
	    else
		Slices (I) := Slices (I - 1);
	    end if;
	end Add_String;
    begin
	if Left.Count = 0 then
	    return Right;
	elsif Right.Count = 0 then
	    return Left;
	end if;

	Slices (1 .. Left.Count) := Left.Slices;
	Buffer (1 .. Left.Buffer_Length) := Left.Buffer;
	Length := Left.Buffer_Length;

	for I in 1 .. Right.Count loop
	    Add_String (Get_String (Right, I), Left.Count + I);
	end loop;

	return (Count => Left.Count + Right.Count, 
		Buffer_Length => Length, 
		Slices => Slices, 
		Buffer => Buffer (1 .. Length));
    end Merge;


    function Unique (List : String_List) return String_List is

	function Content (S : Slice) return String is
	begin
	    return Get_String (List.Buffer, S);
	end Content;
	pragma Inline (Content);

	procedure Remove_Duplicates (Slices : in out Slice_Array;  
				     Last : out Natural) is
	    Local_Last : Natural;
	begin
	    Local_Last := Slices'First;
	    for K in Slices'First + 1 .. Slices'Last loop
		if Content (Slices (K)) /= Content (Slices (Local_Last)) then
		    Last := Local_Last + 1;
		    Slices (Local_Last) := Slices (K);
		end if;
	    end loop;  
	    Last := Local_Last;
	end Remove_Duplicates;

	function Compare (L, R : Slice) return Boolean is
	begin
	    return Content (L) >= Content (R);
	end Compare;

	procedure Sort is new Sort_Utilities.Array_Quick_Sort 
				 (Slice, Positive, Slice_Array, Compare);

	package Unique_Aux_Definitions is
	    Temp : String_List (List.Count, List.Buffer_Length) := List;
	    Slices : Slice_Array renames Temp.Slices;
	    Last : Natural;  
	end Unique_Aux_Definitions;
	use Unique_Aux_Definitions;
    begin
	Sort (Slices, Slices'First, Slices'Last);
	Remove_Duplicates (Slices, Last);
	declare
	    Result : String_List (Last, List.Buffer_Length);
	begin
	    Result.Slices := Slices (1 .. Last);
	    Result.Buffer := List.Buffer;
	    return Result;
	end;
    end Unique;


    function Build (Count : Natural; S : String) return String_List is

	Result : String_List (Count, S'Length);
	The_Slice : constant Slice := (1, Result.Buffer_Length);
    begin
	Result.Buffer := S;
	for I in 1 .. Count loop
	    Result.Slices (I) := The_Slice;
	end loop;
	return Result;
    end Build;

end String_Utilities;
