with Calendar;

package Time_Utilities is

    Minute : constant Duration := 60.0;
    Hour : constant Duration := 3600.0;
    Day : constant Duration := 86_400.0;

    --------------------------------------------------------------------
    -- Time_Utilities.Time is a segmented version of Calendar.Time
    --         with image and value functions
    --------------------------------------------------------------------

    type Years is new Calendar.Year_Number;
    type Months is (January, February, March, April, May, June, July, 
		    August, September, October, November, December);
    type Days is new Calendar.Day_Number;

    type Hours is new Integer range 1 .. 12;
    type Minutes is new Integer range 0 .. 59;
    type Seconds is new Integer range 0 .. 59;

    type Sun_Positions is (Am, Pm);

    type Time is
	record
	    Year : Years;
	    Month : Months;
	    Day : Days;
	    Hour : Hours;
	    Minute : Minutes;
	    Second : Seconds;
	    Sun_Position : Sun_Positions;
	end record;

    function Get_Time return Time;

    function Convert_Time (Date : Calendar.Time) return Time;
    function Convert_Time (Date : Time) return Calendar.Time;

    function Nil return Time;
    function Is_Nil (Date : Time) return Boolean;

    function Nil return Calendar.Time;
    function Is_Nil (Date : Calendar.Time) return Boolean;

    type Time_Format is (Expanded,             -- 11:00:00 PM
			 Military,             -- 23:00:00
			 Short,                -- 23:00
			 Ada                 -- 23_00_00
			 );

    type Date_Format is (Expanded,             -- September 29, 1983
			 Month_Day_Year,       -- 09/29/83
			 Day_Month_Year,       -- 29-SEP-83
			 Year_Month_Day,       -- 83/09/29
			 Ada                 -- 83_09_29
			 );

    type Image_Contents is (Both, Time_Only, Date_Only);

    function Image (Date : Time; 
		    Date_Style : Date_Format := Time_Utilities.Expanded; 
		    Time_Style : Time_Format := Time_Utilities.Expanded; 
		    Contents : Image_Contents := Time_Utilities.Both) 
		   return String;

    function Value (S : String) return Time;
    -- Gives incorrect results for dates earlier than 1924.

    --------------------------------------------------------------------
    -- Time_Utilities.Interval is a segmented version of Duration
    --        with image and value functions
    --------------------------------------------------------------------

    type Day_Count is new Integer range 0 .. Integer'Last;
    type Military_Hours is new Integer range 0 .. 23;
    type Milliseconds is new Integer range 0 .. 999;

    type Interval is
	record
	    Elapsed_Days : Day_Count;
	    Elapsed_Hours : Military_Hours;
	    Elapsed_Minutes : Minutes;
	    Elapsed_Seconds : Seconds;
	    Elapsed_Milliseconds : Milliseconds;
	end record;


    function Convert (I : Interval) return Duration;
    function Convert (D : Duration) return Interval;

    function Image (I : Interval) return String;
    function Value (S : String) return Interval;

    function Image (D : Duration) return String;

    function Duration_Until (T : Time) return Duration;
    function Duration_Until (T : Calendar.Time) return Duration;
    function Duration_Until_Next 
		(H : Military_Hours; M : Minutes := 0; S : Seconds := 0) 
		return Duration;

    -- Day of week support; Monday is 1.
    type Weekday is new Positive range 1 .. 7;

    function Day_Of_Week (T : Calendar.Time) return Weekday;
    function Day_Of_Week (T : Time := Get_Time) return Weekday;
    function Image (D : Weekday) return String;

    function "+" (D : Weekday; I : Integer) return Weekday;
    function "-" (D : Weekday; I : Integer) return Weekday;

end Time_Utilities;
