generic
    Default_Maximum_Length : Natural := 20;
package Unbounded_String is

    -------------------------------------------------------------------------------
    --  Managed Pointer Sequential Unbounded Strings:
    --  Restrictions and assumptions
    -- Storage management performed
    -- All new allocations for extensions are twice requirements
    -- CANNOT be used by multiple tasks unless user guarantees sequential use
    -- := is reference copy, use copy to assign contents
    --    uninitialized or Free'd objects are true null's and changes to one
    --    of the referents will not be reflected in the other;
    --    use Free prior to assignment to prevent garbage
    -- = is object identity, use string_utilities for comparison
    --
    -------------------------------------------------------------------------------


    subtype String_Length is Natural;
    type Variable_String is private;

    -- release storage associated with a string
    procedure Free (V : in out Variable_String);

    -- Get information about current length or contents of a string
    function Length (Source : Variable_String) return String_Length;
    function Char_At (Source : Variable_String; At_Pos : Positive) 
		     return Character;

    function Extract (Source : Variable_String; 
		      Start_Pos : Positive; 
		      End_Pos : Natural) return String;

    function Image (V : Variable_String) return String;
    function Value (S : String) return Variable_String;

    -- Image (Target) := Image (Source);
    procedure Copy (Target : in out Variable_String; Source : Variable_String);
    procedure Copy (Target : in out Variable_String; Source : String);
    procedure Copy (Target : in out Variable_String; Source : Character);

    -- Target := Source; Source := ""; with appropriate storage management
    procedure Move (Target : in out Variable_String; 
		    Source : in out Variable_String);

    -- Target := Target & Source;
    procedure Append (Target : in out Variable_String; 
		      Source : Variable_String);

    procedure Append (Target : in out Variable_String; Source : String);

    procedure Append (Target : in out Variable_String; Source : Character);

    procedure Append (Target : in out Variable_String; 
		      Source : Character; 
		      Count : String_Length);

    -- Target := Target (1..At_Pos-1) & Source & Target (At_Pos..Target'Length)
    procedure Insert (Target : in out Variable_String; 
		      At_Pos : Positive; 
		      Source : Variable_String);

    procedure Insert (Target : in out Variable_String; 
		      At_Pos : Positive; 
		      Source : String);

    procedure Insert (Target : in out Variable_String; 
		      At_Pos : Positive; 
		      Source : Character);

    procedure Insert (Target : in out Variable_String; 
		      At_Pos : Positive; 
		      Source : Character; 
		      Count : String_Length);

    -- Target (At_Pos .. At_Pos + Count -1) := "";
    procedure Delete (Target : in out Variable_String; 
		      At_Pos : Positive; 
		      Count : String_Length := 1);

    -- Target (At_Pos .. At_Pos + Source'Length - 1) := Source;
    procedure Replace (Target : in out Variable_String; 
		       At_Pos : Positive; 
		       Source : Character);

    procedure Replace (Target : in out Variable_String; 
		       At_Pos : Positive; 
		       Source : Character; 
		       Count : String_Length);

    procedure Replace (Target : in out Variable_String; 
		       At_Pos : Positive; 
		       Source : String);

    procedure Replace (Target : in out Variable_String; 
		       At_Pos : Positive; 
		       Source : Variable_String);

    -- Target'Length := New_Length;
    -- Target (Target'Length .. New_Length) := Fill_With;
    procedure Set_Length (Target : in out Variable_String; 
			  New_Length : String_Length; 
			  Fill_With : Character := ' ');

private
    type Pointer is access String;

    type Real_String;
    type Variable_String is access Real_String;
    subtype String_Bound is Integer range -1 .. Integer'Last;

    type Real_String is
	record
	    Length : String_Bound;
	    Contents : Pointer;
	    Next_Free : Variable_String;
	end record;

    Null_String : Pointer := new String (1 .. 0);

    Free_List_Item : constant String_Bound := -1;

    Free_List : Real_String := 
       Real_String'(Free_List_Item, Null_String, 
		    new Real_String'(Free_List_Item, Null_String, null));
end Unbounded_String;
