--| +=========================================================================+
--| |                                                                         |
--| | REGION_SUPPORT (body)                                                   |
--| |                                                                         |
--| | Greg Janee                                                              |
--| | General Research Corporation                                            |
--| |                                                                         |
--| | * Modified 12/12/94 by Chuck Hobin                                      | 
--| |                                                                         |
--| | Added LRM 8.1 semantic checks to functions Denoted_Regions and          |
--| | Denotes_One_Or_More_Regions.                                            |
--| |                                                                         |
--| | Added the function Defines_A_Declarative_Region.                        |
--| +=========================================================================+

with Asis_Debug_Support;
with Msg_Log;

package body Region_Support is

-- Standard renames...

    package Asis_En renames Asis.Environment;
    package Asis_Cu renames Asis.Compilation_Units;
    package Asis_D renames Asis.Declarations;
    package Asis_E renames Asis.Elements;
    package Asis_Rc renames Asis.Representation_Clauses;
    package Asis_S renames Asis.Statements;
    package Asis_Str renames Asis.Strings;
    package Asis_Td renames Asis.Type_Definitions;
    package Asis_X renames Asis.Expressions;

--| The following clause gives us direct visibility of the "&" operator for
--| Asis.Element_List's.

    use Asis_E.Operations;

--| For error reporting purposes...

    Cuid : constant String := "Region_Support";
    Region_Use_Error : exception;

--| Local subprograms.

    procedure Asis_Failure 
		 (Current_Element : in Asis.Element; Puid : in String);

    procedure Asis_Failure 
		 (Current_Unit : in Asis.Compilation_Unit; Puid : in String);

    function Diagnosis return String;

    procedure Inappropriate_Element 
		 (Current_Element : in Asis.Element; Puid : in String);

    procedure Inappropriate_Unit 
		 (Current_Unit : in Asis.Compilation_Unit; Puid : in String);

    procedure Unhandled_Case 
		 (Current_Element : in Asis.Element; Puid : in String);

--| +-------------------------------------------------------------------------+
--| | ASIS_FAILURE/1 (local)                                                  |
--| +-------------------------------------------------------------------------+

    procedure Asis_Failure 
		 (Current_Element : in Asis.Element; Puid : in String) is
    begin

	Msg_Log.Put_Msg_Debug 
	   ("exception Asis_Failed raised; current element is " & 
	    Asis_Debug_Support.Element_Image (Current_Element) & 
	    "; status is " & Asis_En.Error_Kinds'Image (Asis_En.Status) & 
	    "; diagnosis follows");
	Msg_Log.Put_Msg_Debug (Cuid, Puid, Diagnosis);

    end Asis_Failure;

--| +-------------------------------------------------------------------------+
--| | ASIS_FAILURE/2 (local)                                                  |
--| +-------------------------------------------------------------------------+

    procedure Asis_Failure 
		 (Current_Unit : in Asis.Compilation_Unit; Puid : in String) is
    begin

	Msg_Log.Put_Msg_Debug 
	   ("exception Asis_Failed raised; current unit is " & 
	    Asis_Debug_Support.Compilation_Unit_Image (Current_Unit) & 
	    "; status is " & Asis_En.Error_Kinds'Image (Asis_En.Status) & 
	    "; diagnosis follows");
	Msg_Log.Put_Msg_Debug (Cuid, Puid, Diagnosis);

    end Asis_Failure;

--| +-------------------------------------------------------------------------+
--| | DEFINES_A_DECLARATIVE_REGION (exported)                                 |
--| +-------------------------------------------------------------------------+

    function Defines_A_Declarative_Region 
		(The_Element : in Asis.Element) return Boolean is
	Puid : constant String := "Defines_A_Declarative_Region";

    begin

	if not Denotes_One_Or_More_Regions (The_Element, 
					    Expand_Instantiations => True) then
	    return False;
	end if;

	-- Given an element that is known to denote one or more region parts, 
	-- we determine whether the element *defines* a declarative region.  
	-- Certain element kinds, such as A_Package_Declaration, always define 
	-- a region; others, such as A_Procedure_Body_Declaration, do so only 
	-- if there is no preceding declaration of the same entity.

	case Asis_E.Element_Kind (The_Element) is
	    when Asis_E.A_Pragma =>
		return False;
	    when Asis_E.An_Argument_Association =>
		return False;
	    when Asis_E.A_Declaration =>
		case Asis_D.Kind (The_Element) is
		    when Asis_D.A_Variable_Declaration =>
			return False;
		    when Asis_D.A_Component_Declaration =>
			return False;
		    when Asis_D.A_Constant_Declaration =>
			return False;
		    when Asis_D.A_Deferred_Constant_Declaration =>
			return False;
		    when Asis_D.A_Generic_Formal_Object_Declaration =>
			return False;
		    when Asis_D.A_Discriminant_Specification =>
			return False;
		    when Asis_D.A_Parameter_Specification =>
			return False;
		    when Asis_D.An_Integer_Number_Declaration =>
			return False;
		    when Asis_D.A_Real_Number_Declaration =>
			return False;
		    when Asis_D.An_Exception_Declaration =>
			return False;
		    when Asis_D.An_Enumeration_Literal_Specification =>
			return False;
		    when Asis_D.A_Loop_Parameter_Specification =>
			return False;
		    when Asis_D.A_Full_Type_Declaration =>
			declare
			    Entity_Name : constant Asis.
						   Entity_Name_Definition := 
			       Asis_D.Names (The_Element) (1);
			    Name_Definition : Asis.Entity_Name_Definition := 
			       Asis_X.Name_Definition (Entity_Name);
			begin
			    return Asis_E.Is_Equal 
				      (Entity_Name, Name_Definition);
			end;
		    when Asis_D.An_Incomplete_Type_Declaration | 
			 Asis_D.A_Private_Type_Declaration =>
			return True;
		    when Asis_D.A_Subtype_Declaration =>
			return False;
		    when Asis_D.A_Package_Declaration =>
			return True;
		    when Asis_D.A_Package_Body_Declaration =>
			return False;
		    when Asis_D.A_Procedure_Declaration | 
			 Asis_D.A_Function_Declaration =>
			return True;
		    when Asis_D.A_Procedure_Body_Declaration | 
			 Asis_D.A_Function_Body_Declaration | 
			 Asis_D.A_Procedure_Body_Stub | 
			 Asis_D.A_Function_Body_Stub =>
			declare
			    Entity_Name : constant Asis.
						   Entity_Name_Definition := 
			       Asis_D.Names (The_Element) (1);
			    Name_Definition : Asis.Entity_Name_Definition := 
			       Asis_X.Name_Definition (Entity_Name);
			begin
			    return Asis_E.Is_Equal 
				      (Entity_Name, Name_Definition);
			end;
		    when Asis_D.An_Object_Rename_Declaration =>
			return False;
		    when Asis_D.An_Exception_Rename_Declaration =>
			return False;
		    when Asis_D.A_Package_Rename_Declaration =>
			return False;
		    when Asis_D.A_Procedure_Rename_Declaration | 
			 Asis_D.A_Function_Rename_Declaration =>
			return True;
		    when Asis_D.A_Generic_Package_Declaration =>
			return True;
		    when Asis_D.A_Generic_Procedure_Declaration =>
			return True;
		    when Asis_D.A_Generic_Function_Declaration =>
			return True;
		    when Asis_D.A_Package_Instantiation =>
			return True;
		    when Asis_D.A_Procedure_Instantiation =>
			return True;
		    when Asis_D.A_Function_Instantiation =>
			return True;
		    when Asis_D.A_Task_Declaration =>
			return True;
		    when Asis_D.A_Task_Body_Declaration =>
			return False;
		    when Asis_D.A_Task_Type_Declaration =>
			return True;
		    when Asis_D.An_Entry_Declaration =>
			return True;
		    when Asis_D.A_Package_Body_Stub =>
			return False;
		    when Asis_D.A_Task_Body_Stub =>
			return False;
		    when Asis_D.A_Generic_Formal_Type_Declaration =>
			return False;
		    when Asis_D.A_Generic_Formal_Private_Type_Declaration =>
			return True;
		    when Asis_D.A_Generic_Formal_Procedure_Declaration | 
			 Asis_D.A_Generic_Formal_Function_Declaration =>
			return True;
		    when Asis_D.Not_A_Declaration =>
			Unhandled_Case (The_Element, Puid);
		end case;
	    when Asis_E.An_Entity_Name_Definition =>
		return False;
	    when Asis_E.A_Type_Definition =>
		return False;
	    when Asis_E.A_Subtype_Indication =>
		return False;
	    when Asis_E.A_Constraint =>
		return False;
	    when Asis_E.A_Discrete_Range =>
		return False;
	    when Asis_E.A_Discriminant_Association =>
		return False;
	    when Asis_E.A_Variant_Part =>
		return False;
	    when Asis_E.A_Null_Component =>
		return False;
	    when Asis_E.A_Variant =>
		return False;
	    when Asis_E.A_Choice =>
		return False;
	    when Asis_E.A_Component_Association =>
		return False;
	    when Asis_E.An_Expression =>
		return False;
	    when Asis_E.A_Statement =>
		case Asis_S.Kind (The_Element) is
		    when Asis_S.A_Null_Statement =>
			return False;
		    when Asis_S.An_Assignment_Statement =>
			return False;
		    when Asis_S.A_Procedure_Call_Statement =>
			return False;
		    when Asis_S.An_Exit_Statement =>
			return False;
		    when Asis_S.A_Return_Statement =>
			return False;
		    when Asis_S.A_Goto_Statement =>
			return False;
		    when Asis_S.An_Entry_Call_Statement =>
			return False;
		    when Asis_S.A_Delay_Statement =>
			return False;
		    when Asis_S.An_Abort_Statement =>
			return False;
		    when Asis_S.A_Raise_Statement =>
			return False;
		    when Asis_S.A_Code_Statement =>
			return False;
		    when Asis_S.An_If_Statement =>
			return False;
		    when Asis_S.A_Case_Statement =>
			return False;
		    when Asis_S.A_Loop_Statement =>
			return True;
		    when Asis_S.A_Block_Statement =>
			return True;
		    when Asis_S.An_Accept_Statement =>
			return False;
		    when Asis_S.A_Selective_Wait_Statement =>
			return False;
		    when Asis_S.A_Conditional_Entry_Call_Statement =>
			return False;
		    when Asis_S.A_Timed_Entry_Call_Statement =>
			return False;
		    when Asis_S.Not_A_Statement =>
			Unhandled_Case (The_Element, Puid);
		end case;
	    when Asis_E.An_If_Statement_Arm =>
		return False;
	    when Asis_E.A_Case_Statement_Alternative =>
		return False;
	    when Asis_E.A_Parameter_Association =>
		return False;
	    when Asis_E.A_Use_Clause =>
		return False;
	    when Asis_E.A_Select_Statement_Arm =>
		return False;
	    when Asis_E.A_Select_Alternative =>
		return False;
	    when Asis_E.A_With_Clause =>
		return False;
	    when Asis_E.An_Exception_Handler =>
		return False;
	    when Asis_E.A_Representation_Clause =>
		return False;
	    when Asis_E.A_Component_Clause =>
		return False;
	    when Asis_E.Not_An_Element =>
		Unhandled_Case (The_Element, Puid);
	end case;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Inappropriate_Element (The_Element, Puid);
	    raise Region_Error;

	when Asis.Asis_Failed =>
	    Asis_Failure (The_Element, Puid);
	    raise Region_Error;

    end Defines_A_Declarative_Region;

--| +-------------------------------------------------------------------------+
--| | DENOTED_REGIONS (exported)                                              |
--| +-------------------------------------------------------------------------+

    function Denoted_Regions (The_Element : in Asis.Element; 
			      Expand_Instantiations : in Boolean := True; 
			      Include_Instance_Bodies : in Boolean := True) 
			     return Region_List is

	Puid : constant String := "Denoted_Regions";

	List : Region_List (1 .. 3);
	Nregions : Integer;

    begin

	if not Denotes_One_Or_More_Regions 
		  (The_Element, Expand_Instantiations) then
	    raise Region_Use_Error;
	end if;

	case Asis_E.Element_Kind (The_Element) is
	    when Asis_E.A_Declaration =>
		case Asis_D.Kind (The_Element) is
		    when Asis_D.A_Full_Type_Declaration =>
			List (1) := (A_Record_Type_Declaration, 
				     Element => The_Element);
			Nregions := 1;
		    when Asis_D.An_Incomplete_Type_Declaration =>
			List (1) := (An_Incomplete_Type_Declaration, 
				     Element => The_Element);
			Nregions := 1;
		    when Asis_D.A_Private_Type_Declaration =>
			List (1) := (A_Private_Type_Declaration, 
				     Element => The_Element);
			Nregions := 1;
		    when Asis_D.A_Package_Declaration =>
			List (1) := (A_Package_Specification_Visible_Part, 
				     Element => The_Element);
			Nregions := 1;
			if Asis_D.Is_Private_Present (The_Element) then
			    List (2) := (A_Package_Specification_Private_Part, 
					 Element => The_Element);
			    Nregions := 2;
			end if;
		    when Asis_D.A_Package_Body_Declaration =>
			List (1) := (A_Package_Body, Element => The_Element);
			Nregions := 1;
		    when Asis_D.A_Procedure_Declaration =>
			List (1) := (A_Procedure_Specification, 
				     Element => The_Element);
			Nregions := 1;
		    when Asis_D.A_Procedure_Body_Declaration =>
			List (1) := (A_Procedure_Body, Element => The_Element);
			Nregions := 1;
		    when Asis_D.A_Function_Declaration =>
			List (1) := (A_Function_Specification, 
				     Element => The_Element);
			Nregions := 1;
		    when Asis_D.A_Function_Body_Declaration =>
			List (1) := (A_Function_Body, Element => The_Element);
			Nregions := 1;
		    when Asis_D.A_Procedure_Rename_Declaration =>
			List (1) := (A_Procedure_Rename_Declaration, 
				     Element => The_Element);
			Nregions := 1;
		    when Asis_D.A_Function_Rename_Declaration =>
			List (1) := (A_Function_Rename_Declaration, 
				     Element => The_Element);
			Nregions := 1;
		    when Asis_D.A_Generic_Package_Declaration =>
			List (1) := (A_Generic_Package_Formal_Part, 
				     Element => The_Element);
			List (2) := 
			   (A_Generic_Package_Specification_Visible_Part, 
			    Element => The_Element);
			Nregions := 2;
			if Asis_D.Is_Private_Present (The_Element) then
			    List (3) := 
			       (A_Generic_Package_Specification_Private_Part, 
				Element => The_Element);
			    Nregions := 3;
			end if;
		    when Asis_D.A_Generic_Procedure_Declaration =>
			List (1) := (A_Generic_Procedure_Formal_Part, 
				     Element => The_Element);
			List (2) := (A_Generic_Procedure_Specification_Part, 
				     Element => The_Element);
			Nregions := 2;
		    when Asis_D.A_Generic_Function_Declaration =>
			List (1) := (A_Generic_Function_Formal_Part, 
				     Element => The_Element);
			List (2) := (A_Generic_Function_Specification_Part, 
				     Element => The_Element);
			Nregions := 2;
		    when Asis_D.A_Package_Instantiation =>
			declare
			    The_Package : Asis.Declaration := 
			       Asis_D.Corresponding_Specification (The_Element);
			    The_Package_Body : Asis.Declaration := 
			       Asis_D.Corresponding_Body (The_Element);
			begin
			    if Asis_E.Is_Nil (The_Package) then
				Nregions := 0;
			    else
				List (1) := 
				   (A_Package_Specification_Visible_Part, 
				    Element => The_Package);
				Nregions := 1;
				if Asis_D.Is_Private_Present (The_Package) then
				    List (2) := 
				       (A_Package_Specification_Private_Part, 
					Element => The_Package);
				    Nregions := 2;
				end if;
			    end if;
			    if Include_Instance_Bodies and then 
			       (not Asis_E.Is_Nil (The_Package_Body)) then
				Nregions := Nregions + 1;
				List (Nregions) := 
				   (A_Package_Body, 
				    Element => The_Package_Body);
			    end if;
			end;
		    when Asis_D.A_Procedure_Instantiation =>
			declare
			    The_Procedure : Asis.Declaration := 
			       Asis_D.Corresponding_Specification (The_Element);
			    The_Procedure_Body : Asis.Declaration := 
			       Asis_D.Corresponding_Body (The_Element);
			begin
			    if Asis_E.Is_Nil (The_Procedure) then
				Nregions := 0;
			    else
				List (1) := (A_Procedure_Specification, 
					     Element => The_Procedure);
				Nregions := 1;
			    end if;
			    if Include_Instance_Bodies and then 
			       (not Asis_E.Is_Nil (The_Procedure_Body)) then
				Nregions := Nregions + 1;
				List (Nregions) := 
				   (A_Procedure_Body, 
				    Element => The_Procedure_Body);
			    end if;
			end;
		    when Asis_D.A_Function_Instantiation =>
			declare
			    The_Function : Asis.Declaration := 
			       Asis_D.Corresponding_Specification (The_Element);
			    The_Function_Body : Asis.Declaration := 
			       Asis_D.Corresponding_Body (The_Element);
			begin
			    if Asis_E.Is_Nil (The_Function) then
				Nregions := 0;
			    else
				List (1) := (A_Function_Specification, 
					     Element => The_Function);
				Nregions := 1;
			    end if;
			    if Include_Instance_Bodies and then 
			       (not Asis_E.Is_Nil (The_Function_Body)) then
				Nregions := Nregions + 1;
				List (Nregions) := 
				   (A_Function_Body, 
				    Element => The_Function_Body);
			    end if;
			end;
		    when Asis_D.A_Task_Declaration =>
			List (1) := (A_Task_Declaration, 
				     Element => The_Element);
			Nregions := 1;
		    when Asis_D.A_Task_Body_Declaration =>
			List (1) := (A_Task_Body, Element => The_Element);
			Nregions := 1;
		    when Asis_D.A_Task_Type_Declaration =>
			List (1) := (A_Task_Type_Declaration, 
				     Element => The_Element);
			Nregions := 1;
		    when Asis_D.An_Entry_Declaration =>
			List (1) := (An_Entry_Declaration, 
				     Element => The_Element);
			Nregions := 1;
		    when Asis_D.A_Procedure_Body_Stub =>
			List (1) := (A_Procedure_Body_Stub, 
				     Element => The_Element);
			Nregions := 1;
		    when Asis_D.A_Function_Body_Stub =>
			List (1) := (A_Function_Body_Stub, 
				     Element => The_Element);
			Nregions := 1;
		    when Asis_D.A_Package_Body_Stub =>
			List (1) := (A_Package_Body_Stub, 
				     Element => The_Element);
			Nregions := 1;
		    when Asis_D.A_Task_Body_Stub =>
			List (1) := (A_Task_Body_Stub, Element => The_Element);
			Nregions := 1;
		    when Asis_D.A_Generic_Formal_Private_Type_Declaration =>
			List (1) := (A_Generic_Formal_Private_Type_Declaration, 
				     Element => The_Element);
			Nregions := 1;
		    when Asis_D.A_Generic_Formal_Procedure_Declaration =>
			List (1) := (A_Generic_Formal_Procedure_Specification, 
				     Element => The_Element);
			Nregions := 1;
		    when Asis_D.A_Generic_Formal_Function_Declaration =>
			List (1) := (A_Generic_Formal_Function_Specification, 
				     Element => The_Element);
			Nregions := 1;
		    when others =>
			Unhandled_Case (The_Element, Puid);
		end case;
	    when Asis_E.A_Statement =>
		case Asis_S.Kind (The_Element) is
		    when Asis_S.A_Loop_Statement =>
			case Asis_S.Loop_Kind (The_Element) is
			    when Asis_S.A_For_Loop =>
				List (1) := (A_For_Loop_Statement, 
					     Element => The_Element);
				Nregions := 1;
			    when Asis_S.A_While_Loop =>
				List (1) := (A_While_Loop_Statement, 
					     Element => The_Element);
				Nregions := 1;
			    when Asis_S.A_Simple_Loop =>
				List (1) := (A_Simple_Loop_Statement, 
					     Element => The_Element);
				Nregions := 1;
			    when Asis_S.Not_A_Loop =>
				Unhandled_Case (The_Element, Puid);
			end case;
		    when Asis_S.A_Block_Statement =>
			List (1) := (A_Block_Statement, Element => The_Element);
			Nregions := 1;
		    when Asis_S.An_Accept_Statement =>
			List (1) := (An_Accept_Statement, 
				     Element => The_Element);
			Nregions := 1;
		    when others =>
			Unhandled_Case (The_Element, Puid);
		end case;
	    when Asis_E.A_Representation_Clause =>
		case Asis_Rc.Kind (The_Element) is
		    when Asis_Rc.A_Record_Representation_Clause =>
			List (1) := (A_Record_Representation_Clause, 
				     Element => The_Element);
			Nregions := 1;
		    when others =>
			Unhandled_Case (The_Element, Puid);
		end case;
	    when others =>
		Unhandled_Case (The_Element, Puid);
	end case;

	return List (1 .. Nregions);

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Inappropriate_Element (The_Element, Puid);
	    raise Region_Error;

	when Asis.Asis_Failed =>
	    Asis_Failure (The_Element, Puid);
	    raise Region_Error;

    end Denoted_Regions;

--| +-------------------------------------------------------------------------+
--| | DENOTES_ONE_OR_MORE_REGIONS (exported)                                  |
--| +-------------------------------------------------------------------------+

    function Denotes_One_Or_More_Regions 
		(The_Element : in Asis.Element; 
		 Expand_Instantiations : in Boolean := True) return Boolean is

	Puid : constant String := "Denotes_One_Or_More_Regions";

    begin

	case Asis_E.Element_Kind (The_Element) is
	    when Asis_E.A_Pragma =>
		return False;
	    when Asis_E.An_Argument_Association =>
		return False;
	    when Asis_E.A_Declaration =>
		case Asis_D.Kind (The_Element) is
		    when Asis_D.A_Variable_Declaration =>
			return False;
		    when Asis_D.A_Component_Declaration =>
			return False;
		    when Asis_D.A_Constant_Declaration =>
			return False;
		    when Asis_D.A_Deferred_Constant_Declaration =>
			return False;
		    when Asis_D.A_Generic_Formal_Object_Declaration =>
			return False;
		    when Asis_D.A_Discriminant_Specification =>
			return False;
		    when Asis_D.A_Parameter_Specification =>
			return False;
		    when Asis_D.An_Integer_Number_Declaration =>
			return False;
		    when Asis_D.A_Real_Number_Declaration =>
			return False;
		    when Asis_D.An_Exception_Declaration =>
			return False;
		    when Asis_D.An_Enumeration_Literal_Specification =>
			return False;
		    when Asis_D.A_Loop_Parameter_Specification =>
			return False;
		    when Asis_D.A_Full_Type_Declaration =>
			declare
			    The_Type_Definition : Asis.Type_Definition := 
			       Asis_D.Type_Declaration_Definition (The_Element);
			begin
			    return Asis_Td."=" 
				      (Asis_Td.Kind (The_Type_Definition), 
				       Asis_Td.A_Record_Type_Definition);
			end;
		    when Asis_D.An_Incomplete_Type_Declaration | 
			 Asis_D.A_Private_Type_Declaration =>
			-- The corresponding full type declaration must have
			-- a record type definition.
			declare
			    The_Full_Type : Asis.Declaration := 
			       Asis_D.Corresponding_Type_Declaration 
				  (The_Element);
			    The_Type_Definition : Asis.Type_Definition;
			begin
			    if Asis.Elements.Is_Nil (The_Full_Type) then
				-- A nil value can occur in the case of
				-- an incomplete type whose full type
				-- is unknown, i.e., the full type resides
				-- in a different comp unit that has not
				-- been compiled.  If the incomplete
				-- type has a discriminant part, we know
				-- the full type must be record type.
				-- If there is no discriminant part,
				-- we just return false.
				declare
				    Discriminant_Part : 
				       constant Asis.
						Discriminant_Specification_List := 
				       Asis.Declarations.Discriminants 
					  (The_Element);
				begin
				    return not Asis.Elements.Is_Nil 
						  (Discriminant_Part);
				end;
			    else
				The_Type_Definition := 
				   Asis_D.Type_Declaration_Definition 
				      (The_Full_Type);
				return Asis_Td."=" 
					  (Asis_Td.Kind (The_Type_Definition), 
					   Asis_Td.A_Record_Type_Definition);
			    end if;
			end;
		    when Asis_D.A_Subtype_Declaration =>
			return False;
		    when Asis_D.A_Package_Declaration =>
			return True;
		    when Asis_D.A_Package_Body_Declaration =>
			return True;
		    when Asis_D.A_Procedure_Declaration =>
			return True;
		    when Asis_D.A_Procedure_Body_Declaration =>
			return True;
		    when Asis_D.A_Function_Declaration =>
			return True;
		    when Asis_D.A_Function_Body_Declaration =>
			return True;
		    when Asis_D.An_Object_Rename_Declaration =>
			return False;
		    when Asis_D.An_Exception_Rename_Declaration =>
			return False;
		    when Asis_D.A_Package_Rename_Declaration =>
			return False;
		    when Asis_D.A_Procedure_Rename_Declaration | 
			 Asis_D.A_Function_Rename_Declaration =>
			-- The renaming declaration must include a
			-- formal part.
			declare
			    Formal_Part : 
			       constant Asis.Parameter_Specification_List := 
			       Asis.Declarations.Parameters (The_Element);
			begin
			    return not Asis.Elements.Is_Nil (Formal_Part);
			end;
		    when Asis_D.A_Generic_Package_Declaration =>
			return True;
		    when Asis_D.A_Generic_Procedure_Declaration =>
			return True;
		    when Asis_D.A_Generic_Function_Declaration =>
			return True;
		    when Asis_D.A_Package_Instantiation =>
			return Expand_Instantiations;
		    when Asis_D.A_Procedure_Instantiation =>
			return Expand_Instantiations;
		    when Asis_D.A_Function_Instantiation =>
			return Expand_Instantiations;
		    when Asis_D.A_Task_Declaration =>
			return True;
		    when Asis_D.A_Task_Body_Declaration =>
			return True;
		    when Asis_D.A_Task_Type_Declaration =>
			return True;
		    when Asis_D.An_Entry_Declaration =>
			return True;
		    when Asis_D.A_Procedure_Body_Stub =>
			return True;
		    when Asis_D.A_Function_Body_Stub =>
			return True;
		    when Asis_D.A_Package_Body_Stub =>
			return True;
		    when Asis_D.A_Task_Body_Stub =>
			return True;
		    when Asis_D.A_Generic_Formal_Type_Declaration =>
			return False;
		    when Asis_D.A_Generic_Formal_Private_Type_Declaration =>
			-- The private type declaration must include a
			-- discriminant part.
			declare
			    Discriminant_Part : 
			       constant Asis.Discriminant_Specification_List := 
			       Asis.Declarations.Discriminants (The_Element);
			begin
			    return not Asis.Elements.Is_Nil (Discriminant_Part);
			end;
		    when Asis_D.A_Generic_Formal_Procedure_Declaration | 
			 Asis_D.A_Generic_Formal_Function_Declaration =>
			-- The renaming declaration must include a
			-- formal part.
			declare
			    Formal_Part : 
			       constant Asis.Parameter_Specification_List := 
			       Asis.Declarations.Parameters (The_Element);
			begin
			    return not Asis.Elements.Is_Nil (Formal_Part);
			end;
		    when Asis_D.Not_A_Declaration =>
			Unhandled_Case (The_Element, Puid);
		end case;
	    when Asis_E.An_Entity_Name_Definition =>
		return False;
	    when Asis_E.A_Type_Definition =>
		return False;
	    when Asis_E.A_Subtype_Indication =>
		return False;
	    when Asis_E.A_Constraint =>
		return False;
	    when Asis_E.A_Discrete_Range =>
		return False;
	    when Asis_E.A_Discriminant_Association =>
		return False;
	    when Asis_E.A_Variant_Part =>
		return False;
	    when Asis_E.A_Null_Component =>
		return False;
	    when Asis_E.A_Variant =>
		return False;
	    when Asis_E.A_Choice =>
		return False;
	    when Asis_E.A_Component_Association =>
		return False;
	    when Asis_E.An_Expression =>
		return False;
	    when Asis_E.A_Statement =>
		case Asis_S.Kind (The_Element) is
		    when Asis_S.A_Null_Statement =>
			return False;
		    when Asis_S.An_Assignment_Statement =>
			return False;
		    when Asis_S.A_Procedure_Call_Statement =>
			return False;
		    when Asis_S.An_Exit_Statement =>
			return False;
		    when Asis_S.A_Return_Statement =>
			return False;
		    when Asis_S.A_Goto_Statement =>
			return False;
		    when Asis_S.An_Entry_Call_Statement =>
			return False;
		    when Asis_S.A_Delay_Statement =>
			return False;
		    when Asis_S.An_Abort_Statement =>
			return False;
		    when Asis_S.A_Raise_Statement =>
			return False;
		    when Asis_S.A_Code_Statement =>
			return False;
		    when Asis_S.An_If_Statement =>
			return False;
		    when Asis_S.A_Case_Statement =>
			return False;
		    when Asis_S.A_Loop_Statement =>
			return True;
		    when Asis_S.A_Block_Statement =>
			-- A program unit body declaration has an 
			-- implicit block statement as its child.  Such
			-- block statements are not region parts.
			declare
			    Enclosing_Element : Asis.Element := 
			       Asis.Elements.Enclosing_Element (The_Element);
			begin
			    -- The statement is a region part if its
			    -- enclosing element is not a declaration.
			    return Asis_E."/=" (Asis_E.Element_Kind 
						   (Enclosing_Element), 
						Asis_E.A_Declaration);
			end;
		    when Asis_S.An_Accept_Statement =>
			return True;
		    when Asis_S.A_Selective_Wait_Statement =>
			return False;
		    when Asis_S.A_Conditional_Entry_Call_Statement =>
			return False;
		    when Asis_S.A_Timed_Entry_Call_Statement =>
			return False;
		    when Asis_S.Not_A_Statement =>
			Unhandled_Case (The_Element, Puid);
		end case;
	    when Asis_E.An_If_Statement_Arm =>
		return False;
	    when Asis_E.A_Case_Statement_Alternative =>
		return False;
	    when Asis_E.A_Parameter_Association =>
		return False;
	    when Asis_E.A_Use_Clause =>
		return False;
	    when Asis_E.A_Select_Statement_Arm =>
		return False;
	    when Asis_E.A_Select_Alternative =>
		return False;
	    when Asis_E.A_With_Clause =>
		return False;
	    when Asis_E.An_Exception_Handler =>
		return False;
	    when Asis_E.A_Representation_Clause =>
		case Asis_Rc.Kind (The_Element) is
		    when Asis_Rc.A_Length_Clause =>
			return False;
		    when Asis_Rc.An_Enumeration_Representation_Clause =>
			return False;
		    when Asis_Rc.A_Record_Representation_Clause =>
			return True;
		    when Asis_Rc.An_Address_Clause =>
			return False;
		    when Asis_Rc.Not_A_Representation_Clause =>
			Unhandled_Case (The_Element, Puid);
		end case;
	    when Asis_E.A_Component_Clause =>
		return False;
	    when Asis_E.Not_An_Element =>
		Unhandled_Case (The_Element, Puid);
	end case;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Inappropriate_Element (The_Element, Puid);
	    raise Region_Error;

	when Asis.Asis_Failed =>
	    Asis_Failure (The_Element, Puid);
	    raise Region_Error;

    end Denotes_One_Or_More_Regions;

--| +-------------------------------------------------------------------------+
--| | DIAGNOSIS (local)                                                       |
--| +-------------------------------------------------------------------------+

    function Diagnosis return String is

	Asis_Diagnosis : constant String := 
	   Asis_Str.To_Standard_String (Asis_En.Diagnosis);

    begin

	if Asis_Diagnosis = "" then
	    return "<no diagnosis available>";
	else
	    return Asis_Diagnosis;
	end if;

    end Diagnosis;

--| +-------------------------------------------------------------------------+
--| | EQUIVALENT_REGION (exported)                                            |
--| +-------------------------------------------------------------------------+

    function Equivalent_Region 
		(The_Unit : in Asis.Compilation_Unit) return Region is

	The_Region : Region;

    begin

	The_Region := (A_Compilation_Unit, Unit => The_Unit);
	return The_Region;

    end Equivalent_Region;

--| +-------------------------------------------------------------------------+
--| | HEAD_ELEMENT (exported)                                                 |
--| +-------------------------------------------------------------------------+

    function Head_Element (The_Region : in Region) return Asis.Element is
    begin

	if The_Region.Kind = A_Compilation_Unit then
	    raise Region_Use_Error;
	end if;

	return The_Region.Element;

    end Head_Element;

--| +-------------------------------------------------------------------------+
--| | HEAD_UNIT (exported)                                                    |
--| +-------------------------------------------------------------------------+

    function Head_Unit (The_Region : in Region) return Asis.Compilation_Unit is
    begin

	if The_Region.Kind /= A_Compilation_Unit then
	    raise Region_Use_Error;
	end if;

	return The_Region.Unit;

    end Head_Unit;

--| +-------------------------------------------------------------------------+
--| | INAPPROPRIATE_ELEMENT (local)                                           |
--| +-------------------------------------------------------------------------+

    procedure Inappropriate_Element 
		 (Current_Element : in Asis.Element; Puid : in String) is
    begin

	Msg_Log.Put_Msg_Debug 
	   ("exception Asis_Inappropriate_Element raised; current element is " & 
	    Asis_Debug_Support.Element_Image (Current_Element) & 
	    "; status is " & Asis_En.Error_Kinds'Image (Asis_En.Status) & 
	    "; diagnosis follows");
	Msg_Log.Put_Msg_Debug (Cuid, Puid, Diagnosis);

    end Inappropriate_Element;

--| +-------------------------------------------------------------------------+
--| | INAPPROPRIATE_UNIT (local)                                              |
--| +-------------------------------------------------------------------------+

    procedure Inappropriate_Unit 
		 (Current_Unit : in Asis.Compilation_Unit; Puid : in String) is
    begin

	Msg_Log.Put_Msg_Debug 
	   ("exception Asis_Inappropriate_Compilation_Unit raised; " & 
	    "current unit is " & 
	    Asis_Debug_Support.Compilation_Unit_Image (Current_Unit) & 
	    "; status is " & Asis_En.Error_Kinds'Image (Asis_En.Status) & 
	    "; diagnosis follows");
	Msg_Log.Put_Msg_Debug (Cuid, Puid, Diagnosis);

    end Inappropriate_Unit;

--| +-------------------------------------------------------------------------+
--| | KIND (exported)                                                         |
--| +-------------------------------------------------------------------------+

    function Kind (The_Region : in Region) return Region_Kinds is
    begin

	return The_Region.Kind;

    end Kind;

--| +-------------------------------------------------------------------------+
--| | NON_REGION_SUBELEMENTS (exported)                                       |
--| +-------------------------------------------------------------------------+

    function Non_Region_Subelements (The_Element : in Asis.Element;  
				     Expand_Instantiations : in Boolean := True; 
				     Include_Pragmas : in Boolean := True) 
				    return Asis.Element_List is

	Puid : constant String := "Non_Region_Subelements";

    begin

	if not Denotes_One_Or_More_Regions 
		  (The_Element, Expand_Instantiations) then
	    raise Region_Use_Error;
	end if;

	case Asis_E.Element_Kind (The_Element) is
	    when Asis_E.A_Declaration =>
		case Asis_D.Kind (The_Element) is
		    when Asis_D.A_Full_Type_Declaration =>
			return Asis_D.Names (The_Element);
		    when Asis_D.An_Incomplete_Type_Declaration =>
			return Asis_D.Names (The_Element);
		    when Asis_D.A_Private_Type_Declaration =>
			declare
			    Names : constant Asis.Entity_Name_Definition_List := 
			       Asis_D.Names (The_Element);
			    The_Type : Asis.Type_Definition := 
			       Asis_D.Type_Declaration_Definition (The_Element);
			begin
			    return Names & The_Type;
			end;
		    when Asis_D.A_Package_Declaration =>
			return Asis_D.Names (The_Element);
		    when Asis_D.A_Package_Body_Declaration =>
			return Asis_D.Names (The_Element);
		    when Asis_D.A_Procedure_Declaration =>
			return Asis_D.Names (The_Element);
		    when Asis_D.A_Procedure_Body_Declaration =>
			return Asis_D.Names (The_Element);
		    when Asis_D.A_Function_Declaration =>
			return Asis_D.Names (The_Element);
		    when Asis_D.A_Function_Body_Declaration =>
			return Asis_D.Names (The_Element);
		    when Asis_D.A_Procedure_Rename_Declaration =>
			declare
			    Names : constant Asis.Entity_Name_Definition_List := 
			       Asis_D.Names (The_Element);
			    Renamed_Procedure : Asis.Expression := 
			       Asis_D.Renamed_Entity (The_Element);
			begin
			    return Names & Renamed_Procedure;
			end;
		    when Asis_D.A_Function_Rename_Declaration =>
			declare
			    Names : constant Asis.Entity_Name_Definition_List := 
			       Asis_D.Names (The_Element);
			    Renamed_Function : Asis.Expression := 
			       Asis_D.Renamed_Entity (The_Element);
			begin
			    return Names & Renamed_Function;
			end;
		    when Asis_D.A_Generic_Package_Declaration =>
			return Asis_D.Names (The_Element);
		    when Asis_D.A_Generic_Procedure_Declaration =>
			return Asis_D.Names (The_Element);
		    when Asis_D.A_Generic_Function_Declaration =>
			return Asis_D.Names (The_Element);
		    when Asis_D.A_Package_Instantiation =>
			return Asis.Nil_Element_List;
		    when Asis_D.A_Procedure_Instantiation =>
			return Asis.Nil_Element_List;
		    when Asis_D.A_Function_Instantiation =>
			return Asis.Nil_Element_List;
		    when Asis_D.A_Task_Declaration =>
			return Asis_D.Names (The_Element);
		    when Asis_D.A_Task_Body_Declaration =>
			return Asis_D.Names (The_Element);
		    when Asis_D.A_Task_Type_Declaration =>
			return Asis_D.Names (The_Element);
		    when Asis_D.An_Entry_Declaration =>
			declare
			    Index : Asis.Discrete_Range := 
			       Asis_D.Family_Index (The_Element);
			    Names : constant Asis.Entity_Name_Definition_List := 
			       Asis_D.Names (The_Element);
			begin
			    if Asis_E.Is_Nil (Index) then
				return Names;
			    else
				return Names & Index;
			    end if;
			end;
		    when Asis_D.A_Procedure_Body_Stub =>
			return Asis_D.Names (The_Element);
		    when Asis_D.A_Function_Body_Stub =>
			return Asis_D.Names (The_Element);
		    when Asis_D.A_Package_Body_Stub =>
			return Asis_D.Names (The_Element);
		    when Asis_D.A_Task_Body_Stub =>
			return Asis_D.Names (The_Element);
		    when Asis_D.A_Generic_Formal_Private_Type_Declaration =>
			declare
			    Names : constant Asis.Entity_Name_Definition_List := 
			       Asis_D.Names (The_Element);
			    The_Type : Asis.Type_Definition := 
			       Asis_D.Type_Declaration_Definition (The_Element);
			begin
			    return Names & The_Type;
			end;
		    when Asis_D.A_Generic_Formal_Procedure_Declaration =>
			declare
			    Default_Kind : 
			       Asis_D.Generic_Formal_Subprogram_Default_Kinds := 
			       Asis_D.Generic_Formal_Subprogram_Default_Kind 
				  (The_Element);
			    Names : constant Asis.Entity_Name_Definition_List := 
			       Asis_D.Names (The_Element);
			begin
			    if Asis_D."=" (Default_Kind, Asis_D.A_Name) then
				return 
				   Names & 
				      Asis_D.Generic_Formal_Subprogram_Default 
					 (The_Element);
			    else
				return Names;
			    end if;
			end;
		    when Asis_D.A_Generic_Formal_Function_Declaration =>
			declare
			    Default_Kind : 
			       Asis_D.Generic_Formal_Subprogram_Default_Kinds := 
			       Asis_D.Generic_Formal_Subprogram_Default_Kind 
				  (The_Element);
			    Names : constant Asis.Entity_Name_Definition_List := 
			       Asis_D.Names (The_Element);
			begin
			    if Asis_D."=" (Default_Kind, Asis_D.A_Name) then
				return 
				   Names & 
				      Asis_D.Generic_Formal_Subprogram_Default 
					 (The_Element);
			    else
				return Names;
			    end if;
			end;
		    when others =>
			Unhandled_Case (The_Element, Puid);
		end case;
	    when Asis_E.A_Statement =>
		case Asis_S.Kind (The_Element) is
		    when Asis_S.A_Loop_Statement =>
			return Asis_S.Label_Names (The_Element);
		    when Asis_S.A_Block_Statement =>
			return Asis_S.Label_Names (The_Element);
		    when Asis_S.An_Accept_Statement =>
			declare
			    Entry_Name : Asis.Simple_Name := 
			       Asis_S.Accept_Entry_Simple_Name (The_Element);
			    Index : Asis.Expression := 
			       Asis_S.Family_Index (The_Element);
			    Labels : constant Asis.
					      Entity_Name_Definition_List := 
			       Asis_S.Label_Names (The_Element);
			begin
			    if Asis_E.Is_Nil (Index) then
				return Labels & Entry_Name;
			    else
				return Labels & Entry_Name & Index;
			    end if;
			end;
		    when others =>
			Unhandled_Case (The_Element, Puid);
		end case;
	    when Asis_E.A_Representation_Clause =>
		return Asis.Nil_Element_List & 
			  Asis_Rc.Record_Representation_Clause_Type_Simple_Name 
			     (The_Element);
	    when others =>
		Unhandled_Case (The_Element, Puid);
	end case;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Inappropriate_Element (The_Element, Puid);
	    raise Region_Error;

	when Asis.Asis_Failed =>
	    Asis_Failure (The_Element, Puid);
	    raise Region_Error;

    end Non_Region_Subelements;

--| +-------------------------------------------------------------------------+
--| | SUBELEMENTS (exported)                                                  |
--| +-------------------------------------------------------------------------+

    function Subelements (The_Region : in Region; 
			  Include_Pragmas : in Boolean := True) 
			 return Asis.Element_List is

	Puid : constant String := "Subelements";

    begin

	case The_Region.Kind is
	    when A_Compilation_Unit =>
		declare
		    Context_Clauses : constant Asis.Context_Clause_List := 
		       Asis_Cu.Context_Clause_Elements 
			  (The_Region.Unit, Include_Pragmas);
		    Declared_Unit : Asis.Declaration := 
		       Asis_Cu.Unit_Declaration (The_Region.Unit);
		begin
		    return Context_Clauses & Declared_Unit;
		end;
	    when A_Record_Type_Declaration =>
		declare
		    Discriminants : constant Asis.
					     Discriminant_Specification_List := 
		       Asis_D.Discriminants (The_Region.Element);
		    The_Type : Asis.Type_Definition := 
		       Asis_D.Type_Declaration_Definition (The_Region.Element);
		begin
		    return Discriminants & The_Type;
		end;
	    when An_Incomplete_Type_Declaration =>
		return Asis_D.Discriminants (The_Region.Element);
	    when A_Private_Type_Declaration =>
		return Asis_D.Discriminants (The_Region.Element);
	    when A_Package_Specification_Visible_Part =>
		return Asis_D.Visible_Part_Declarative_Items 
			  (The_Region.Element, Include_Pragmas);
	    when A_Package_Specification_Private_Part =>
		return Asis_D.Private_Part_Declarative_Items 
			  (The_Region.Element, Include_Pragmas);
	    when A_Package_Body =>
		return Asis.Nil_Element_List & 
			  Asis_D.Package_Body_Block (The_Region.Element);
	    when A_Procedure_Specification =>
		return Asis_D.Parameters (The_Region.Element);
	    when A_Procedure_Body =>
		declare
		    Parameters : constant Asis.Parameter_Specification_List := 
		       Asis_D.Parameters (The_Region.Element);
		    The_Body : Asis.Statement := 
		       Asis_D.Subprogram_Body_Block (The_Region.Element);
		begin
		    return Parameters & The_Body;
		end;
	    when A_Function_Specification =>
		declare
		    Parameters : constant Asis.Parameter_Specification_List := 
		       Asis_D.Parameters (The_Region.Element);
		    Return_Type : Asis.Expression := 
		       Asis_D.Return_Type (The_Region.Element);
		begin
		    return Parameters & Return_Type;
		end;
	    when A_Function_Body =>
		declare
		    Parameters : constant Asis.Parameter_Specification_List := 
		       Asis_D.Parameters (The_Region.Element);
		    Return_Type : Asis.Expression := 
		       Asis_D.Return_Type (The_Region.Element);
		    The_Body : Asis.Statement := 
		       Asis_D.Subprogram_Body_Block (The_Region.Element);
		begin
		    return Parameters & Return_Type & The_Body;
		end;
	    when A_Procedure_Rename_Declaration =>
		return Asis_D.Parameters (The_Region.Element);
	    when A_Function_Rename_Declaration =>
		declare
		    Parameters : constant Asis.Parameter_Specification_List := 
		       Asis_D.Parameters (The_Region.Element);
		    Return_Type : Asis.Expression := 
		       Asis_D.Return_Type (The_Region.Element);
		begin
		    return Parameters & Return_Type;
		end;
	    when A_Generic_Package_Formal_Part =>
		return Asis_D.Generic_Formal_Parameters 
			  (The_Region.Element, Include_Pragmas);
	    when A_Generic_Package_Specification_Visible_Part =>
		return Asis_D.Visible_Part_Declarative_Items 
			  (The_Region.Element, Include_Pragmas);
	    when A_Generic_Package_Specification_Private_Part =>
		return Asis_D.Private_Part_Declarative_Items 
			  (The_Region.Element, Include_Pragmas);
	    when A_Generic_Procedure_Formal_Part =>
		return Asis_D.Generic_Formal_Parameters 
			  (The_Region.Element, Include_Pragmas);
	    when A_Generic_Procedure_Specification_Part =>
		return Asis_D.Parameters (The_Region.Element);
	    when A_Generic_Function_Formal_Part =>
		return Asis_D.Generic_Formal_Parameters 
			  (The_Region.Element, Include_Pragmas);
	    when A_Generic_Function_Specification_Part =>
		declare
		    Parameters : constant Asis.Parameter_Specification_List := 
		       Asis_D.Parameters (The_Region.Element);
		    Return_Type : Asis.Expression := 
		       Asis_D.Return_Type (The_Region.Element);
		begin
		    return Parameters & Return_Type;
		end;
	    when A_Task_Declaration =>
		return Asis_D.Task_Declaration_Declarative_Items 
			  (The_Region.Element, Include_Pragmas);
	    when A_Task_Body =>
		return Asis.Nil_Element_List & 
			  Asis_D.Task_Body_Block (The_Region.Element);
	    when A_Task_Type_Declaration =>
		return Asis.Nil_Element_List & 
			  Asis_D.Type_Declaration_Definition 
			     (The_Region.Element);
	    when An_Entry_Declaration =>
		return Asis_D.Parameters (The_Region.Element);
	    when A_Procedure_Body_Stub =>
		return Asis_D.Parameters (The_Region.Element);
	    when A_Function_Body_Stub =>
		declare
		    Parameters : constant Asis.Parameter_Specification_List := 
		       Asis_D.Parameters (The_Region.Element);
		    Return_Type : Asis.Expression := 
		       Asis_D.Return_Type (The_Region.Element);
		begin
		    return Parameters & Return_Type;
		end;
	    when A_Package_Body_Stub =>
		return Asis.Nil_Element_List;
	    when A_Task_Body_Stub =>
		return Asis.Nil_Element_List;
	    when A_Generic_Formal_Private_Type_Declaration =>
		return Asis_D.Discriminants (The_Region.Element);
	    when A_Generic_Formal_Procedure_Specification =>
		return Asis_D.Parameters (The_Region.Element);
	    when A_Generic_Formal_Function_Specification =>
		declare
		    Parameters : constant Asis.Parameter_Specification_List := 
		       Asis_D.Parameters (The_Region.Element);
		    Return_Type : Asis.Expression := 
		       Asis_D.Return_Type (The_Region.Element);
		begin
		    return Parameters & Return_Type;
		end;
	    when A_For_Loop_Statement =>
		declare
		    Loop_Name : Asis.Entity_Name_Definition := 
		       Asis_S.Loop_Simple_Name (The_Region.Element);
		    Scheme : Asis.Declaration := 
		       Asis_S.For_Loop_Parameter_Specification 
			  (The_Region.Element);
		    Statements : constant Asis.Statement_List := 
		       Asis_S.Loop_Statements 
			  (The_Region.Element, Include_Pragmas);
		begin
		    if Asis_E.Is_Nil (Loop_Name) then
			return Scheme & Statements;
		    else
			return Loop_Name & Scheme & Statements;
		    end if;
		end;
	    when A_While_Loop_Statement =>
		declare
		    Condition : Asis.Expression := 
		       Asis_S.While_Condition (The_Region.Element);
		    Loop_Name : Asis.Entity_Name_Definition := 
		       Asis_S.Loop_Simple_Name (The_Region.Element);
		    Statements : constant Asis.Statement_List := 
		       Asis_S.Loop_Statements 
			  (The_Region.Element, Include_Pragmas);
		begin
		    if Asis_E.Is_Nil (Loop_Name) then
			return Condition & Statements;
		    else
			return Loop_Name & Condition & Statements;
		    end if;
		end;
	    when A_Simple_Loop_Statement =>
		declare
		    Loop_Name : Asis.Entity_Name_Definition := 
		       Asis_S.Loop_Simple_Name (The_Region.Element);
		    Statements : constant Asis.Statement_List := 
		       Asis_S.Loop_Statements 
			  (The_Region.Element, Include_Pragmas);
		begin
		    if Asis_E.Is_Nil (Loop_Name) then
			return Statements;
		    else
			return Loop_Name & Statements;
		    end if;
		end;
	    when A_Block_Statement =>
		declare
		    Block_Name : Asis.Entity_Name_Definition := 
		       Asis_S.Block_Simple_Name (The_Region.Element);
		    Declarations : constant Asis.Declarative_Item_List := 
		       Asis_S.Declarative_Items 
			  (The_Region.Element, Include_Pragmas);
		    Handlers : constant Asis.Exception_Handler_List := 
		       Asis_S.Block_Exception_Handlers (The_Region.Element);
		    Statements : constant Asis.Statement_List := 
		       Asis_S.Block_Body_Statements 
			  (The_Region.Element, Include_Pragmas);
		begin
		    if Asis_E.Is_Nil (Block_Name) then
			return Declarations & Statements & Handlers;
		    else
			return Block_Name & Declarations & 
				  Statements & Handlers;
		    end if;
		end;
	    when An_Accept_Statement =>
		declare
		    Parameters : constant Asis.Parameter_Specification_List := 
		       Asis_S.Accept_Parameters (The_Region.Element);
		    Statements : constant Asis.Statement_List := 
		       Asis_S.Accept_Body_Statements 
			  (The_Region.Element, Include_Pragmas);
		begin
		    return Parameters & Statements;
		end;
	    when A_Record_Representation_Clause =>
		declare
		    Alignment : Asis.Expression := 
		       Asis_Rc.
			  Record_Representation_Clause_Alignment_Clause_Expression 
			  (The_Region.Element);
		    Component_Clauses : constant Asis.Component_Clause_List := 
		       Asis_Rc.Component_Clauses 
			  (The_Region.Element, Include_Pragmas);
		    Pragmas : constant Asis.Element_List := 
		       Asis_Rc.
			  Record_Representation_Clause_Alignment_Clause_Pragmas 
			  (The_Region.Element);
		begin
		    if Asis_E.Is_Nil (Alignment) then
			if Include_Pragmas then
			    return Pragmas & Component_Clauses;
			else
			    return Component_Clauses;
			end if;
		    else
			if Include_Pragmas then
			    return Pragmas & Alignment & Component_Clauses;
			else
			    return Alignment & Component_Clauses;
			end if;
		    end if;
		end;
	end case;

    exception

	when Asis.Asis_Inappropriate_Compilation_Unit | 
	     Asis.Asis_Inappropriate_Element =>
	    if The_Region.Kind = A_Compilation_Unit then
		Inappropriate_Unit (The_Region.Unit, Puid);
	    else
		Inappropriate_Element (The_Region.Element, Puid);
	    end if;
	    raise Region_Error;

	when Asis.Asis_Failed =>
	    if The_Region.Kind = A_Compilation_Unit then
		Asis_Failure (The_Region.Unit, Puid);
	    else
		Asis_Failure (The_Region.Element, Puid);
	    end if;
	    raise Region_Error;

    end Subelements;

--| +-------------------------------------------------------------------------+
--| | UNHANDLED_CASE (local)                                                  |
--| +-------------------------------------------------------------------------+

    procedure Unhandled_Case 
		 (Current_Element : in Asis.Element; Puid : in String) is
    begin

	Msg_Log.Put_Msg_Debug 
	   (Cuid, Puid, "unhandled case; current element is " & 
			   Asis_Debug_Support.Element_Image (Current_Element));

	raise Region_Error;

    end Unhandled_Case;

end Region_Support;
