--| +=========================================================================+
--| |                                                                         |
--| | NAMESPACE_SCAN.SCAN_DECLARATION (body)                                  |
--| |                                                                         |
--| | Greg Janee                                                              |
--| | General Research Corporation                                            |
--| |                                                                         |
--| +=========================================================================+

separate (Namespace_Scan)
procedure Scan_Declaration (The_Declaration : in Asis.Declaration; 
			    The_Context : in Context) is

    Cuid : constant String := "Namespace_Scan.Scan_Declaration";
    Puid : constant String := "Scan_Declaration";

    package Error renames Namespace_Scan.Error_Handling_Support;
    package Trace renames Namespace_Scan.Trace_Support;

--| +-------------------------------------------------------------------------+
--| | SCAN_COMPONENT_DECLARATION (local)                                      |
--| +-------------------------------------------------------------------------+

    procedure Scan_Component_Declaration (The_Declaration : in Asis.Declaration; 
					  The_Context : in Context) is

	Puid : constant String := "Scan_Component_Declaration";

    begin

	declare
	    Default_Value : Asis.Expression := 
	       Asis_D.Initial_Value (The_Declaration);
	    Names : constant Asis.Entity_Name_Definition_List := 
	       Asis_D.Names (The_Declaration);
	    The_Type : Asis.Type_Definition := 
	       Asis_D.Object_Declaration_Definition (The_Declaration);
	begin

	    if Trace.On then
		Trace.Log (The_Declaration);
		Trace.Add_Level (Names'Length + 1 + 
				 Namespace_Scan.One_If_Present (Default_Value));
	    end if;

	    Namespace_Scan.Scan_Entity_Name_Definition_List 
	       (Names, The_Declaration);

	    Namespace_Scan.Scan_Type_Definition (The_Type, The_Context);

	    if not Asis_E.Is_Nil (Default_Value) then
		Namespace_Scan.Scan_Expression (Default_Value, The_Context);
	    end if;

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Declaration);
	    raise;

    end Scan_Component_Declaration;

--| +-------------------------------------------------------------------------+
--| | SCAN_CONSTANT_DECLARATION (local)                                       |
--| +-------------------------------------------------------------------------+

    procedure Scan_Constant_Declaration (The_Declaration : in Asis.Declaration; 
					 The_Context : in Context) is

	Puid : constant String := "Scan_Constant_Declaration";

    begin

	declare
	    Names : constant Asis.Entity_Name_Definition_List := 
	       Asis_D.Names (The_Declaration);
	    The_Type : Asis.Type_Definition := 
	       Asis_D.Object_Declaration_Definition (The_Declaration);
	    Value : Asis.Expression := Asis_D.Initial_Value (The_Declaration);
	begin

	    if Trace.On then
		Trace.Log (The_Declaration);
		Trace.Add_Level (Names'Length + 2);
	    end if;

	    Namespace_Scan.Scan_Entity_Name_Definition_List 
	       (Names, The_Declaration);

	    Namespace_Scan.Scan_Type_Definition (The_Type, The_Context);

	    Namespace_Scan.Scan_Expression (Value, The_Context);

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Declaration);
	    raise;

    end Scan_Constant_Declaration;

--| +-------------------------------------------------------------------------+
--| | SCAN_DEFERRED_CONSTANT_DECLARATION (local)                              |
--| +-------------------------------------------------------------------------+

    procedure Scan_Deferred_Constant_Declaration 
		 (The_Declaration : in Asis.Declaration; 
		  The_Context : in Context) is

	Puid : constant String := "Scan_Deferred_Constant_Declaration";

    begin

	declare
	    Names : constant Asis.Entity_Name_Definition_List := 
	       Asis_D.Names (The_Declaration);
	    Type_Mark : Asis.Expression := Asis_D.Type_Mark (The_Declaration);
	begin

	    if Trace.On then
		Trace.Log (The_Declaration);
		Trace.Add_Level (Names'Length + 1);
	    end if;

	    Namespace_Scan.Scan_Entity_Name_Definition_List 
	       (Names, The_Declaration);

	    Namespace_Scan.Scan_Expression (Type_Mark, The_Context);

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Declaration);
	    raise;

    end Scan_Deferred_Constant_Declaration;

--| +-------------------------------------------------------------------------+
--| | SCAN_DISCRIMINANT_SPECIFICATION (local)                                 |
--| +-------------------------------------------------------------------------+

    procedure Scan_Discriminant_Specification 
		 (The_Declaration : in Asis.Declaration; 
		  The_Context : in Context) is

	Puid : constant String := "Scan_Discriminant_Specification";

    begin

	declare
	    Default_Value : Asis.Expression := 
	       Asis_D.Initial_Value (The_Declaration);
	    Names : constant Asis.Entity_Name_Definition_List := 
	       Asis_D.Names (The_Declaration);
	    The_Type : Asis.Expression := Asis_D.Type_Mark (The_Declaration);
	begin

	    if Trace.On then
		Trace.Log (The_Declaration);
		Trace.Add_Level (Names'Length + 1 + 
				 Namespace_Scan.One_If_Present (Default_Value));
	    end if;

	    Namespace_Scan.Scan_Entity_Name_Definition_List 
	       (Names, The_Declaration);

	    Namespace_Scan.Scan_Expression (The_Type, The_Context);

	    if not Asis_E.Is_Nil (Default_Value) then
		Namespace_Scan.Scan_Expression (Default_Value, The_Context);
	    end if;

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Declaration);
	    raise;

    end Scan_Discriminant_Specification;

--| +-------------------------------------------------------------------------+
--| | SCAN_ENTRY_DECLARATION (local)                                          |
--| +-------------------------------------------------------------------------+

    procedure Scan_Entry_Declaration (The_Declaration : in Asis.Declaration; 
				      The_Context : in Context) is

	Puid : constant String := "Scan_Entry_Declaration";

    begin

	declare
	    Index : Asis.Discrete_Range := 
	       Asis_D.Family_Index (The_Declaration);
	    Names : constant Asis.Entity_Name_Definition_List := 
	       Asis_D.Names (The_Declaration);
	    Parameters : constant Asis.Parameter_Specification_List := 
	       Asis_D.Parameters (The_Declaration);
	begin

	    if Trace.On then
		Trace.Log (The_Declaration);
		Trace.Add_Level (Names'Length + 
				 Namespace_Scan.One_If_Present (Index) + 
				 Parameters'Length);
	    end if;

	    Namespace_Scan.Scan_Entity_Name_Definition_List 
	       (Names, The_Declaration);

	    if not Asis_E.Is_Nil (Index) then
		Namespace_Scan.Scan_Discrete_Range (Index, The_Context);
	    end if;

	    Namespace_Scan.Scan_Declaration_List (Parameters, The_Context);

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Declaration);
	    raise;

    end Scan_Entry_Declaration;

--| +-------------------------------------------------------------------------+
--| | SCAN_ENUMERATION_LITERAL_SPECIFICATION (local)                          |
--| +-------------------------------------------------------------------------+

    procedure Scan_Enumeration_Literal_Specification 
		 (The_Declaration : in Asis.Declaration; 
		  The_Context : in Context) is

	Puid : constant String := "Scan_Enumeration_Literal_Specification";

    begin

	declare
	    Names : constant Asis.Entity_Name_Definition_List := 
	       Asis_D.Names (The_Declaration);
	begin

	    if Trace.On then
		Trace.Log (The_Declaration);
		Trace.Add_Level (Names'Length);
	    end if;

	    Namespace_Scan.Scan_Entity_Name_Definition_List 
	       (Names, The_Declaration);

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Declaration);
	    raise;

    end Scan_Enumeration_Literal_Specification;

--| +-------------------------------------------------------------------------+
--| | SCAN_EXCEPTION_DECLARATION (local)                                      |
--| +-------------------------------------------------------------------------+

    procedure Scan_Exception_Declaration (The_Declaration : in Asis.Declaration; 
					  The_Context : in Context) is

	Puid : constant String := "Scan_Exception_Declaration";

    begin

	declare
	    Names : constant Asis.Entity_Name_Definition_List := 
	       Asis_D.Names (The_Declaration);
	begin

	    if Trace.On then
		Trace.Log (The_Declaration);
		Trace.Add_Level (Names'Length);
	    end if;

	    Namespace_Scan.Scan_Entity_Name_Definition_List 
	       (Names, The_Declaration);

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Declaration);
	    raise;

    end Scan_Exception_Declaration;

--| +-------------------------------------------------------------------------+
--| | SCAN_EXCEPTION_RENAME_DECLARATION (local)                               |
--| +-------------------------------------------------------------------------+

    procedure Scan_Exception_Rename_Declaration 
		 (The_Declaration : in Asis.Declaration; 
		  The_Context : in Context) is

	Puid : constant String := "Scan_Exception_Rename_Declaration";

    begin

	declare
	    Names : constant Asis.Entity_Name_Definition_List := 
	       Asis_D.Names (The_Declaration);
	    Renamed_Exception : Asis.Expression := 
	       Asis_D.Renamed_Entity (The_Declaration);
	begin

	    if Trace.On then
		Trace.Log (The_Declaration);
		Trace.Add_Level (Names'Length + 1);
	    end if;

	    Namespace_Scan.Scan_Entity_Name_Definition_List 
	       (Names, The_Declaration);

	    Namespace_Scan.Scan_Expression (Renamed_Exception, The_Context);

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Declaration);
	    raise;

    end Scan_Exception_Rename_Declaration;

--| +-------------------------------------------------------------------------+
--| | SCAN_FULL_TYPE_DECLARATION (local)                                      |
--| +-------------------------------------------------------------------------+

    procedure Scan_Full_Type_Declaration (The_Declaration : in Asis.Declaration; 
					  The_Context : in Context) is

	Puid : constant String := "Scan_Full_Type_Declaration";

    begin

	declare
	    Discriminants : constant Asis.Discriminant_Specification_List := 
	       Asis_D.Discriminants (The_Declaration);
	    Names : constant Asis.Entity_Name_Definition_List := 
	       Asis_D.Names (The_Declaration);
	    The_Type : Asis.Type_Definition := 
	       Asis_D.Type_Declaration_Definition (The_Declaration);
	begin

	    if Trace.On then
		Trace.Log (The_Declaration);
		Trace.Add_Level (Names'Length + Discriminants'Length + 1);
	    end if;

	    Namespace_Scan.Scan_Entity_Name_Definition_List 
	       (Names, The_Declaration);

	    Namespace_Scan.Scan_Declaration_List (Discriminants, The_Context);

	    Namespace_Scan.Scan_Type_Definition (The_Type, The_Context);

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Declaration);
	    raise;

    end Scan_Full_Type_Declaration;

--| +-------------------------------------------------------------------------+
--| | SCAN_FUNCTION_BODY_DECLARATION (local)                                  |
--| +-------------------------------------------------------------------------+

    procedure Scan_Function_Body_Declaration 
		 (The_Declaration : in Asis.Declaration; 
		  The_Context : in Context) is

	Puid : constant String := "Scan_Function_Body_Declaration";

    begin

	declare
	    Names : constant Asis.Entity_Name_Definition_List := 
	       Asis_D.Names (The_Declaration);
	    Parameters : constant Asis.Parameter_Specification_List := 
	       Asis_D.Parameters (The_Declaration);
	    Return_Type : Asis.Expression := 
	       Asis_D.Return_Type (The_Declaration);
	    The_Body : Asis.Statement := 
	       Asis_D.Subprogram_Body_Block (The_Declaration);
	begin

	    if Trace.On then
		Trace.Log (The_Declaration);
		Trace.Add_Level (Names'Length + Parameters'Length + 2);
	    end if;

	    Namespace_Scan.Scan_Entity_Name_Definition_List 
	       (Names, The_Declaration);

	    Namespace_Scan.Scan_Declaration_List (Parameters, The_Context);

	    Namespace_Scan.Scan_Expression (Return_Type, The_Context);

	    Namespace_Scan.Scan_Statement (The_Body, The_Context);

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Declaration);
	    raise;

    end Scan_Function_Body_Declaration;

--| +-------------------------------------------------------------------------+
--| | SCAN_FUNCTION_BODY_STUB (local)                                         |
--| +-------------------------------------------------------------------------+

    procedure Scan_Function_Body_Stub (The_Declaration : in Asis.Declaration; 
				       The_Context : in Context) is

	Puid : constant String := "Scan_Function_Body_Stub";

    begin

	declare
	    Names : constant Asis.Entity_Name_Definition_List := 
	       Asis_D.Names (The_Declaration);
	    Parameters : constant Asis.Parameter_Specification_List := 
	       Asis_D.Parameters (The_Declaration);
	    Return_Type : Asis.Expression := 
	       Asis_D.Return_Type (The_Declaration);
	begin

	    if Trace.On then
		Trace.Log (The_Declaration);
		Trace.Add_Level (Names'Length + Parameters'Length + 1);
	    end if;

	    Namespace_Scan.Scan_Entity_Name_Definition_List 
	       (Names, The_Declaration);

	    Namespace_Scan.Scan_Declaration_List (Parameters, The_Context);

	    Namespace_Scan.Scan_Expression (Return_Type, The_Context);

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Declaration);
	    raise;

    end Scan_Function_Body_Stub;

--| +-------------------------------------------------------------------------+
--| | SCAN_FUNCTION_DECLARATION (local)                                       |
--| +-------------------------------------------------------------------------+

    procedure Scan_Function_Declaration (The_Declaration : in Asis.Declaration; 
					 The_Context : in Context) is

	Puid : constant String := "Scan_Function_Declaration";

    begin

	declare
	    Names : constant Asis.Entity_Name_Definition_List := 
	       Asis_D.Names (The_Declaration);
	    Parameters : constant Asis.Parameter_Specification_List := 
	       Asis_D.Parameters (The_Declaration);
	    Return_Type : Asis.Expression := 
	       Asis_D.Return_Type (The_Declaration);
	begin

	    if Trace.On then
		Trace.Log (The_Declaration);
		Trace.Add_Level (Names'Length + Parameters'Length + 1);
	    end if;

	    Namespace_Scan.Scan_Entity_Name_Definition_List 
	       (Names, The_Declaration);

	    Namespace_Scan.Scan_Declaration_List (Parameters, The_Context);

	    Namespace_Scan.Scan_Expression (Return_Type, The_Context);

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Declaration);
	    raise;

    end Scan_Function_Declaration;

--| +-------------------------------------------------------------------------+
--| | SCAN_FUNCTION_INSTANTIATION (local)                                     |
--| +-------------------------------------------------------------------------+

    procedure Scan_Function_Instantiation 
		 (The_Declaration : in Asis.Declaration; 
		  The_Context : in Context) is

	Puid : constant String := "Scan_Function_Instantiation";

    begin

	if Namespace_Scan.Expand_Function_Instantiations then

	    declare
		Specification : Asis.Declaration := 
		   Asis_D.Corresponding_Specification (The_Declaration);
		The_Body : Asis.Declaration := 
		   Asis_D.Corresponding_Body (The_Declaration);
	    begin
		if Trace.On then
		    Trace.Log (The_Declaration);
		    Trace.Add_Level (Namespace_Scan.One_If_Present 
					(Specification) + 
				     Namespace_Scan.One_If_Present (The_Body));
		end if;
		if not Asis_E.Is_Nil (Specification) then
		    Namespace_Scan.Scan_Declaration 
		       (Specification, The_Context);
		end if;
		if not Asis_E.Is_Nil (The_Body) then
		    Namespace_Scan.Scan_Declaration (The_Body, The_Context);
		end if;
	    end;

	else

	    declare
		Arguments : constant Asis.Association_List := 
		   Asis_D.Generic_Parameters 
		      (The_Declaration, Namespace_Scan.
					   Normalize_Instantiation_Parameters);
		Names : constant Asis.Entity_Name_Definition_List := 
		   Asis_D.Names (The_Declaration);
		The_Function : Asis.Expression := 
		   Asis_D.Generic_Unit_Name (The_Declaration);
	    begin
		if Trace.On then
		    Trace.Log (The_Declaration);
		    Trace.Add_Level (Names'Length + 1 + Arguments'Length);
		end if;
		Namespace_Scan.Scan_Entity_Name_Definition_List 
		   (Names, The_Declaration);
		Namespace_Scan.Scan_Expression (The_Function, The_Context);
		Namespace_Scan.Scan_Parameter_Association_List 
		   (Arguments, The_Context);
	    end;

	end if;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Declaration);
	    raise;

    end Scan_Function_Instantiation;

--| +-------------------------------------------------------------------------+
--| | SCAN_FUNCTION_RENAME_DECLARATION (local)                                |
--| +-------------------------------------------------------------------------+

    procedure Scan_Function_Rename_Declaration 
		 (The_Declaration : in Asis.Declaration; 
		  The_Context : in Context) is

	Puid : constant String := "Scan_Function_Rename_Declaration";

    begin

	declare
	    Names : constant Asis.Entity_Name_Definition_List := 
	       Asis_D.Names (The_Declaration);
	    Parameters : constant Asis.Parameter_Specification_List := 
	       Asis_D.Parameters (The_Declaration);
	    Renamed_Function : Asis.Expression := 
	       Asis_D.Renamed_Entity (The_Declaration);
	    Return_Type : Asis.Expression := 
	       Asis_D.Return_Type (The_Declaration);
	begin

	    if Trace.On then
		Trace.Log (The_Declaration);
		Trace.Add_Level (Names'Length + Parameters'Length + 2);
	    end if;

	    Namespace_Scan.Scan_Entity_Name_Definition_List 
	       (Names, The_Declaration);

	    Namespace_Scan.Scan_Declaration_List (Parameters, The_Context);

	    Namespace_Scan.Scan_Expression (Return_Type, The_Context);

	    Namespace_Scan.Scan_Expression (Renamed_Function, The_Context);

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Declaration);
	    raise;

    end Scan_Function_Rename_Declaration;

--| +-------------------------------------------------------------------------+
--| | SCAN_GENERIC_FORMAL_FUNCTION_DECLARATION (local)                        |
--| +-------------------------------------------------------------------------+

    procedure Scan_Generic_Formal_Function_Declaration 
		 (The_Declaration : in Asis.Declaration; 
		  The_Context : in Context) is

	Puid : constant String := "Scan_Generic_Formal_Function_Declaration";

    begin

	declare
	    Default_Kind : Asis_D.Generic_Formal_Subprogram_Default_Kinds := 
	       Asis_D.Generic_Formal_Subprogram_Default_Kind (The_Declaration);
	    Names : constant Asis.Entity_Name_Definition_List := 
	       Asis_D.Names (The_Declaration);
	    Parameters : constant Asis.Parameter_Specification_List := 
	       Asis_D.Parameters (The_Declaration);
	    Return_Type : Asis.Expression := 
	       Asis_D.Return_Type (The_Declaration);
	begin

	    if Trace.On then
		Trace.Log (The_Declaration);
		Trace.Add_Level (Names'Length + Parameters'Length + 1);
		if Asis_D."=" (Default_Kind, Asis_D.A_Name) then
		    Trace.Add_Children (1);
		end if;
	    end if;

	    Namespace_Scan.Scan_Entity_Name_Definition_List 
	       (Names, The_Declaration);

	    Namespace_Scan.Scan_Declaration_List (Parameters, The_Context);

	    Namespace_Scan.Scan_Expression (Return_Type, The_Context);

	    if Asis_D."=" (Default_Kind, Asis_D.A_Name) then
		Namespace_Scan.Scan_Expression 
		   (Asis_D.Generic_Formal_Subprogram_Default (The_Declaration), 
		    The_Context);
	    end if;

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Declaration);
	    raise;

    end Scan_Generic_Formal_Function_Declaration;

--| +-------------------------------------------------------------------------+
--| | SCAN_GENERIC_FORMAL_OBJECT_DECLARATION (local)                          |
--| +-------------------------------------------------------------------------+

    procedure Scan_Generic_Formal_Object_Declaration 
		 (The_Declaration : in Asis.Declaration; 
		  The_Context : in Context) is

	Puid : constant String := "Scan_Generic_Formal_Object_Declaration";

    begin

	declare
	    Default_Value : Asis.Expression := 
	       Asis_D.Initial_Value (The_Declaration);
	    Names : constant Asis.Entity_Name_Definition_List := 
	       Asis_D.Names (The_Declaration);
	    The_Type : Asis.Expression := Asis_D.Type_Mark (The_Declaration);
	begin

	    if Trace.On then
		Trace.Log (The_Declaration);
		Trace.Add_Level (Names'Length + 1 + 
				 Namespace_Scan.One_If_Present (Default_Value));
	    end if;

	    Namespace_Scan.Scan_Entity_Name_Definition_List 
	       (Names, The_Declaration);

	    Namespace_Scan.Scan_Expression (The_Type, The_Context);

	    if not Asis_E.Is_Nil (Default_Value) then
		Namespace_Scan.Scan_Expression (Default_Value, The_Context);
	    end if;

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Declaration);
	    raise;

    end Scan_Generic_Formal_Object_Declaration;

--| +-------------------------------------------------------------------------+
--| | SCAN_GENERIC_FORMAL_PRIVATE_TYPE_DECLARATION (local)                    |
--| +-------------------------------------------------------------------------+

    procedure Scan_Generic_Formal_Private_Type_Declaration 
		 (The_Declaration : in Asis.Declaration; 
		  The_Context : in Context) is

	Puid : constant String := 
	   "Scan_Generic_Formal_Private_Type_Declaration";

    begin

	declare
	    Discriminants : constant Asis.Discriminant_Specification_List := 
	       Asis_D.Discriminants (The_Declaration);
	    Names : constant Asis.Entity_Name_Definition_List := 
	       Asis_D.Names (The_Declaration);
	    The_Type : Asis.Type_Definition := 
	       Asis_D.Type_Declaration_Definition (The_Declaration);
	begin

	    if Trace.On then
		Trace.Log (The_Declaration);
		Trace.Add_Level (Names'Length + Discriminants'Length + 1);
	    end if;

	    Namespace_Scan.Scan_Entity_Name_Definition_List 
	       (Names, The_Declaration);

	    Namespace_Scan.Scan_Declaration_List (Discriminants, The_Context);

	    Namespace_Scan.Scan_Type_Definition (The_Type, The_Context);

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Declaration);
	    raise;

    end Scan_Generic_Formal_Private_Type_Declaration;

--| +-------------------------------------------------------------------------+
--| | SCAN_GENERIC_FORMAL_PROCEDURE_DECLARATION (local)                       |
--| +-------------------------------------------------------------------------+

    procedure Scan_Generic_Formal_Procedure_Declaration 
		 (The_Declaration : in Asis.Declaration; 
		  The_Context : in Context) is

	Puid : constant String := "Scan_Generic_Formal_Procedure_Declaration";

    begin

	declare
	    Default_Kind : Asis_D.Generic_Formal_Subprogram_Default_Kinds := 
	       Asis_D.Generic_Formal_Subprogram_Default_Kind (The_Declaration);
	    Names : constant Asis.Entity_Name_Definition_List := 
	       Asis_D.Names (The_Declaration);
	    Parameters : constant Asis.Parameter_Specification_List := 
	       Asis_D.Parameters (The_Declaration);
	begin

	    if Trace.On then
		Trace.Log (The_Declaration);
		Trace.Add_Level (Names'Length + Parameters'Length);
		if Asis_D."=" (Default_Kind, Asis_D.A_Name) then
		    Trace.Add_Children (1);
		end if;
	    end if;

	    Namespace_Scan.Scan_Entity_Name_Definition_List 
	       (Names, The_Declaration);

	    Namespace_Scan.Scan_Declaration_List (Parameters, The_Context);

	    if Asis_D."=" (Default_Kind, Asis_D.A_Name) then
		Namespace_Scan.Scan_Expression 
		   (Asis_D.Generic_Formal_Subprogram_Default (The_Declaration), 
		    The_Context);
	    end if;

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Declaration);
	    raise;

    end Scan_Generic_Formal_Procedure_Declaration;

--| +-------------------------------------------------------------------------+
--| | SCAN_GENERIC_FORMAL_TYPE_DECLARATION (local)                            |
--| +-------------------------------------------------------------------------+

    procedure Scan_Generic_Formal_Type_Declaration 
		 (The_Declaration : in Asis.Declaration; 
		  The_Context : in Context) is

	Puid : constant String := "Scan_Generic_Formal_Type_Declaration";

    begin

	declare
	    Names : constant Asis.Entity_Name_Definition_List := 
	       Asis_D.Names (The_Declaration);
	    The_Type : Asis.Type_Definition := 
	       Asis_D.Type_Declaration_Definition (The_Declaration);
	begin

	    if Trace.On then
		Trace.Log (The_Declaration);
		Trace.Add_Level (Names'Length + 1);
	    end if;

	    Namespace_Scan.Scan_Entity_Name_Definition_List 
	       (Names, The_Declaration);

	    Namespace_Scan.Scan_Type_Definition (The_Type, The_Context);

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Declaration);
	    raise;

    end Scan_Generic_Formal_Type_Declaration;

--| +-------------------------------------------------------------------------+
--| | SCAN_GENERIC_FUNCTION_DECLARATION (local)                               |
--| +-------------------------------------------------------------------------+

    procedure Scan_Generic_Function_Declaration 
		 (The_Declaration : in Asis.Declaration; 
		  The_Context : in Context) is

	Puid : constant String := "Scan_Generic_Function_Declaration";

    begin

	declare
	    Formal_Parameters : constant Asis.Generic_Formal_Parameter_List := 
	       Asis_D.Generic_Formal_Parameters 
		  (The_Declaration, Namespace_Scan.
				       Include_Generic_Formal_Part_Pragmas);
	    Names : constant Asis.Entity_Name_Definition_List := 
	       Asis_D.Names (The_Declaration);
	    Parameters : constant Asis.Parameter_Specification_List := 
	       Asis_D.Parameters (The_Declaration);
	    Return_Type : Asis.Expression := 
	       Asis_D.Return_Type (The_Declaration);
	begin

	    if Trace.On then
		Trace.Log (The_Declaration);
		Trace.Add_Level (Formal_Parameters'Length + 
				 Names'Length + Parameters'Length + 1);
	    end if;

	    Namespace_Scan.Scan_Declarative_Item_List 
	       (Formal_Parameters, The_Context);

	    Namespace_Scan.Scan_Entity_Name_Definition_List 
	       (Names, The_Declaration);

	    Namespace_Scan.Scan_Declaration_List (Parameters, The_Context);

	    Namespace_Scan.Scan_Expression (Return_Type, The_Context);

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Declaration);
	    raise;

    end Scan_Generic_Function_Declaration;

--| +-------------------------------------------------------------------------+
--| | SCAN_GENERIC_PACKAGE_DECLARATION (local)                                |
--| +-------------------------------------------------------------------------+

    procedure Scan_Generic_Package_Declaration 
		 (The_Declaration : in Asis.Declaration; 
		  The_Context : in Context) is

	Puid : constant String := "Scan_Generic_Package_Declaration";

    begin

	declare
	    Formal_Parameters : constant Asis.Generic_Formal_Parameter_List := 
	       Asis_D.Generic_Formal_Parameters 
		  (The_Declaration, Namespace_Scan.
				       Include_Generic_Formal_Part_Pragmas);
	    Names : constant Asis.Entity_Name_Definition_List := 
	       Asis_D.Names (The_Declaration);
	    Private_Declarations : constant Asis.Declarative_Item_List := 
	       Asis_D.Private_Part_Declarative_Items 
		  (The_Declaration, Namespace_Scan.Include_Package_Pragmas);
	    Public_Declarations : constant Asis.Declarative_Item_List := 
	       Asis_D.Visible_Part_Declarative_Items 
		  (The_Declaration, Namespace_Scan.Include_Package_Pragmas);
	begin

	    if Trace.On then
		Trace.Log (The_Declaration);
		Trace.Add_Level (Formal_Parameters'Length + Names'Length + 
				 Public_Declarations'Length + 
				 Private_Declarations'Length);
	    end if;

	    Namespace_Scan.Scan_Declarative_Item_List 
	       (Formal_Parameters, The_Context);

	    Namespace_Scan.Scan_Entity_Name_Definition_List 
	       (Names, The_Declaration);

	    Namespace_Scan.Scan_Declarative_Item_List 
	       (Public_Declarations, The_Context);

	    Namespace_Scan.Scan_Declarative_Item_List 
	       (Private_Declarations, The_Context);

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Declaration);
	    raise;

    end Scan_Generic_Package_Declaration;

--| +-------------------------------------------------------------------------+
--| | SCAN_GENERIC_PROCEDURE_DECLARATION (local)                              |
--| +-------------------------------------------------------------------------+

    procedure Scan_Generic_Procedure_Declaration 
		 (The_Declaration : in Asis.Declaration; 
		  The_Context : in Context) is

	Puid : constant String := "Scan_Generic_Procedure_Declaration";

    begin

	declare
	    Formal_Parameters : constant Asis.Generic_Formal_Parameter_List := 
	       Asis_D.Generic_Formal_Parameters 
		  (The_Declaration, Namespace_Scan.
				       Include_Generic_Formal_Part_Pragmas);
	    Names : constant Asis.Entity_Name_Definition_List := 
	       Asis_D.Names (The_Declaration);
	    Parameters : constant Asis.Parameter_Specification_List := 
	       Asis_D.Parameters (The_Declaration);
	begin

	    if Trace.On then
		Trace.Log (The_Declaration);
		Trace.Add_Level (Formal_Parameters'Length + 
				 Names'Length + Parameters'Length);
	    end if;

	    Namespace_Scan.Scan_Declarative_Item_List 
	       (Formal_Parameters, The_Context);

	    Namespace_Scan.Scan_Entity_Name_Definition_List 
	       (Names, The_Declaration);

	    Namespace_Scan.Scan_Declaration_List (Parameters, The_Context);

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Declaration);
	    raise;

    end Scan_Generic_Procedure_Declaration;

--| +-------------------------------------------------------------------------+
--| | SCAN_INCOMPLETE_TYPE_DECLARATION (local)                                |
--| +-------------------------------------------------------------------------+

    procedure Scan_Incomplete_Type_Declaration 
		 (The_Declaration : in Asis.Declaration; 
		  The_Context : in Context) is

	Puid : constant String := "Scan_Incomplete_Type_Declaration";

    begin

	declare
	    Discriminants : constant Asis.Discriminant_Specification_List := 
	       Asis_D.Discriminants (The_Declaration);
	    Names : constant Asis.Entity_Name_Definition_List := 
	       Asis_D.Names (The_Declaration);
	begin

	    if Trace.On then
		Trace.Log (The_Declaration);
		Trace.Add_Level (Names'Length + Discriminants'Length);
	    end if;

	    Namespace_Scan.Scan_Entity_Name_Definition_List 
	       (Names, The_Declaration);

	    Namespace_Scan.Scan_Declaration_List (Discriminants, The_Context);

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Declaration);
	    raise;

    end Scan_Incomplete_Type_Declaration;

--| +-------------------------------------------------------------------------+
--| | SCAN_INTEGER_NUMBER_DECLARATION (local)                                 |
--| +-------------------------------------------------------------------------+

    procedure Scan_Integer_Number_Declaration 
		 (The_Declaration : in Asis.Declaration; 
		  The_Context : in Context) is

	Puid : constant String := "Scan_Integer_Number_Declaration";

    begin

	declare
	    Names : constant Asis.Entity_Name_Definition_List := 
	       Asis_D.Names (The_Declaration);
	    Value : Asis.Expression := Asis_D.Initial_Value (The_Declaration);
	begin

	    if Trace.On then
		Trace.Log (The_Declaration);
		Trace.Add_Level (Names'Length + 1);
	    end if;

	    Namespace_Scan.Scan_Entity_Name_Definition_List 
	       (Names, The_Declaration);

	    Namespace_Scan.Scan_Expression (Value, The_Context);

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Declaration);
	    raise;

    end Scan_Integer_Number_Declaration;

--| +-------------------------------------------------------------------------+
--| | SCAN_LOOP_PARAMETER_SPECIFICATION (local)                               |
--| +-------------------------------------------------------------------------+

    procedure Scan_Loop_Parameter_Specification 
		 (The_Declaration : in Asis.Declaration; 
		  The_Context : in Context) is

	Puid : constant String := "Scan_Loop_Parameter_Specification";

    begin

	declare
	    Names : constant Asis.Entity_Name_Definition_List := 
	       Asis_D.Names (The_Declaration);
	    The_Range : Asis.Discrete_Range := 
	       Asis_S.Loop_Parameter_Range (The_Declaration);
	begin

	    if Trace.On then
		Trace.Log (The_Declaration);
		Trace.Add_Level (Names'Length + 1);
	    end if;

	    Namespace_Scan.Scan_Entity_Name_Definition_List 
	       (Names, The_Declaration);

	    Namespace_Scan.Scan_Discrete_Range (The_Range, The_Context);

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Declaration);
	    raise;

    end Scan_Loop_Parameter_Specification;

--| +-------------------------------------------------------------------------+
--| | SCAN_OBJECT_RENAME_DECLARATION (local)                                  |
--| +-------------------------------------------------------------------------+

    procedure Scan_Object_Rename_Declaration 
		 (The_Declaration : in Asis.Declaration; 
		  The_Context : in Context) is

	Puid : constant String := "Scan_Object_Rename_Declaration";

    begin

	declare
	    Names : constant Asis.Entity_Name_Definition_List := 
	       Asis_D.Names (The_Declaration);
	    Renamed_Object : Asis.Expression := 
	       Asis_D.Renamed_Entity (The_Declaration);
	    The_Type : Asis.Expression := Asis_D.Type_Mark (The_Declaration);
	begin

	    if Trace.On then
		Trace.Log (The_Declaration);
		Trace.Add_Level (Names'Length + 2);
	    end if;

	    Namespace_Scan.Scan_Entity_Name_Definition_List 
	       (Names, The_Declaration);

	    Namespace_Scan.Scan_Expression (The_Type, The_Context);

	    Namespace_Scan.Scan_Expression (Renamed_Object, The_Context);

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Declaration);
	    raise;

    end Scan_Object_Rename_Declaration;

--| +-------------------------------------------------------------------------+
--| | SCAN_PACKAGE_BODY_DECLARATION (local)                                   |
--| +-------------------------------------------------------------------------+

    procedure Scan_Package_Body_Declaration 
		 (The_Declaration : in Asis.Declaration; 
		  The_Context : in Context) is

	Puid : constant String := "Scan_Package_Body_Declaration";

    begin

	declare
	    Names : constant Asis.Entity_Name_Definition_List := 
	       Asis_D.Names (The_Declaration);
	    The_Body : Asis.Statement := 
	       Asis_D.Package_Body_Block (The_Declaration);
	begin

	    if Trace.On then
		Trace.Log (The_Declaration);
		Trace.Add_Level (Names'Length + 1);
	    end if;

	    Namespace_Scan.Scan_Entity_Name_Definition_List 
	       (Names, The_Declaration);

	    Namespace_Scan.Scan_Statement (The_Body, The_Context);

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Declaration);
	    raise;

    end Scan_Package_Body_Declaration;

--| +-------------------------------------------------------------------------+
--| | SCAN_PACKAGE_BODY_STUB (local)                                          |
--| +-------------------------------------------------------------------------+

    procedure Scan_Package_Body_Stub (The_Declaration : in Asis.Declaration; 
				      The_Context : in Context) is

	Puid : constant String := "Scan_Package_Body_Stub";

    begin

	declare
	    Names : constant Asis.Entity_Name_Definition_List := 
	       Asis_D.Names (The_Declaration);
	begin

	    if Trace.On then
		Trace.Log (The_Declaration);
		Trace.Add_Level (Names'Length);
	    end if;

	    Namespace_Scan.Scan_Entity_Name_Definition_List 
	       (Names, The_Declaration);

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Declaration);
	    raise;

    end Scan_Package_Body_Stub;

--| +-------------------------------------------------------------------------+
--| | SCAN_PACKAGE_DECLARATION (local)                                        |
--| +-------------------------------------------------------------------------+

    procedure Scan_Package_Declaration (The_Declaration : in Asis.Declaration; 
					The_Context : in Context) is

	Puid : constant String := "Scan_Package_Declaration";

    begin

	declare
	    Names : constant Asis.Entity_Name_Definition_List := 
	       Asis_D.Names (The_Declaration);
	    Private_Declarations : constant Asis.Declarative_Item_List := 
	       Asis_D.Private_Part_Declarative_Items 
		  (The_Declaration, Namespace_Scan.Include_Package_Pragmas);
	    Public_Declarations : constant Asis.Declarative_Item_List := 
	       Asis_D.Visible_Part_Declarative_Items 
		  (The_Declaration, Namespace_Scan.Include_Package_Pragmas);
	begin

	    if Trace.On then
		Trace.Log (The_Declaration);
		Trace.Add_Level (Names'Length + Public_Declarations'Length + 
				 Private_Declarations'Length);
	    end if;

	    Namespace_Scan.Scan_Entity_Name_Definition_List 
	       (Names, The_Declaration);

	    Namespace_Scan.Scan_Declarative_Item_List 
	       (Public_Declarations, The_Context);

	    Namespace_Scan.Scan_Declarative_Item_List 
	       (Private_Declarations, The_Context);

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Declaration);
	    raise;

    end Scan_Package_Declaration;

--| +-------------------------------------------------------------------------+
--| | SCAN_PACKAGE_INSTANTIATION (local)                                      |
--| +-------------------------------------------------------------------------+

    procedure Scan_Package_Instantiation (The_Declaration : in Asis.Declaration; 
					  The_Context : in Context) is

	Puid : constant String := "Scan_Package_Instantiation";

    begin

	if Namespace_Scan.Expand_Package_Instantiations then

	    declare
		Specification : Asis.Declaration := 
		   Asis_D.Corresponding_Specification (The_Declaration);
		The_Body : Asis.Declaration := 
		   Asis_D.Corresponding_Body (The_Declaration);
	    begin
		if Trace.On then
		    Trace.Log (The_Declaration);
		    Trace.Add_Level (Namespace_Scan.One_If_Present 
					(Specification) + 
				     Namespace_Scan.One_If_Present (The_Body));
		end if;
		if not Asis_E.Is_Nil (Specification) then
		    Namespace_Scan.Scan_Declaration 
		       (Specification, The_Context);
		end if;
		if not Asis_E.Is_Nil (The_Body) then
		    Namespace_Scan.Scan_Declaration (The_Body, The_Context);
		end if;
	    end;

	else

	    declare
		Arguments : constant Asis.Association_List := 
		   Asis_D.Generic_Parameters 
		      (The_Declaration, Namespace_Scan.
					   Normalize_Instantiation_Parameters);
		Names : constant Asis.Entity_Name_Definition_List := 
		   Asis_D.Names (The_Declaration);
		The_Package : Asis.Expression := 
		   Asis_D.Generic_Unit_Name (The_Declaration);
	    begin
		if Trace.On then
		    Trace.Log (The_Declaration);
		    Trace.Add_Level (Names'Length + 1 + Arguments'Length);
		end if;
		Namespace_Scan.Scan_Entity_Name_Definition_List 
		   (Names, The_Declaration);
		Namespace_Scan.Scan_Expression (The_Package, The_Context);
		Namespace_Scan.Scan_Parameter_Association_List 
		   (Arguments, The_Context);
	    end;

	end if;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Declaration);
	    raise;

    end Scan_Package_Instantiation;

--| +-------------------------------------------------------------------------+
--| | SCAN_PACKAGE_RENAME_DECLARATION (local)                                 |
--| +-------------------------------------------------------------------------+

    procedure Scan_Package_Rename_Declaration 
		 (The_Declaration : in Asis.Declaration; 
		  The_Context : in Context) is

	Puid : constant String := "Scan_Package_Rename_Declaration";

    begin

	declare
	    Names : constant Asis.Entity_Name_Definition_List := 
	       Asis_D.Names (The_Declaration);
	    Renamed_Package : Asis.Expression := 
	       Asis_D.Renamed_Entity (The_Declaration);
	begin

	    if Trace.On then
		Trace.Log (The_Declaration);
		Trace.Add_Level (Names'Length + 1);
	    end if;

	    Namespace_Scan.Scan_Entity_Name_Definition_List 
	       (Names, The_Declaration);

	    Namespace_Scan.Scan_Expression (Renamed_Package, The_Context);

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Declaration);
	    raise;

    end Scan_Package_Rename_Declaration;

--| +-------------------------------------------------------------------------+
--| | SCAN_PARAMETER_SPECIFICATION (local)                                    |
--| +-------------------------------------------------------------------------+

    procedure Scan_Parameter_Specification 
		 (The_Declaration : in Asis.Declaration; 
		  The_Context : in Context) is

	Puid : constant String := "Scan_Parameter_Specification";

    begin

	declare
	    Default_Value : Asis.Expression := 
	       Asis_D.Initial_Value (The_Declaration);
	    Names : constant Asis.Entity_Name_Definition_List := 
	       Asis_D.Names (The_Declaration);
	    The_Type : Asis.Expression := Asis_D.Type_Mark (The_Declaration);
	begin

	    if Trace.On then
		Trace.Log (The_Declaration);
		Trace.Add_Level (Names'Length + 1 + 
				 Namespace_Scan.One_If_Present (Default_Value));
	    end if;

	    Namespace_Scan.Scan_Entity_Name_Definition_List 
	       (Names, The_Declaration);

	    Namespace_Scan.Scan_Expression (The_Type, The_Context);

	    if not Asis_E.Is_Nil (Default_Value) then
		Namespace_Scan.Scan_Expression (Default_Value, The_Context);
	    end if;

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Declaration);
	    raise;

    end Scan_Parameter_Specification;

--| +-------------------------------------------------------------------------+
--| | SCAN_PRIVATE_TYPE_DECLARATION (local)                                   |
--| +-------------------------------------------------------------------------+

    procedure Scan_Private_Type_Declaration 
		 (The_Declaration : in Asis.Declaration; 
		  The_Context : in Context) is

	Puid : constant String := "Scan_Private_Type_Declaration";

    begin

	declare
	    Discriminants : constant Asis.Discriminant_Specification_List := 
	       Asis_D.Discriminants (The_Declaration);
	    Names : constant Asis.Entity_Name_Definition_List := 
	       Asis_D.Names (The_Declaration);
	    The_Type : Asis.Type_Definition := 
	       Asis_D.Type_Declaration_Definition (The_Declaration);
	begin

	    if Trace.On then
		Trace.Log (The_Declaration);
		Trace.Add_Level (Names'Length + Discriminants'Length + 1);
	    end if;

	    Namespace_Scan.Scan_Entity_Name_Definition_List 
	       (Names, The_Declaration);

	    Namespace_Scan.Scan_Declaration_List (Discriminants, The_Context);

	    Namespace_Scan.Scan_Type_Definition (The_Type, The_Context);

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Declaration);
	    raise;

    end Scan_Private_Type_Declaration;

--| +-------------------------------------------------------------------------+
--| | SCAN_PROCEDURE_BODY_DECLARATION (local)                                 |
--| +-------------------------------------------------------------------------+

    procedure Scan_Procedure_Body_Declaration 
		 (The_Declaration : in Asis.Declaration; 
		  The_Context : in Context) is

	Puid : constant String := "Scan_Procedure_Body_Declaration";

    begin

	declare
	    Names : constant Asis.Entity_Name_Definition_List := 
	       Asis_D.Names (The_Declaration);
	    Parameters : constant Asis.Parameter_Specification_List := 
	       Asis_D.Parameters (The_Declaration);
	    The_Body : Asis.Statement := 
	       Asis_D.Subprogram_Body_Block (The_Declaration);
	begin

	    if Trace.On then
		Trace.Log (The_Declaration);
		Trace.Add_Level (Names'Length + Parameters'Length + 1);
	    end if;

	    Namespace_Scan.Scan_Entity_Name_Definition_List 
	       (Names, The_Declaration);

	    Namespace_Scan.Scan_Declaration_List (Parameters, The_Context);

	    Namespace_Scan.Scan_Statement (The_Body, The_Context);

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Declaration);
	    raise;

    end Scan_Procedure_Body_Declaration;

--| +-------------------------------------------------------------------------+
--| | SCAN_PROCEDURE_BODY_STUB (local)                                        |
--| +-------------------------------------------------------------------------+

    procedure Scan_Procedure_Body_Stub (The_Declaration : in Asis.Declaration; 
					The_Context : in Context) is

	Puid : constant String := "Scan_Procedure_Body_Stub";

    begin

	declare
	    Names : constant Asis.Entity_Name_Definition_List := 
	       Asis_D.Names (The_Declaration);
	    Parameters : constant Asis.Parameter_Specification_List := 
	       Asis_D.Parameters (The_Declaration);
	begin

	    if Trace.On then
		Trace.Log (The_Declaration);
		Trace.Add_Level (Names'Length + Parameters'Length);
	    end if;

	    Namespace_Scan.Scan_Entity_Name_Definition_List 
	       (Names, The_Declaration);

	    Namespace_Scan.Scan_Declaration_List (Parameters, The_Context);

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Declaration);
	    raise;

    end Scan_Procedure_Body_Stub;

--| +-------------------------------------------------------------------------+
--| | SCAN_PROCEDURE_DECLARATION (local)                                      |
--| +-------------------------------------------------------------------------+

    procedure Scan_Procedure_Declaration (The_Declaration : in Asis.Declaration; 
					  The_Context : in Context) is

	Puid : constant String := "Scan_Procedure_Declaration";

    begin

	declare
	    Names : constant Asis.Entity_Name_Definition_List := 
	       Asis_D.Names (The_Declaration);
	    Parameters : constant Asis.Parameter_Specification_List := 
	       Asis_D.Parameters (The_Declaration);
	begin

	    if Trace.On then
		Trace.Log (The_Declaration);
		Trace.Add_Level (Names'Length + Parameters'Length);
	    end if;

	    Namespace_Scan.Scan_Entity_Name_Definition_List 
	       (Names, The_Declaration);

	    Namespace_Scan.Scan_Declaration_List (Parameters, The_Context);

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Declaration);
	    raise;

    end Scan_Procedure_Declaration;

--| +-------------------------------------------------------------------------+
--| | SCAN_PROCEDURE_INSTANTIATION (local)                                    |
--| +-------------------------------------------------------------------------+

    procedure Scan_Procedure_Instantiation 
		 (The_Declaration : in Asis.Declaration; 
		  The_Context : in Context) is

	Puid : constant String := "Scan_Procedure_Instantiation";

    begin

	if Namespace_Scan.Expand_Procedure_Instantiations then

	    declare
		Specification : Asis.Declaration := 
		   Asis_D.Corresponding_Specification (The_Declaration);
		The_Body : Asis.Declaration := 
		   Asis_D.Corresponding_Body (The_Declaration);
	    begin
		if Trace.On then
		    Trace.Log (The_Declaration);
		    Trace.Add_Level (Namespace_Scan.One_If_Present 
					(Specification) + 
				     Namespace_Scan.One_If_Present (The_Body));
		end if;
		if not Asis_E.Is_Nil (Specification) then
		    Namespace_Scan.Scan_Declaration 
		       (Specification, The_Context);
		end if;
		if not Asis_E.Is_Nil (The_Body) then
		    Namespace_Scan.Scan_Declaration (The_Body, The_Context);
		end if;
	    end;

	else

	    declare
		Arguments : constant Asis.Association_List := 
		   Asis_D.Generic_Parameters 
		      (The_Declaration, Namespace_Scan.
					   Normalize_Instantiation_Parameters);
		Names : constant Asis.Entity_Name_Definition_List := 
		   Asis_D.Names (The_Declaration);
		The_Procedure : Asis.Expression := 
		   Asis_D.Generic_Unit_Name (The_Declaration);
	    begin
		if Trace.On then
		    Trace.Log (The_Declaration);
		    Trace.Add_Level (Names'Length + 1 + Arguments'Length);
		end if;
		Namespace_Scan.Scan_Entity_Name_Definition_List 
		   (Names, The_Declaration);
		Namespace_Scan.Scan_Expression (The_Procedure, The_Context);
		Namespace_Scan.Scan_Parameter_Association_List 
		   (Arguments, The_Context);
	    end;

	end if;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Declaration);
	    raise;

    end Scan_Procedure_Instantiation;

--| +-------------------------------------------------------------------------+
--| | SCAN_PROCEDURE_RENAME_DECLARATION (local)                               |
--| +-------------------------------------------------------------------------+

    procedure Scan_Procedure_Rename_Declaration 
		 (The_Declaration : in Asis.Declaration; 
		  The_Context : in Context) is

	Puid : constant String := "Scan_Procedure_Rename_Declaration";

    begin

	declare
	    Names : constant Asis.Entity_Name_Definition_List := 
	       Asis_D.Names (The_Declaration);
	    Parameters : constant Asis.Parameter_Specification_List := 
	       Asis_D.Parameters (The_Declaration);
	    Renamed_Procedure : Asis.Expression := 
	       Asis_D.Renamed_Entity (The_Declaration);
	begin

	    if Trace.On then
		Trace.Log (The_Declaration);
		Trace.Add_Level (Names'Length + Parameters'Length + 1);
	    end if;

	    Namespace_Scan.Scan_Entity_Name_Definition_List 
	       (Names, The_Declaration);

	    Namespace_Scan.Scan_Declaration_List (Parameters, The_Context);

	    Namespace_Scan.Scan_Expression (Renamed_Procedure, The_Context);

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Declaration);
	    raise;

    end Scan_Procedure_Rename_Declaration;

--| +-------------------------------------------------------------------------+
--| | SCAN_REAL_NUMBER_DECLARATION (local)                                    |
--| +-------------------------------------------------------------------------+

    procedure Scan_Real_Number_Declaration 
		 (The_Declaration : in Asis.Declaration; 
		  The_Context : in Context) is

	Puid : constant String := "Scan_Real_Number_Declaration";

    begin

	declare
	    Names : constant Asis.Entity_Name_Definition_List := 
	       Asis_D.Names (The_Declaration);
	    Value : Asis.Expression := Asis_D.Initial_Value (The_Declaration);
	begin

	    if Trace.On then
		Trace.Log (The_Declaration);
		Trace.Add_Level (Names'Length + 1);
	    end if;

	    Namespace_Scan.Scan_Entity_Name_Definition_List 
	       (Names, The_Declaration);

	    Namespace_Scan.Scan_Expression (Value, The_Context);

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Declaration);
	    raise;

    end Scan_Real_Number_Declaration;

--| +-------------------------------------------------------------------------+
--| | SCAN_SUBTYPE_DECLARATION (local)                                        |
--| +-------------------------------------------------------------------------+

    procedure Scan_Subtype_Declaration (The_Declaration : in Asis.Declaration; 
					The_Context : in Context) is

	Puid : constant String := "Scan_Subtype_Declaration";

    begin

	declare
	    Names : constant Asis.Entity_Name_Definition_List := 
	       Asis_D.Names (The_Declaration);
	    The_Type : Asis.Type_Definition := 
	       Asis_D.Type_Declaration_Definition (The_Declaration);
	begin

	    if Trace.On then
		Trace.Log (The_Declaration);
		Trace.Add_Level (Names'Length + 1);
	    end if;

	    Namespace_Scan.Scan_Entity_Name_Definition_List 
	       (Names, The_Declaration);

	    Namespace_Scan.Scan_Type_Definition (The_Type, The_Context);

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Declaration);
	    raise;

    end Scan_Subtype_Declaration;

--| +-------------------------------------------------------------------------+
--| | SCAN_TASK_BODY_DECLARATION (local)                                      |
--| +-------------------------------------------------------------------------+

    procedure Scan_Task_Body_Declaration (The_Declaration : in Asis.Declaration; 
					  The_Context : in Context) is

	Puid : constant String := "Scan_Task_Body_Declaration";

    begin

	declare
	    Names : constant Asis.Entity_Name_Definition_List := 
	       Asis_D.Names (The_Declaration);
	    The_Body : Asis.Statement := 
	       Asis_D.Task_Body_Block (The_Declaration);
	begin

	    if Trace.On then
		Trace.Log (The_Declaration);
		Trace.Add_Level (Names'Length + 1);
	    end if;

	    Namespace_Scan.Scan_Entity_Name_Definition_List 
	       (Names, The_Declaration);

	    Namespace_Scan.Scan_Statement (The_Body, The_Context);

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Declaration);
	    raise;

    end Scan_Task_Body_Declaration;

--| +-------------------------------------------------------------------------+
--| | SCAN_TASK_BODY_STUB (local)                                             |
--| +-------------------------------------------------------------------------+

    procedure Scan_Task_Body_Stub (The_Declaration : in Asis.Declaration; 
				   The_Context : in Context) is

	Puid : constant String := "Scan_Task_Body_Stub";

    begin

	declare
	    Names : constant Asis.Entity_Name_Definition_List := 
	       Asis_D.Names (The_Declaration);
	begin

	    if Trace.On then
		Trace.Log (The_Declaration);
		Trace.Add_Level (Names'Length);
	    end if;

	    Namespace_Scan.Scan_Entity_Name_Definition_List 
	       (Names, The_Declaration);

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Declaration);
	    raise;

    end Scan_Task_Body_Stub;

--| +-------------------------------------------------------------------------+
--| | SCAN_TASK_DECLARATION (local)                                           |
--| +-------------------------------------------------------------------------+

    procedure Scan_Task_Declaration (The_Declaration : in Asis.Declaration; 
				     The_Context : in Context) is

	Puid : constant String := "Scan_Task_Declaration";

    begin

	declare
	    Declarations : constant Asis.Declarative_Item_List := 
	       Asis_D.Task_Declaration_Declarative_Items 
		  (The_Declaration, Namespace_Scan.
				       Include_Task_Declaration_Pragmas);
	    Names : constant Asis.Entity_Name_Definition_List := 
	       Asis_D.Names (The_Declaration);
	begin

	    if Trace.On then
		Trace.Log (The_Declaration);
		Trace.Add_Level (Names'Length + Declarations'Length);
	    end if;

	    Namespace_Scan.Scan_Entity_Name_Definition_List 
	       (Names, The_Declaration);

	    Namespace_Scan.Scan_Declarative_Item_List 
	       (Declarations, The_Context);

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Declaration);
	    raise;

    end Scan_Task_Declaration;

--| +-------------------------------------------------------------------------+
--| | SCAN_TASK_TYPE_DECLARATION (local)                                      |
--| +-------------------------------------------------------------------------+

    procedure Scan_Task_Type_Declaration (The_Declaration : in Asis.Declaration; 
					  The_Context : in Context) is

	Puid : constant String := "Scan_Task_Type_Declaration";

    begin

	declare
	    Names : constant Asis.Entity_Name_Definition_List := 
	       Asis_D.Names (The_Declaration);
	    The_Type : Asis.Type_Definition := 
	       Asis_D.Type_Declaration_Definition (The_Declaration);
	begin

	    if Trace.On then
		Trace.Log (The_Declaration);
		Trace.Add_Level (Names'Length + 1);
	    end if;

	    Namespace_Scan.Scan_Entity_Name_Definition_List 
	       (Names, The_Declaration);

	    Namespace_Scan.Scan_Type_Definition (The_Type, The_Context);

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Declaration);
	    raise;

    end Scan_Task_Type_Declaration;

--| +-------------------------------------------------------------------------+
--| | SCAN_VARIABLE_DECLARATION (local)                                       |
--| +-------------------------------------------------------------------------+

    procedure Scan_Variable_Declaration (The_Declaration : in Asis.Declaration; 
					 The_Context : in Context) is

	Puid : constant String := "Scan_Variable_Declaration";

    begin

	declare
	    Initial_Value : Asis.Expression := 
	       Asis_D.Initial_Value (The_Declaration);
	    Names : constant Asis.Entity_Name_Definition_List := 
	       Asis_D.Names (The_Declaration);
	    The_Type : Asis.Type_Definition := 
	       Asis_D.Object_Declaration_Definition (The_Declaration);
	begin

	    if Trace.On then
		Trace.Log (The_Declaration);
		Trace.Add_Level (Names'Length + 1 + 
				 Namespace_Scan.One_If_Present (Initial_Value));
	    end if;

	    Namespace_Scan.Scan_Entity_Name_Definition_List 
	       (Names, The_Declaration);

	    Namespace_Scan.Scan_Type_Definition (The_Type, The_Context);

	    if not Asis_E.Is_Nil (Initial_Value) then
		Namespace_Scan.Scan_Expression (Initial_Value, The_Context);
	    end if;

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Declaration);
	    raise;

    end Scan_Variable_Declaration;

--| +-------------------------------------------------------------------------+
--| | SCAN_DECLARATION (exported)                                             |
--| +-------------------------------------------------------------------------+

begin

    if Namespace_Scan.Obeying_Regions and then 
       Rgn.Denotes_One_Or_More_Regions (The_Declaration) then
	declare
	    Remaining_Parent_Elements : constant Asis.Element_List := 
	       Rgn.Non_Region_Subelements 
		  (The_Declaration, Include_Pragmas => False);
	begin
	    Namespace_Scan.Scan_Any_List 
	       (Remaining_Parent_Elements, The_Declaration);
	end;
	return;
    end if;

    case Asis_D.Kind (The_Declaration) is
	when Asis_D.A_Variable_Declaration =>
	    Scan_Variable_Declaration (The_Declaration, The_Context);
	when Asis_D.A_Component_Declaration =>
	    Scan_Component_Declaration (The_Declaration, The_Context);
	when Asis_D.A_Constant_Declaration =>
	    Scan_Constant_Declaration (The_Declaration, The_Context);
	when Asis_D.A_Deferred_Constant_Declaration =>
	    Scan_Deferred_Constant_Declaration (The_Declaration, The_Context);
	when Asis_D.A_Generic_Formal_Object_Declaration =>
	    Scan_Generic_Formal_Object_Declaration 
	       (The_Declaration, The_Context);
	when Asis_D.A_Discriminant_Specification =>
	    Scan_Discriminant_Specification (The_Declaration, The_Context);
	when Asis_D.A_Parameter_Specification =>
	    Scan_Parameter_Specification (The_Declaration, The_Context);
	when Asis_D.An_Integer_Number_Declaration =>
	    Scan_Integer_Number_Declaration (The_Declaration, The_Context);
	when Asis_D.A_Real_Number_Declaration =>
	    Scan_Real_Number_Declaration (The_Declaration, The_Context);
	when Asis_D.An_Exception_Declaration =>
	    Scan_Exception_Declaration (The_Declaration, The_Context);
	when Asis_D.An_Enumeration_Literal_Specification =>
	    Scan_Enumeration_Literal_Specification 
	       (The_Declaration, The_Context);
	when Asis_D.A_Loop_Parameter_Specification =>
	    Scan_Loop_Parameter_Specification (The_Declaration, The_Context);
	when Asis_D.A_Full_Type_Declaration =>
	    Scan_Full_Type_Declaration (The_Declaration, The_Context);
	when Asis_D.An_Incomplete_Type_Declaration =>
	    Scan_Incomplete_Type_Declaration (The_Declaration, The_Context);
	when Asis_D.A_Private_Type_Declaration =>
	    Scan_Private_Type_Declaration (The_Declaration, The_Context);
	when Asis_D.A_Subtype_Declaration =>
	    Scan_Subtype_Declaration (The_Declaration, The_Context);
	when Asis_D.A_Package_Declaration =>
	    Scan_Package_Declaration (The_Declaration, The_Context);
	when Asis_D.A_Package_Body_Declaration =>
	    Scan_Package_Body_Declaration (The_Declaration, The_Context);
	when Asis_D.A_Procedure_Declaration =>
	    Scan_Procedure_Declaration (The_Declaration, The_Context);
	when Asis_D.A_Procedure_Body_Declaration =>
	    Scan_Procedure_Body_Declaration (The_Declaration, The_Context);
	when Asis_D.A_Function_Declaration =>
	    Scan_Function_Declaration (The_Declaration, The_Context);
	when Asis_D.A_Function_Body_Declaration =>
	    Scan_Function_Body_Declaration (The_Declaration, The_Context);
	when Asis_D.An_Object_Rename_Declaration =>
	    Scan_Object_Rename_Declaration (The_Declaration, The_Context);
	when Asis_D.An_Exception_Rename_Declaration =>
	    Scan_Exception_Rename_Declaration (The_Declaration, The_Context);
	when Asis_D.A_Package_Rename_Declaration =>
	    Scan_Package_Rename_Declaration (The_Declaration, The_Context);
	when Asis_D.A_Procedure_Rename_Declaration =>
	    Scan_Procedure_Rename_Declaration (The_Declaration, The_Context);
	when Asis_D.A_Function_Rename_Declaration =>
	    Scan_Function_Rename_Declaration (The_Declaration, The_Context);
	when Asis_D.A_Generic_Package_Declaration =>
	    Scan_Generic_Package_Declaration (The_Declaration, The_Context);
	when Asis_D.A_Generic_Procedure_Declaration =>
	    Scan_Generic_Procedure_Declaration (The_Declaration, The_Context);
	when Asis_D.A_Generic_Function_Declaration =>
	    Scan_Generic_Function_Declaration (The_Declaration, The_Context);
	when Asis_D.A_Package_Instantiation =>
	    Scan_Package_Instantiation (The_Declaration, The_Context);
	when Asis_D.A_Procedure_Instantiation =>
	    Scan_Procedure_Instantiation (The_Declaration, The_Context);
	when Asis_D.A_Function_Instantiation =>
	    Scan_Function_Instantiation (The_Declaration, The_Context);
	when Asis_D.A_Task_Declaration =>
	    Scan_Task_Declaration (The_Declaration, The_Context);
	when Asis_D.A_Task_Body_Declaration =>
	    Scan_Task_Body_Declaration (The_Declaration, The_Context);
	when Asis_D.A_Task_Type_Declaration =>
	    Scan_Task_Type_Declaration (The_Declaration, The_Context);
	when Asis_D.An_Entry_Declaration =>
	    Scan_Entry_Declaration (The_Declaration, The_Context);
	when Asis_D.A_Procedure_Body_Stub =>
	    Scan_Procedure_Body_Stub (The_Declaration, The_Context);
	when Asis_D.A_Function_Body_Stub =>
	    Scan_Function_Body_Stub (The_Declaration, The_Context);
	when Asis_D.A_Package_Body_Stub =>
	    Scan_Package_Body_Stub (The_Declaration, The_Context);
	when Asis_D.A_Task_Body_Stub =>
	    Scan_Task_Body_Stub (The_Declaration, The_Context);
	when Asis_D.A_Generic_Formal_Type_Declaration =>
	    Scan_Generic_Formal_Type_Declaration (The_Declaration, The_Context);
	when Asis_D.A_Generic_Formal_Private_Type_Declaration =>
	    Scan_Generic_Formal_Private_Type_Declaration 
	       (The_Declaration, The_Context);
	when Asis_D.A_Generic_Formal_Procedure_Declaration =>
	    Scan_Generic_Formal_Procedure_Declaration 
	       (The_Declaration, The_Context);
	when Asis_D.A_Generic_Formal_Function_Declaration =>
	    Scan_Generic_Formal_Function_Declaration 
	       (The_Declaration, The_Context);
	when Asis_D.Not_A_Declaration =>
	    Error.Log (Error.An_Unhandled_Case, Cuid, Puid, The_Declaration);
	    if Namespace_Scan.Raise_Exception_On_Unhandled_Case then
		raise Traversal_Error;
	    else
		if Trace.On then
		    Trace.Log (The_Declaration);
		end if;
	    end if;
    end case;

end Scan_Declaration;
