--| +=========================================================================+
--| |                                                                         |
--| | NAMESPACE_SCAN.SCAN_DISCRETE_RANGE (body)                               |
--| |                                                                         |
--| | Greg Janee                                                              |
--| | General Research Corporation                                            |
--| |                                                                         |
--| +=========================================================================+

separate (Namespace_Scan)
procedure Scan_Discrete_Range 
	     (The_Range : in Asis.Discrete_Range; The_Context : in Context) is

    Cuid : constant String := "Namespace_Scan.Scan_Discrete_Range";
    Puid : constant String := "Scan_Discrete_Range";

    package Error renames Namespace_Scan.Error_Handling_Support;
    package Trace renames Namespace_Scan.Trace_Support;

--| +-------------------------------------------------------------------------+
--| | SCAN_DISCRETE_SUBTYPE_INDICATION (local)                                |
--| +-------------------------------------------------------------------------+

    procedure Scan_Discrete_Subtype_Indication 
		 (The_Range : in Asis.Discrete_Range; 
		  The_Context : in Context) is

	Puid : constant String := "Scan_Discrete_Subtype_Indication";

    begin

	declare
	    Constraint : Asis.Constraint := 
	       Asis_Td.Subtype_Constraint (The_Range);
	    Type_Mark : Asis.Expression := Asis_Td.Type_Mark (The_Range);
	begin

	    if Trace.On then
		Trace.Log (The_Range);
		Trace.Add_Level (1 + 
				 Namespace_Scan.One_If_Present (Constraint));
	    end if;

	    Namespace_Scan.Scan_Expression (Type_Mark, The_Context);

	    if not Asis_E.Is_Nil (Constraint) then
		Namespace_Scan.Scan_Constraint (Constraint, The_Context);
	    end if;

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Range);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Range);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Range);
	    raise;

    end Scan_Discrete_Subtype_Indication;

--| +-------------------------------------------------------------------------+
--| | SCAN_RANGE_ATTRIBUTE (local)                                            |
--| +-------------------------------------------------------------------------+

    procedure Scan_Range_Attribute (The_Range : in Asis.Discrete_Range; 
				    The_Context : in Context) is

	Puid : constant String := "Scan_Range_Attribute";

    begin

	declare
	    Attribute : Asis.Expression := Asis_Td.Range_Attribute (The_Range);
	begin

	    if Trace.On then
		Trace.Log (The_Range);
		Trace.Add_Level (1);
	    end if;

	    Namespace_Scan.Scan_Expression (Attribute, The_Context);

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Range);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Range);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Range);
	    raise;

    end Scan_Range_Attribute;

--| +-------------------------------------------------------------------------+
--| | SCAN_SIMPLE_RANGE (local)                                               |
--| +-------------------------------------------------------------------------+

    procedure Scan_Simple_Range (The_Range : in Asis.Discrete_Range; 
				 The_Context : in Context) is

	Puid : constant String := "Scan_Simple_Range";

    begin

	declare
	    Lower_Bound : Asis.Expression := Asis_Td.Lower_Bound (The_Range);
	    Upper_Bound : Asis.Expression := Asis_Td.Upper_Bound (The_Range);
	begin

	    if Trace.On then
		Trace.Log (The_Range);
		Trace.Add_Level (2);
	    end if;

	    Namespace_Scan.Scan_Expression (Lower_Bound, The_Context);

	    Namespace_Scan.Scan_Expression (Upper_Bound, The_Context);

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Range);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Range);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Range);
	    raise;

    end Scan_Simple_Range;

--| +-------------------------------------------------------------------------+
--| | SCAN_DISCRETE_RANGE (exported)                                          |
--| +-------------------------------------------------------------------------+

begin

    if Namespace_Scan.Obeying_Regions and then 
       Rgn.Denotes_One_Or_More_Regions (The_Range) then
	declare
	    Remaining_Parent_Elements : constant Asis.Element_List := 
	       Rgn.Non_Region_Subelements (The_Range, Include_Pragmas => False);
	begin
	    Namespace_Scan.Scan_Any_List (Remaining_Parent_Elements, The_Range);
	end;
	return;
    end if;

    case Asis_Td.Discrete_Range_Kind (The_Range) is
	when Asis_Td.A_Simple_Range =>
	    Scan_Simple_Range (The_Range, The_Context);
	when Asis_Td.A_Range_Attribute =>
	    Scan_Range_Attribute (The_Range, The_Context);
	when Asis_Td.A_Discrete_Subtype_Indication =>
	    Scan_Discrete_Subtype_Indication (The_Range, The_Context);
	when Asis_Td.Not_A_Discrete_Range =>
	    Error.Log (Error.An_Unhandled_Case, Cuid, Puid, The_Range);
	    if Namespace_Scan.Raise_Exception_On_Unhandled_Case then
		raise Traversal_Error;
	    else
		if Trace.On then
		    Trace.Log (The_Range);
		end if;
	    end if;
    end case;

end Scan_Discrete_Range;
