--| +=========================================================================+
--| |                                                                         |
--| | READ_REFERENCE_VIEW                                                     |
--| |                                                                         |
--| | Usage: read_reference_view <library>                                    |
--| |                                                                         |
--| | This program tests the Reference_View package's ability to load a       |
--| | reference view from a pair of files "z.ctl" and "z.ids".  It accepts    |
--| | one command line argument, the name of the relevant Ada library         |
--| | (e.g., the name of an Apex view).  A textual dump of the reference view |
--| | is written to standard output.                                          |
--| |                                                                         |
--| | Greg Janee                                                              |
--| | General Research Corporation                                            |
--| |                                                                         |
--| +=========================================================================+

with Asis;
with Command;
with Msg_Log;
with Reference_View;
with Reference_View_Structures;
with Text_Io;

procedure Read_Reference_View is

--| Standard Asis renames...

    package Asis_En renames Asis.Environment;
    package Asis_Id_Io renames Asis.Ids.Id_Io;
    package Asis_L renames Asis.Libraries;
    package Asis_Str renames Asis.Strings;

    package Cli renames Command;

    Argument_Count : Integer := Cli.Argc - 1;
    Arguments : constant Cli.String_Ptr_Array := Cli.Arguments;
    Cfile : Text_Io.File_Type;
    Ifile : Asis_Id_Io.File_Type;
    The_Library : Asis.Library;
    The_View : Reference_View_Structures.Reference_View;

begin

    Msg_Log.Set_Program ("rrv");

    if Argument_Count /= 1 then
	Msg_Log.Put_Msg (Msg_Log.Error, 
			 "usage is ""read_reference_view <library>""");
	return;
    end if;

    Asis_En.Initialize;
    Asis_L.Associate (The_Library, Asis_Str.To_Asis_String (Arguments (1).all));
    Asis_L.Open (The_Library);

    Text_Io.Open (Cfile, Text_Io.In_File, "z.ctl");
    Asis_Id_Io.Open (Ifile, Asis_Id_Io.In_File, "z.ids");

    The_View := Reference_View.Input (Cfile, Ifile, The_Library);
    Reference_View.Dump (The_View);

    Reference_View.Free (The_View);

    Asis_L.Close (The_Library);
    Asis_L.Dissociate (The_Library);
    Asis_En.Finalize;

exception

    when Asis.Asis_Failed =>
	Msg_Log.Put_Msg (Msg_Log.Error, 
			 "exception Asis_Failed raised; status is " & 
			    Asis_En.Error_Kinds'Image (Asis_En.Status) & 
			    "; diagnosis follows");
	Msg_Log.Put_Msg (Msg_Log.Error, 
			 Asis_Str.To_Standard_String (Asis_En.Diagnosis));

    when Reference_View.Traversal_Error =>
	Msg_Log.Put_Msg (Msg_Log.Error, "exception Traversal_Error raised");

end Read_Reference_View;
