--| +=========================================================================+
--| |                                                                         |
--| | REFERENCE_SCAN.SCAN_EXPRESSION (body)                                   |
--| |                                                                         |
--| | Greg Janee                                                              |
--| | General Research Corporation                                            |
--| |                                                                         |
--| +=========================================================================+

separate (Reference_Scan)
procedure Scan_Expression (The_Expression : in Asis.Expression; 
			   The_Context : in Context) is

    Cuid : constant String := "Reference_Scan.Scan_Expression";
    Puid : constant String := "Scan_Expression";

    package Ctx renames Reference_Scan.Context_Support;
    package Error renames Reference_Scan.Error_Handling_Support;
    package Trace renames Reference_Scan.Trace_Support;

--| Local subprograms.

    procedure Log_Reference (The_Expression : in Asis.Expression; 
			     The_Context : in Context; 
			     Is_Collection_Reference : in Boolean := False);

    procedure Scan_Possible_Access_Subexpression 
		 (The_Expression : in Asis.Expression; 
		  The_Context : in Context);

--| +-------------------------------------------------------------------------+
--| | LOG_REFERENCE (local)                                                   |
--| +-------------------------------------------------------------------------+
--|
--| Logs a reference.

    procedure Log_Reference (The_Expression : in Asis.Expression; 
			     The_Context : in Context; 
			     Is_Collection_Reference : in Boolean := False) is

	Puid : constant String := "Log_Reference";

	H : Reference_Holder_Ptr;
	R : Rvs.Reference;

    begin

	case The_Context.Basic_Context is
	    when Rvs.A_Read | Rvs.An_Update | Rvs.A_Read_And_Update =>
		if Rvs."=" (The_Context.Data_Access_Context, null) then
		    Error.Malformed_Reference_Error 
		       (The_Expression, "empty data access context");
		    return;
		end if;
	    when Rvs.A_Type_Mark =>
		if Rvs."=" (The_Context.Type_Mark_Context, Rvs.Unknown) then
		    Error.Malformed_Reference_Error 
		       (The_Expression, "unknown type mark context");
		    return;
		end if;
	    when Rvs.Unknown =>
		Error.Malformed_Reference_Error 
		   (The_Expression, "unknown reference kind");
		return;
	    when others =>
		null;
	end case;

	R.Reference_Element := The_Expression;
	R.Context := new Rvs.Reference_Context (The_Context.Basic_Context);
	R.Is_Collection_Reference := Is_Collection_Reference;

	case The_Context.Basic_Context is
	    when Rvs.A_Read | Rvs.An_Update | Rvs.A_Read_And_Update =>
		R.Context.Data_Access_Context := 
		   new Rvs.Data_Access_Context 
			  (1 .. The_Context.Data_Access_Context'Length);
		R.Context.Data_Access_Context.all := 
		   The_Context.Data_Access_Context.all;
	    when Rvs.A_Type_Mark =>
		R.Context.Type_Mark_Context := The_Context.Type_Mark_Context;
	    when others =>
		null;
	end case;

	H := new Reference_Holder;
	H.Next := Reference_Scan.Reference_List;
	Reference_Scan.Reference_List := H;
	H.Reference := R;

    end Log_Reference;

--| +-------------------------------------------------------------------------+
--| | SCAN_ACCESS_OBJECT_SELECTION (local)                                    |
--| +-------------------------------------------------------------------------+

    procedure Scan_Access_Object_Selection (The_Expression : in Asis.Expression; 
					    The_Context : in Context) is

	Puid : constant String := "Scan_Access_Object_Selection";

    begin

	declare
	    Prefix : Asis.Expression := Asis_X.Prefix (The_Expression);
	begin

	    if Trace.On then
		Trace.Log (The_Expression, The_Context);
		Trace.Add_Level (1);
	    end if;

	    Log_Reference (Prefix, The_Context, 
			   Is_Collection_Reference => True);

	    Reference_Scan.Scan_Expression 
	       (Prefix, Ctx.Set (The_Context, 
				 Basic_Context => Rvs.A_Read, 
				 Weight => 95, 
				 New_Weight => 5, 
				 Add_Data_Access_Context => Rvs.A_Dereference));

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Expression);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Expression);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Expression);
	    raise;

    end Scan_Access_Object_Selection;

--| +-------------------------------------------------------------------------+
--| | SCAN_AGGREGATE (local)                                                  |
--| +-------------------------------------------------------------------------+

    procedure Scan_Aggregate (The_Expression : in Asis.Expression; 
			      The_Context : in Context) is

	Puid : constant String := "Scan_Aggregate";

	Normalize : Boolean;

    begin

	Normalize := Reference_Scan.Normalize_Record_Aggregate_Components;

--| Components will raise Asis_Inappropriate_Element if normalization is
--| requested and the aggregate is an array aggregate.  Unfortunately, there's
--| no easy way to determine the type of the aggregate before calling
--| Components.  Thus if normalization is requested we first "practice"
--| calling Components to see if the above exception will be raised.  If it
--| will be, and if the exception appears to be due to the
--| aggregate being an array aggregate (and not, say, to the element being
--| completely bogus) we turn normalization off.

	if Normalize then
	    begin
		declare
		    Components : constant Asis.Component_Association_List := 
		       Asis_X.Components (The_Expression, Normalize);
		begin
		    null;
		end;
	    exception
		when Asis.Asis_Inappropriate_Element =>
		    if Asis_E."=" (Asis_E.Element_Kind (The_Expression), 
				   Asis_E.An_Expression) and 
		       Asis_X."=" (Asis_X.Kind (The_Expression), 
				   Asis_X.An_Aggregate) then
			Normalize := False;
		    end if;
	    end;
	end if;

	declare
	    Components : constant Asis.Component_Association_List := 
	       Asis_X.Components (The_Expression, Normalize);
	begin

	    if Trace.On then
		Trace.Log (The_Expression, The_Context);
		Trace.Add_Level (Components'Length);
	    end if;

	    Reference_Scan.Scan_Component_Association_List 
	       (Components, The_Context);

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Expression);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Expression);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Expression);
	    raise;

    end Scan_Aggregate;

--| +-------------------------------------------------------------------------+
--| | SCAN_ALLOCATION_FROM_QUALIFIED_EXPRESSION (local)                       |
--| +-------------------------------------------------------------------------+

    procedure Scan_Allocation_From_Qualified_Expression 
		 (The_Expression : in Asis.Expression; 
		  The_Context : in Context) is

	Puid : constant String := "Scan_Allocation_From_Qualified_Expression";

    begin

	declare
	    Qualified_Expression : Asis.Expression := 
	       Asis_X.Qualified_Object_Expression (The_Expression);
	begin

	    if Trace.On then
		Trace.Log (The_Expression, The_Context);
		Trace.Add_Level (1);
	    end if;

	    Reference_Scan.Scan_Expression 
	       (Qualified_Expression, 
		Ctx.Set (The_Context, 
			 Basic_Context => Rvs.A_Read, 
			 Weight => 95, 
			 New_Weight => 5, 
			 Add_Data_Access_Context => Rvs.An_Allocator));

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Expression);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Expression);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Expression);
	    raise;

    end Scan_Allocation_From_Qualified_Expression;

--| +-------------------------------------------------------------------------+
--| | SCAN_ALLOCATION_FROM_SUBTYPE (local)                                    |
--| +-------------------------------------------------------------------------+

    procedure Scan_Allocation_From_Subtype (The_Expression : in Asis.Expression; 
					    The_Context : in Context) is

	Puid : constant String := "Scan_Allocation_From_Subtype";

    begin

	declare
	    The_Subtype : Asis.Subtype_Indication := 
	       Asis_X.Allocation_Type (The_Expression);
	begin

	    if Trace.On then
		Trace.Log (The_Expression, The_Context);
		Trace.Add_Level (1);
	    end if;

	    Reference_Scan.Scan_Subtype_Indication 
	       (The_Subtype, Ctx.Set (The_Context, 
				      Type_Mark_Context => Rvs.An_Allocator));

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Expression);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Expression);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Expression);
	    raise;

    end Scan_Allocation_From_Subtype;

--| +-------------------------------------------------------------------------+
--| | SCAN_ATTRIBUTE (local)                                                  |
--| +-------------------------------------------------------------------------+

    procedure Scan_Attribute (The_Expression : in Asis.Expression; 
			      The_Context : in Context) is

	Puid : constant String := "Scan_Attribute";

	Attribute_Kind : Asis_X.Attribute_Designator_Kinds;

    begin

	declare
	    Arguments : constant Asis.Expression_List := 
	       Asis_X.Attribute_Designator_Arguments (The_Expression);
	    Prefix : Asis.Expression := Asis_X.Prefix (The_Expression);
	begin

	    if Trace.On then
		Trace.Log (The_Expression, The_Context);
		Trace.Add_Level (1 + Arguments'Length);
	    end if;

	    Attribute_Kind := Asis_X.Attribute_Designator_Kind 
				 (Asis_X.Attribute_Designator_Name 
				     (The_Expression));
	    if Asis_X."=" (Attribute_Kind, Asis_X.An_Address_Attribute) or 
	       Asis_X."=" (Attribute_Kind, Asis_X.A_Size_Attribute) then
		Reference_Scan.Scan_Expression 
		   (Prefix, Ctx.Set (The_Context, 
				     Basic_Context => Rvs.An_Attribute, 
				     Weight => 90));
	    else
		Scan_Possible_Access_Subexpression 
		   (Prefix, Ctx.Set (The_Context, 
				     Basic_Context => Rvs.An_Attribute, 
				     Weight => 90));
	    end if;

	    Reference_Scan.Scan_Expression_List 
	       (Arguments, Ctx.Set (The_Context, 
				    Basic_Context => Rvs.A_Read, 
				    Weight => 95, 
				    New_Weight => 5, 
				    Add_Data_Access_Context => 
				       Rvs.An_Attribute_Designator));

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Expression);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Expression);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Expression);
	    raise;

    end Scan_Attribute;

--| +-------------------------------------------------------------------------+
--| | SCAN_CHARACTER_LITERAL (local)                                          |
--| +-------------------------------------------------------------------------+

    procedure Scan_Character_Literal (The_Expression : in Asis.Expression; 
				      The_Context : in Context) is

	Puid : constant String := "Scan_Character_Literal";

    begin

	if Trace.On then
	    Trace.Log (The_Expression, The_Context, Is_Reference => True);
	end if;

	if Rvs."=" (The_Context.Basic_Context, Rvs.A_Read) then
	    Log_Reference (The_Expression, 
			   Ctx.Set (The_Context, 
				    Basic_Context => Rvs.An_Evaluate, 
				    Weight => 95));
	else
	    Log_Reference (The_Expression, The_Context);
	end if;

    end Scan_Character_Literal;

--| +-------------------------------------------------------------------------+
--| | SCAN_DISCRIMINANT_SELECTION (local)                                     |
--| +-------------------------------------------------------------------------+

    procedure Scan_Discriminant_Selection (The_Expression : in Asis.Expression; 
					   The_Context : in Context) is

	Puid : constant String := "Scan_Discriminant_Selection";

    begin

	declare
	    Discriminant : Asis.Expression := Asis_X.Selector (The_Expression);
	    Prefix : Asis.Expression := Asis_X.Prefix (The_Expression);
	begin

	    if Trace.On then
		Trace.Log (The_Expression, The_Context);
		Trace.Add_Level (2);
	    end if;

	    Scan_Possible_Access_Subexpression (Prefix, The_Context);

	    Reference_Scan.Scan_Expression 
	       (Discriminant, 
		Ctx.Set (The_Context, 
			 Basic_Context => Rvs.A_Selected_Component, 
			 Weight => 90));

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Expression);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Expression);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Expression);
	    raise;

    end Scan_Discriminant_Selection;

--| +-------------------------------------------------------------------------+
--| | SCAN_ENUMERATION_LITERAL (local)                                        |
--| +-------------------------------------------------------------------------+

    procedure Scan_Enumeration_Literal (The_Expression : in Asis.Expression; 
					The_Context : in Context) is

	Puid : constant String := "Scan_Enumeration_Literal";

    begin

	if Trace.On then
	    Trace.Log (The_Expression, The_Context, Is_Reference => True);
	end if;

	if Rvs."=" (The_Context.Basic_Context, Rvs.A_Read) then
	    Log_Reference (The_Expression, 
			   Ctx.Set (The_Context, 
				    Basic_Context => Rvs.An_Evaluate, 
				    Weight => 95));
	else
	    Log_Reference (The_Expression, The_Context);
	end if;

    end Scan_Enumeration_Literal;

--| +-------------------------------------------------------------------------+
--| | SCAN_EXPANDED_NAME_SELECTION (local)                                    |
--| +-------------------------------------------------------------------------+

    procedure Scan_Expanded_Name_Selection (The_Expression : in Asis.Expression; 
					    The_Context : in Context) is

	Puid : constant String := "Scan_Expanded_Name_Selection";

    begin

	declare
	    Name : Asis.Expression := Asis_X.Selector (The_Expression);
	    Prefix : Asis.Expression := Asis_X.Prefix (The_Expression);
	begin

	    if Trace.On then
		Trace.Log (The_Expression, The_Context);
		Trace.Add_Level (2);
	    end if;

	    Reference_Scan.Scan_Expression 
	       (Prefix, Ctx.Set (The_Context, 
				 Basic_Context => Rvs.An_Expanded_Name, 
				 Weight => 95));

	    Reference_Scan.Scan_Expression (Name, The_Context);

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Expression);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Expression);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Expression);
	    raise;

    end Scan_Expanded_Name_Selection;

--| +-------------------------------------------------------------------------+
--| | SCAN_FUNCTION_CALL (local)                                              |
--| +-------------------------------------------------------------------------+

    procedure Scan_Function_Call (The_Expression : in Asis.Expression; 
				  The_Context : in Context) is

	Puid : constant String := "Scan_Function_Call";

    begin

	declare
	    Arguments : constant Asis.Association_List := 
	       Asis_X.Function_Call_Parameters 
		  (The_Expression, Reference_Scan.
				      Normalize_Function_Call_Parameters);
	    Prefix : Asis.Expression := Asis_X.Prefix (The_Expression);
	begin

	    if Trace.On then
		Trace.Log (The_Expression, The_Context);
		Trace.Add_Level (1 + Arguments'Length);
	    end if;

	    if not Asis_X.Is_Prefix_Call (The_Expression) and 
	       Arguments'Length = 2 then
		Reference_Scan.Scan_Parameter_Association 
		   (Arguments (1), The_Context);
		Reference_Scan.Scan_Expression 
		   (Prefix, Ctx.Set (The_Context, 
				     Basic_Context => Rvs.A_Function_Call, 
				     Weight => 90));
		Reference_Scan.Scan_Parameter_Association 
		   (Arguments (2), The_Context);
	    else
		Reference_Scan.Scan_Expression 
		   (Prefix, Ctx.Set (The_Context, 
				     Basic_Context => Rvs.A_Function_Call, 
				     Weight => 90));
		Reference_Scan.Scan_Parameter_Association_List 
		   (Arguments, The_Context);
	    end if;

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Expression);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Expression);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Expression);
	    raise;

    end Scan_Function_Call;

--| +-------------------------------------------------------------------------+
--| | SCAN_INDEXED_COMPONENT (local)                                          |
--| +-------------------------------------------------------------------------+

    procedure Scan_Indexed_Component (The_Expression : in Asis.Expression; 
				      The_Context : in Context) is

	Puid : constant String := "Scan_Indexed_Component";

    begin

	declare
	    Indexes : constant Asis.Expression_List := 
	       Asis_X.Index_Expressions (The_Expression);
	    Prefix : Asis.Expression := Asis_X.Prefix (The_Expression);
	begin

	    if Trace.On then
		Trace.Log (The_Expression, The_Context);
		Trace.Add_Level (1 + Indexes'Length);
	    end if;

	    Scan_Possible_Access_Subexpression (Prefix, The_Context);

	    Reference_Scan.Scan_Expression_List 
	       (Indexes, Ctx.Set (The_Context, 
				  Basic_Context => Rvs.A_Read, 
				  Weight => 95, 
				  New_Weight => 5, 
				  Add_Data_Access_Context => 
				     Rvs.An_Indexed_Component));

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Expression);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Expression);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Expression);
	    raise;

    end Scan_Indexed_Component;

--| +-------------------------------------------------------------------------+
--| | SCAN_INTEGER_LITERAL (local)                                            |
--| +-------------------------------------------------------------------------+

    procedure Scan_Integer_Literal (The_Expression : in Asis.Expression; 
				    The_Context : in Context) is

	Puid : constant String := "Scan_Integer_Literal";

    begin

	if Trace.On then
	    Trace.Log (The_Expression, The_Context);
	end if;

    end Scan_Integer_Literal;

--| +-------------------------------------------------------------------------+
--| | SCAN_NULL_LITERAL (local)                                               |
--| +-------------------------------------------------------------------------+

    procedure Scan_Null_Literal (The_Expression : in Asis.Expression; 
				 The_Context : in Context) is

	Puid : constant String := "Scan_Null_Literal";

    begin

	if Trace.On then
	    Trace.Log (The_Expression, The_Context);
	end if;

    end Scan_Null_Literal;

--| +-------------------------------------------------------------------------+
--| | SCAN_OPERATOR_SYMBOL (local)                                            |
--| +-------------------------------------------------------------------------+

    procedure Scan_Operator_Symbol (The_Expression : in Asis.Expression; 
				    The_Context : in Context) is

	Puid : constant String := "Scan_Operator_Symbol";

    begin

	if Trace.On then
	    Trace.Log (The_Expression, The_Context, Is_Reference => True);
	end if;

	Log_Reference (The_Expression, The_Context);

    end Scan_Operator_Symbol;

--| +-------------------------------------------------------------------------+
--| | SCAN_PARENTHESIZED_EXPRESSION (local)                                   |
--| +-------------------------------------------------------------------------+

    procedure Scan_Parenthesized_Expression 
		 (The_Expression : in Asis.Expression; 
		  The_Context : in Context) is

	Puid : constant String := "Scan_Parenthesized_Expression";

    begin

	declare
	    Enclosed_Expression : Asis.Expression := 
	       Asis_X.Expression_Parenthesized (The_Expression);
	begin

	    if Trace.On then
		Trace.Log (The_Expression, The_Context);
		Trace.Add_Level (1);
	    end if;

	    Reference_Scan.Scan_Expression (Enclosed_Expression, The_Context);

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Expression);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Expression);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Expression);
	    raise;

    end Scan_Parenthesized_Expression;

--| +-------------------------------------------------------------------------+
--| | SCAN_POSSIBLE_ACCESS_SUBEXPRESSION (local)                              |
--| +-------------------------------------------------------------------------+
--|
--| Scans a subexpression that is possibly of an access type.  If it is,
--| a reference is logged to the ground type of the access type and
--| the current reference kind is changed to A_Read.

    procedure Scan_Possible_Access_Subexpression 
		 (The_Expression : in Asis.Expression; 
		  The_Context : in Context) is

	Puid : constant String := "Scan_Possible_Access_Subexpression";

	The_Expression_Ground_Type : Asis.Type_Definition;
	The_Expression_Type : Asis.Type_Definition;

    begin

	The_Expression_Type := Asis_X.Expression_Type (The_Expression);

	if Asis_E.Is_Nil (The_Expression_Type) then
	    Reference_Scan.Scan_Expression (The_Expression, The_Context);
	else
	    The_Expression_Ground_Type := 
	       Asis_Td.Ground_Type (The_Expression_Type);
	    if Asis_Td."=" (Asis_Td.Kind (The_Expression_Ground_Type), 
			    Asis_Td.An_Access_Type_Definition) then
		Log_Reference (The_Expression, The_Context, 
			       Is_Collection_Reference => True);
		Reference_Scan.Scan_Expression 
		   (The_Expression, 
		    Ctx.Set (The_Context, 
			     Basic_Context => Rvs.A_Read, 
			     Weight => 95, 
			     New_Weight => 5, 
			     Add_Data_Access_Context => Rvs.A_Dereference));
	    else
		Reference_Scan.Scan_Expression (The_Expression, The_Context);
	    end if;
	end if;

    exception

	when Asis.Asis_Inappropriate_Element | Asis.Asis_Failed =>
	    Error.Semantic_Error ("exception raised", 
				  "starting from.expression type.ground type.", 
				  (The_Expression, The_Expression_Type, 
				   The_Expression_Ground_Type), Cuid, Puid);
	    raise;

    end Scan_Possible_Access_Subexpression;

--| +-------------------------------------------------------------------------+
--| | SCAN_QUALIFIED_EXPRESSION (local)                                       |
--| +-------------------------------------------------------------------------+

    procedure Scan_Qualified_Expression (The_Expression : in Asis.Expression; 
					 The_Context : in Context) is

	Puid : constant String := "Scan_Qualified_Expression";

	New_Context : Context;

    begin

	declare
	    Qualified_Expression : Asis.Expression := 
	       Asis_X.Converted_Or_Qualified_Expression (The_Expression);
	    Type_Mark : Asis.Expression := Asis_X.Type_Mark (The_Expression);
	begin

	    if Trace.On then
		Trace.Log (The_Expression, The_Context);
		Trace.Add_Level (2);
	    end if;

	    if Rvs."=" (The_Context.Type_Mark_Context, 
			Rvs.A_Code_Statement) then
		Reference_Scan.Scan_Expression 
		   (Type_Mark, Ctx.Set (The_Context, 
					Basic_Context => Rvs.A_Type_Mark, 
					Weight => 90));
		New_Context := The_Context;
		New_Context.Type_Mark_Context := Rvs.Unknown;
		Reference_Scan.Scan_Expression 
		   (Qualified_Expression, New_Context);
	    else
		Reference_Scan.Scan_Expression 
		   (Type_Mark, Ctx.Set (The_Context, 
					Basic_Context => Rvs.A_Type_Mark, 
					Weight => 90, 
					Type_Mark_Context => 
					   Rvs.A_Qualified_Expression));
		Reference_Scan.Scan_Expression 
		   (Qualified_Expression, The_Context);
	    end if;

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Expression);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Expression);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Expression);
	    raise;

    end Scan_Qualified_Expression;

--| +-------------------------------------------------------------------------+
--| | SCAN_RANGE_OPERATION (local)                                            |
--| +-------------------------------------------------------------------------+

    procedure Scan_Range_Operation (The_Expression : in Asis.Expression; 
				    The_Context : in Context) is

	Puid : constant String := "Scan_Range_Operation";

    begin

	declare
	    Left_Side : Asis.Expression := 
	       Asis_X.Special_Operation_Left_Hand_Side (The_Expression);
	    Right_Side : Asis.Range_Constraint := 
	       Asis_X.In_Range_Operation_Right_Hand_Side (The_Expression);
	begin

	    if Trace.On then
		Trace.Log (The_Expression, The_Context);
		Trace.Add_Level (2);
	    end if;

	    Reference_Scan.Scan_Expression (Left_Side, The_Context);

	    Reference_Scan.Scan_Constraint (Right_Side, The_Context);

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Expression);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Expression);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Expression);
	    raise;

    end Scan_Range_Operation;

--| +-------------------------------------------------------------------------+
--| | SCAN_REAL_LITERAL (local)                                               |
--| +-------------------------------------------------------------------------+

    procedure Scan_Real_Literal (The_Expression : in Asis.Expression; 
				 The_Context : in Context) is

	Puid : constant String := "Scan_Real_Literal";

    begin

	if Trace.On then
	    Trace.Log (The_Expression, The_Context);
	end if;

    end Scan_Real_Literal;

--| +-------------------------------------------------------------------------+
--| | SCAN_RECORD_COMPONENT_SELECTION (local)                                 |
--| +-------------------------------------------------------------------------+

    procedure Scan_Record_Component_Selection 
		 (The_Expression : in Asis.Expression; 
		  The_Context : in Context) is

	Puid : constant String := "Scan_Record_Component_Selection";

    begin

	declare
	    Component : Asis.Expression := Asis_X.Selector (The_Expression);
	    Prefix : Asis.Expression := Asis_X.Prefix (The_Expression);
	begin

	    if Trace.On then
		Trace.Log (The_Expression, The_Context);
		Trace.Add_Level (2);
	    end if;

	    Scan_Possible_Access_Subexpression (Prefix, The_Context);

	    Reference_Scan.Scan_Expression 
	       (Component, Ctx.Set (The_Context, 
				    Basic_Context => Rvs.A_Selected_Component, 
				    Weight => 90));

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Expression);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Expression);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Expression);
	    raise;

    end Scan_Record_Component_Selection;

--| +-------------------------------------------------------------------------+
--| | SCAN_SHORT_CIRCUIT_OPERATION (local)                                    |
--| +-------------------------------------------------------------------------+

    procedure Scan_Short_Circuit_Operation (The_Expression : in Asis.Expression; 
					    The_Context : in Context) is

	Puid : constant String := "Scan_Short_Circuit_Operation";

    begin

	declare
	    Left_Side : Asis.Expression := 
	       Asis_X.Special_Operation_Left_Hand_Side (The_Expression);
	    Right_Side : Asis.Expression := 
	       Asis_X.Short_Circuit_Operation_Right_Hand_Side (The_Expression);
	begin

	    if Trace.On then
		Trace.Log (The_Expression, The_Context);
		Trace.Add_Level (2);
	    end if;

	    Reference_Scan.Scan_Expression (Left_Side, The_Context);

	    Reference_Scan.Scan_Expression (Right_Side, The_Context);

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Expression);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Expression);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Expression);
	    raise;

    end Scan_Short_Circuit_Operation;

--| +-------------------------------------------------------------------------+
--| | SCAN_SIMPLE_NAME (local)                                                |
--| +-------------------------------------------------------------------------+

    procedure Scan_Simple_Name (The_Expression : in Asis.Expression; 
				The_Context : in Context) is

	Puid : constant String := "Scan_Simple_Name";

	Is_Reference : Boolean := True;

    begin

	if Rvs."=" (The_Context.Basic_Context, Rvs.A_Pragma) then
	    declare
		Simple_Name_Definition : Asis.Entity_Name_Definition;
	    begin
		Simple_Name_Definition := 
		   Asis_X.Name_Definition (The_Expression);
		if Asis_E.Is_Nil (Simple_Name_Definition) then
		    Is_Reference := False;
		end if;
	    exception
		when Asis.Asis_Inappropriate_Element =>
		    Is_Reference := False;
		when Asis.Asis_Failed =>
		    Error.Semantic_Error 
		       ("exception raised", "starting from.definition.", 
			(The_Expression, Simple_Name_Definition), Cuid, Puid);
		    raise;
	    end;
	end if;

	if Trace.On then
	    Trace.Log (The_Expression, The_Context, Is_Reference);
	end if;

	if Is_Reference then
	    Log_Reference (The_Expression, The_Context);
	end if;

    exception

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Expression);
	    raise Traversal_Error;

    end Scan_Simple_Name;

--| +-------------------------------------------------------------------------+
--| | SCAN_SLICE (local)                                                      |
--| +-------------------------------------------------------------------------+

    procedure Scan_Slice (The_Expression : in Asis.Expression; 
			  The_Context : in Context) is

	Puid : constant String := "Scan_Slice";

    begin

	declare
	    Prefix : Asis.Expression := Asis_X.Prefix (The_Expression);
	    The_Range : Asis.Discrete_Range := 
	       Asis_X.Slice_Range (The_Expression);
	begin

	    if Trace.On then
		Trace.Log (The_Expression, The_Context);
		Trace.Add_Level (2);
	    end if;

	    Scan_Possible_Access_Subexpression (Prefix, The_Context);

	    Reference_Scan.Scan_Discrete_Range 
	       (The_Range, Ctx.Set (The_Context, 
				    Type_Mark_Context => Rvs.A_Slice));

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Expression);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Expression);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Expression);
	    raise;

    end Scan_Slice;

--| +-------------------------------------------------------------------------+
--| | SCAN_STRING_LITERAL (local)                                             |
--| +-------------------------------------------------------------------------+

    procedure Scan_String_Literal (The_Expression : in Asis.Expression; 
				   The_Context : in Context) is

	Puid : constant String := "Scan_String_Literal";

    begin

	if Trace.On then
	    Trace.Log (The_Expression, The_Context);
	end if;

    end Scan_String_Literal;

--| +-------------------------------------------------------------------------+
--| | SCAN_TASK_ENTRY_SELECTION (local)                                       |
--| +-------------------------------------------------------------------------+

    procedure Scan_Task_Entry_Selection (The_Expression : in Asis.Expression; 
					 The_Context : in Context) is

	Puid : constant String := "Scan_Task_Entry_Selection";

    begin

	declare
	    Prefix : Asis.Expression := Asis_X.Prefix (The_Expression);
	    The_Entry : Asis.Expression := Asis_X.Selector (The_Expression);
	begin

	    if Trace.On then
		Trace.Log (The_Expression, The_Context);
		Trace.Add_Level (2);
	    end if;

	    Scan_Possible_Access_Subexpression 
	       (Prefix, Ctx.Set (The_Context, 
				 Basic_Context => Rvs.A_Task_Entry_Selection, 
				 Weight => 90));

	    Reference_Scan.Scan_Expression (The_Entry, The_Context);

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Expression);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Expression);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Expression);
	    raise;

    end Scan_Task_Entry_Selection;

--| +-------------------------------------------------------------------------+
--| | SCAN_TYPE_CONVERSION (local)                                            |
--| +-------------------------------------------------------------------------+

    procedure Scan_Type_Conversion (The_Expression : in Asis.Expression; 
				    The_Context : in Context) is

	Puid : constant String := "Scan_Type_Conversion";

    begin

	declare
	    Converted_Expression : Asis.Expression := 
	       Asis_X.Converted_Or_Qualified_Expression (The_Expression);
	    Type_Mark : Asis.Expression := Asis_X.Type_Mark (The_Expression);
	begin

	    if Trace.On then
		Trace.Log (The_Expression, The_Context);
		Trace.Add_Level (2);
	    end if;

	    Reference_Scan.Scan_Expression 
	       (Type_Mark, Ctx.Set 
			      (The_Context, 
			       Basic_Context => Rvs.A_Type_Mark, 
			       Weight => 90, 
			       Type_Mark_Context => Rvs.A_Type_Conversion));

	    Reference_Scan.Scan_Expression (Converted_Expression, The_Context);

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Expression);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Expression);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Expression);
	    raise;

    end Scan_Type_Conversion;

--| +-------------------------------------------------------------------------+
--| | SCAN_TYPE_OPERATION (local)                                             |
--| +-------------------------------------------------------------------------+

    procedure Scan_Type_Operation (The_Expression : in Asis.Expression; 
				   The_Context : in Context) is

	Puid : constant String := "Scan_Type_Operation";

    begin

	declare
	    Left_Side : Asis.Expression := 
	       Asis_X.Special_Operation_Left_Hand_Side (The_Expression);
	    Right_Side : Asis.Expression := 
	       Asis_X.In_Type_Operation_Right_Hand_Side (The_Expression);
	begin

	    if Trace.On then
		Trace.Log (The_Expression, The_Context);
		Trace.Add_Level (2);
	    end if;

	    Reference_Scan.Scan_Expression (Left_Side, The_Context);

	    Reference_Scan.Scan_Expression 
	       (Right_Side, Ctx.Set 
			       (The_Context, 
				Basic_Context => Rvs.A_Type_Mark, 
				Weight => 90, 
				Type_Mark_Context => Rvs.A_Membership_Test));

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Expression);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Expression);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Expression);
	    raise;

    end Scan_Type_Operation;

--| +-------------------------------------------------------------------------+
--| | SCAN_EXPRESSION (exported)                                              |
--| +-------------------------------------------------------------------------+

begin

    case Asis_X.Kind (The_Expression) is
	when Asis_X.A_Simple_Name =>
	    Scan_Simple_Name (The_Expression, The_Context);
	when Asis_X.An_Operator_Symbol =>
	    Scan_Operator_Symbol (The_Expression, The_Context);
	when Asis_X.A_Character_Literal =>
	    Scan_Character_Literal (The_Expression, The_Context);
	when Asis_X.An_Enumeration_Literal =>
	    Scan_Enumeration_Literal (The_Expression, The_Context);
	when Asis_X.An_Indexed_Component =>
	    Scan_Indexed_Component (The_Expression, The_Context);
	when Asis_X.A_Slice =>
	    Scan_Slice (The_Expression, The_Context);
	when Asis_X.A_Selected_Component =>
	    case Asis_X.Selection_Kind (The_Expression) is
		when Asis_X.A_Discriminant =>
		    Scan_Discriminant_Selection (The_Expression, The_Context);
		when Asis_X.A_Record_Component =>
		    Scan_Record_Component_Selection 
		       (The_Expression, The_Context);
		when Asis_X.A_Task_Entry =>
		    Scan_Task_Entry_Selection (The_Expression, The_Context);
		when Asis_X.An_Access_Object =>
		    Scan_Access_Object_Selection (The_Expression, The_Context);
		when Asis_X.An_Expanded_Name =>
		    Scan_Expanded_Name_Selection (The_Expression, The_Context);
		when Asis_X.Not_A_Selection =>
		    Error.Log (Error.An_Unhandled_Case, Cuid, 
			       Puid & "/1", The_Expression);
		    if Reference_Scan.Raise_Exception_On_Unhandled_Case then
			raise Traversal_Error;
		    else
			if Trace.On then
			    Trace.Log (The_Expression, The_Context);
			end if;
		    end if;
	    end case;
	when Asis_X.An_Attribute =>
	    Scan_Attribute (The_Expression, The_Context);
	when Asis_X.A_Type_Conversion =>
	    Scan_Type_Conversion (The_Expression, The_Context);
	when Asis_X.A_Qualified_Expression =>
	    Scan_Qualified_Expression (The_Expression, The_Context);
	when Asis_X.A_Function_Call =>
	    Scan_Function_Call (The_Expression, The_Context);
	when Asis_X.A_Null_Literal =>
	    Scan_Null_Literal (The_Expression, The_Context);
	when Asis_X.A_String_Literal =>
	    Scan_String_Literal (The_Expression, The_Context);
	when Asis_X.An_Integer_Literal =>
	    Scan_Integer_Literal (The_Expression, The_Context);
	when Asis_X.A_Real_Literal =>
	    Scan_Real_Literal (The_Expression, The_Context);
	when Asis_X.An_Aggregate =>
	    Scan_Aggregate (The_Expression, The_Context);
	when Asis_X.A_Parenthesized_Expression =>
	    Scan_Parenthesized_Expression (The_Expression, The_Context);
	when Asis_X.A_Special_Operation =>
	    case Asis_X.Special_Operation_Kind (The_Expression) is
		when Asis_X.An_In_Range =>
		    Scan_Range_Operation (The_Expression, The_Context);
		when Asis_X.A_Not_In_Range =>
		    Scan_Range_Operation (The_Expression, The_Context);
		when Asis_X.An_In_Type =>
		    Scan_Type_Operation (The_Expression, The_Context);
		when Asis_X.A_Not_In_Type =>
		    Scan_Type_Operation (The_Expression, The_Context);
		when Asis_X.An_And_Then =>
		    Scan_Short_Circuit_Operation (The_Expression, The_Context);
		when Asis_X.An_Or_Else =>
		    Scan_Short_Circuit_Operation (The_Expression, The_Context);
		when Asis_X.Not_A_Special_Operation =>
		    Error.Log (Error.An_Unhandled_Case, Cuid, 
			       Puid & "/2", The_Expression);
		    if Reference_Scan.Raise_Exception_On_Unhandled_Case then
			raise Traversal_Error;
		    else
			if Trace.On then
			    Trace.Log (The_Expression, The_Context);
			end if;
		    end if;
	    end case;
	when Asis_X.An_Allocation_From_Subtype =>
	    Scan_Allocation_From_Subtype (The_Expression, The_Context);
	when Asis_X.An_Allocation_From_Qualified_Expression =>
	    Scan_Allocation_From_Qualified_Expression 
	       (The_Expression, The_Context);
	when Asis_X.Not_An_Expression =>
	    Error.Log (Error.An_Unhandled_Case, Cuid, 
		       Puid & "/3", The_Expression);
	    if Reference_Scan.Raise_Exception_On_Unhandled_Case then
		raise Traversal_Error;
	    else
		if Trace.On then
		    Trace.Log (The_Expression, The_Context);
		end if;
	    end if;
    end case;

end Scan_Expression;
