--| +=========================================================================+
--| |                                                                         |
--| | REFERENCE_VIEW - an Asis abstraction                                    |
--| |                                                                         |
--| | This package provides the operations to construct and manage the        |
--| | reference view of a declarative region part.                            |
--| | The reference view is simply the set of all entity references in the    |
--| | region along with some limited context in which the references          |
--| | occur (read, update, type mark, etc.).  References occurring in         |
--| | sub-regions of the region part are not included.                        |
--| |                                                                         |
--| | All subprograms in this package that accept a reference view as         |
--| | input interpret a null view as an empty view.                           |
--| |                                                                         |
--| | Greg Janee                                                              |
--| | General Research Corporation                                            |
--| |                                                                         |
--| +=========================================================================+

with Asis;
with Reference_Scan;
with Reference_View_Structures;
with Region_Support;
with Text_Io;

package Reference_View is

    package Asis_Id_Io renames Asis.Ids.Id_Io;
    package Rvs renames Reference_View_Structures;

--| Construct builds a reference view.  There are two versions:  the first
--| builds a view for an entire compilation unit, the second for a
--| declarative region part.  The resultant view is merged into The_View.
--|
--| Include_Pragmas determines if pragmas are scanned.  Normalize_Associations
--| determines if the normalized forms of associations are scanned (note
--| that the underlying Asis implementation may always normalize
--| associations).  Expand_Instantiations determines if instantiations are
--| expanded in place.  Trace determines if the reference scan is
--| traced and written to Trace_File (if open) or to the default output file
--| (otherwise).

    Trace_File : Text_Io.File_Type renames Reference_Scan.Trace_File;

    procedure Construct (The_View : in out Rvs.Reference_View; 
			 For_Unit : in Asis.Compilation_Unit; 
			 Include_Pragmas : in Boolean := True; 
			 Normalize_Associations : in Boolean := False; 
			 Expand_Instantiations : in Boolean := False; 
			 Trace : in Boolean := False);

    procedure Construct (The_View : in out Rvs.Reference_View; 
			 For_Region : in Region_Support.Region; 
			 Include_Pragmas : in Boolean := True; 
			 Normalize_Associations : in Boolean := False; 
			 Expand_Instantiations : in Boolean := False; 
			 Trace : in Boolean := False);

--| Free frees all memory held by a reference view.  The_View is set to null.

    procedure Free (The_View : in out Rvs.Reference_View);

--| Dump writes a textual representation of a reference view to a file for
--| debugging purposes.

    procedure Dump (The_View : in Rvs.Reference_View; 
		    To_The_File : in Text_Io.File_Type := 
		       Text_Io.Standard_Output);

--| Merge merges two reference views.  References to the same entity are
--| kept in source code order.

    function Merge (The_View : in Rvs.Reference_View; 
		    And_The_View : in Rvs.Reference_View) 
		   return Rvs.Reference_View;

--| Output and Input communicate a reference view to and from a file, thus
--| giving the view persistence.  For Output, To_Control_File and
--| To_Id_File should already have been opened for writing.  Data is
--| written sequentially starting at the current positions in both files.
--| The files are left open.  For Input, From_Control_File and From_Id_File
--| should already have been opened for reading.  From_Control_File is
--| read sequentially starting at the current position.  The input is not
--| checked for errors.  The files are left open.

    procedure Output (The_View : in Rvs.Reference_View; 
		      To_Control_File : in Text_Io.File_Type; 
		      To_Id_File : in Asis_Id_Io.File_Type);

    function Input (From_Control_File : in Text_Io.File_Type; 
		    From_Id_File : in Asis_Id_Io.File_Type; 
		    Library : Asis.Library) return Rvs.Reference_View;

--| The following exception is raised if an Asis problem is encountered.

    Traversal_Error : exception;

end Reference_View;
