--| +=========================================================================+
--| |                                                                         |
--| | BUILD_REGION_VIEW                                                       |
--| |                                                                         |
--| | Usage: build_region_view <library> <library-unit>                       |
--| |                                                                         |
--| | This program is a simple driver for the Region_View package.  It        |
--| | accepts two command line arguments, the name of an ASIS library and the |
--| | name of a library unit contained therein, and it writes a               |
--| | textual representation of the declarative region tree for               |
--| | the library unit and all of its secondary units to standard output.     |
--| |                                                                         |
--| | Chuck Hobin                                                             |
--| | General Research Corporation                                            |
--| |                                                                         |
--| +=========================================================================+

with Asis;
with Command;
with Msg_Log;
with Region_View;
with Region_View_Structures;
with Region_Support;
with Dump_Region_View;
with Text_Io;

procedure Build_Region_View is

--| Standard Asis renames...

    package Asis_Cu renames Asis.Compilation_Units;
    package Asis_D renames Asis.Declarations;
    package Asis_E renames Asis.Elements;
    package Asis_En renames Asis.Environment;
    package Asis_L renames Asis.Libraries;
    package Asis_Str renames Asis.Strings;

    package Cli renames Command;

    Argument_Count : Integer := Cli.Argc - 1;
    Arguments : constant Cli.String_Ptr_Array := Cli.Arguments;
    The_Library : Asis.Library;
    The_Unit : Asis.Compilation_Unit;

    Find_Error : exception;
    Usage_Error : exception;

    The_View : Region_View_Structures.Region_View;

--------------------------------------------------------------------------------

    procedure Pre_Operation (Element : in Asis.Element; 
			     Control : in out Asis.Elements.Traverse_Control; 
			     View : in out Region_View_Structures.Region_View);

    procedure Post_Operation (Element : in Asis.Element; 
			      Control : in out Asis.Elements.Traverse_Control; 
			      View : in out Region_View_Structures.Region_View);

    -- Instantiate Traverse_Element generic ...

    procedure Visit_Regions is 
       new Asis.Elements.Traverse_Element 
	      (State_Information => Region_View_Structures.Region_View, 
	       Pre_Operation => Pre_Operation, 
	       Post_Operation => Post_Operation);

    procedure Pre_Operation 
		 (Element : in Asis.Element; 
		  Control : in out Asis.Elements.Traverse_Control; 
		  View : in out Region_View_Structures.Region_View) is
    begin
	if Region_Support.Denotes_One_Or_More_Regions 
	      (Element, Expand_Instantiations => True) then
	    declare
		Region_List : constant Region_Support.Region_List := 
		   Region_Support.Denoted_Regions 
		      (Element, 
		       Expand_Instantiations => True, 
		       Include_Instance_Bodies => False);
	    begin
		for I in Region_List'Range loop
		    declare
			Subelements : constant Asis.Element_List := 
			   Region_Support.Subelements (Region_List (I));
			Recurse_Control :  
			   Asis.Elements.Traverse_Control;
		    begin
			Region_View.Enter_Region (Region_List (I), View);
			for J in Subelements'Range loop
			    Recurse_Control := Asis.Elements.Continue;
			    Visit_Regions (Element => Subelements (J), 
					   Control => Recurse_Control, 
					   State => View);
			end loop;
			Region_View.Leave_Region (Region_List (I), View);
		    end;
		end loop;
	    end;
	    Control := Asis.Elements.Abandon_Children;
	end if;

    end Pre_Operation;

    procedure Post_Operation 
		 (Element : in Asis.Element; 
		  Control : in out Asis.Elements.Traverse_Control; 
		  View : in out Region_View_Structures.Region_View) is
    begin
	null;
    end Post_Operation;

--------------------------------------------------------------------------------

    procedure Analyze_Unit (The_Unit : in Asis.Compilation_Unit) is
	The_Element : Asis.Element;
	Comp_Unit_Region : Region_Support.Region 
			      (Region_Support.A_Compilation_Unit);
	Control : Asis.Elements.Traverse_Control := Asis.Elements.Continue;
    begin
	Msg_Log.Put_Msg (Msg_Log.Note, 
			 "Analyzing " & Asis_Cu.Name (The_Unit) & 
			    ' ' & Asis_Cu.Compilation_Unit_Kinds'Image 
				     (Asis_Cu.Kind (The_Unit)));
	The_Element := Asis.Compilation_Units.Unit_Declaration (The_Unit);

	Region_View.Start_Comp_Unit (The_Unit, The_View);
	Comp_Unit_Region := Region_Support.Equivalent_Region (The_Unit);
	Region_View.Enter_Region (Comp_Unit_Region, The_View);

	Visit_Regions (The_Element, Control, The_View);

	Region_View.Leave_Region (Comp_Unit_Region, The_View);
	Region_View.Finish_Comp_Unit (The_Unit, The_View);
    end Analyze_Unit;

--------------------------------------------------------------------------------

    procedure Analyze_Subunits_Of_Unit (The_Unit : in Asis.Compilation_Unit) is
	Subunits : constant Asis.Compilation_Unit_List := 
	   Asis_Cu.Subunits (The_Unit);
    begin
	for I in Subunits'Range loop
	    case Asis_Cu.Kind (Subunits (I)) is
		when Asis_Cu.A_Subunit =>
		    Analyze_Unit (Subunits (I));
		    -- Recursively analyze any subunits of this subunit.
		    Analyze_Subunits_Of_Unit (Subunits (I));
		when others =>
		    -- Asis_Cu.Subunits can return non-existent units; we
		    -- ignore them.
		    null;
	    end case;
	end loop;
    end Analyze_Subunits_Of_Unit;

--------------------------------------------------------------------------------

begin -- main program

    Msg_Log.Set_Program ("rv");

    if Argument_Count < 2 then
	raise Usage_Error;
    end if;

    Asis_En.Initialize;
    Asis_L.Associate (The_Library, Asis_Str.To_Asis_String (Arguments (1).all));
    Asis_L.Open (The_Library);

    The_Unit := Asis_Cu.Library_Unit 
		   (Arguments (Arguments'Last).all, The_Library);
    if Asis_Cu.Is_Nil (The_Unit) then
	raise Find_Error;
    end if;

    Region_View.Initialize (The_View);

    Analyze_Unit (The_Unit);

    case Asis_D.Kind (Asis_Cu.Unit_Declaration (The_Unit)) is
	when Asis_D.A_Procedure_Body_Declaration | 
	     Asis_D.A_Function_Body_Declaration =>

	    -- This is the case where a subprogram body is a library
	    -- unit; thus, there is no corresponding secondary unit.
	    -- Analyze any subunits of the body.

	    Analyze_Subunits_Of_Unit (The_Unit);

	when others =>

	    The_Unit := Asis_Cu.Secondary_Unit 
			   (Arguments (Arguments'Last).all, The_Library);
	    if not Asis_Cu.Is_Nil (The_Unit) then
		Analyze_Unit (The_Unit);
		Analyze_Subunits_Of_Unit (The_Unit);
	    end if;
    end case;

    Dump_Region_View.Dump (The_View, Text_Io.Standard_Output);

    Region_View.Free (The_View);

    Asis_L.Close (The_Library);
    Asis_L.Dissociate (The_Library);
    Asis_En.Finalize;

exception

    when Usage_Error =>
	Msg_Log.Put_Msg 
	   (Msg_Log.Error, 
	    "usage is ""build_region_view <library> <library-unit>""");

    when Find_Error =>
	Msg_Log.Put_Msg (Msg_Log.Error, "unit not found");

    when Asis.Asis_Inappropriate_Library =>
	Msg_Log.Put_Msg 
	   (Msg_Log.Error, 
	    "exception Asis_Inappropriate_Library raised; status is " & 
	       Asis_En.Error_Kinds'Image (Asis_En.Status) & 
	       "; diagnosis follows");
	Msg_Log.Put_Msg (Msg_Log.Error, 
			 Asis_Str.To_Standard_String (Asis_En.Diagnosis));

    when Asis.Asis_Failed =>
	Msg_Log.Put_Msg (Msg_Log.Error, 
			 "exception Asis_Failed raised; status is " & 
			    Asis_En.Error_Kinds'Image (Asis_En.Status) & 
			    "; diagnosis follows");
	Msg_Log.Put_Msg (Msg_Log.Error, 
			 Asis_Str.To_Standard_String (Asis_En.Diagnosis));

end Build_Region_View;

