--| +=========================================================================+
--| |                                                                         |
--| | SCAN.SCAN_CHOICE (body)                                                 |
--| |                                                                         |
--| | Greg Janee                                                              |
--| | General Research Corporation                                            |
--| |                                                                         |
--| +=========================================================================+

separate (Scan)
procedure Scan_Choice (The_Choice : in Asis.Choice; 
		       The_Context : in out Context) is

    Cuid : constant String := "Scan.Scan_Choice";
    Puid : constant String := "Scan_Choice";

    package Error renames Scan.Error_Handling_Support;
    package Trace renames Scan.Trace_Support;

--| +-------------------------------------------------------------------------+
--| | SCAN_DISCRETE_RANGE (local)                                             |
--| +-------------------------------------------------------------------------+

    procedure Scan_Discrete_Range (The_Choice : in Asis.Choice; 
				   The_Context : in out Context) is

	Puid : constant String := "Scan_Discrete_Range";

    begin

	declare
	    The_Range : Asis.Discrete_Range := 
	       Asis_Td.Choice_Discrete_Range (The_Choice);
	begin

	    if Trace.On then
		Trace.Log (The_Choice);
		Trace.Add_Level (1);
	    end if;

	    Scan.Scan_Discrete_Range (The_Range, The_Context);

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Choice);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Choice);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Choice);
	    raise;

    end Scan_Discrete_Range;

--| +-------------------------------------------------------------------------+
--| | SCAN_EXCEPTION_NAME (local)                                             |
--| +-------------------------------------------------------------------------+

    procedure Scan_Exception_Name (The_Choice : in Asis.Choice; 
				   The_Context : in out Context) is

	Puid : constant String := "Scan_Exception_Name";

    begin

	declare
	    Name : Asis.Expression := Asis_Td.Choice_Name (The_Choice);
	begin

	    if Trace.On then
		Trace.Log (The_Choice);
		Trace.Add_Level (1);
	    end if;

	    Scan.Scan_Expression (Name, The_Context);

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Choice);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Choice);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Choice);
	    raise;

    end Scan_Exception_Name;

--| +-------------------------------------------------------------------------+
--| | SCAN_OTHERS_CHOICE (local)                                              |
--| +-------------------------------------------------------------------------+

    procedure Scan_Others_Choice (The_Choice : in Asis.Choice; 
				  The_Context : in out Context) is

	Puid : constant String := "Scan_Others_Choice";

    begin

	if Trace.On then
	    Trace.Log (The_Choice);
	end if;

    end Scan_Others_Choice;

--| +-------------------------------------------------------------------------+
--| | SCAN_SIMPLE_EXPRESSION (local)                                          |
--| +-------------------------------------------------------------------------+

    procedure Scan_Simple_Expression 
		 (The_Choice : in Asis.Choice; The_Context : in out Context) is

	Puid : constant String := "Scan_Simple_Expression";

    begin

	declare
	    The_Expression : Asis.Expression := 
	       Asis_Td.Choice_Simple_Expression (The_Choice);
	begin

	    if Trace.On then
		Trace.Log (The_Choice);
		Trace.Add_Level (1);
	    end if;

	    Scan.Scan_Expression (The_Expression, The_Context);

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Choice);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Choice);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Choice);
	    raise;

    end Scan_Simple_Expression;

--| +-------------------------------------------------------------------------+
--| | SCAN_SIMPLE_NAME (local)                                                |
--| +-------------------------------------------------------------------------+

    procedure Scan_Simple_Name (The_Choice : in Asis.Choice; 
				The_Context : in out Context) is

	Puid : constant String := "Scan_Simple_Name";

    begin

	declare
	    Name : Asis.Expression := Asis_Td.Choice_Name (The_Choice);
	begin

	    if Trace.On then
		Trace.Log (The_Choice);
		Trace.Add_Level (1);
	    end if;

	    Scan.Scan_Expression (Name, The_Context);

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Choice);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Choice);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Choice);
	    raise;

    end Scan_Simple_Name;

--| +-------------------------------------------------------------------------+
--| | SCAN_CHOICE (exported)                                                  |
--| +-------------------------------------------------------------------------+

begin

    case Asis_Td.Choice_Kind (The_Choice) is
	when Asis_Td.A_Simple_Expression =>
	    Scan_Simple_Expression (The_Choice, The_Context);
	when Asis_Td.A_Discrete_Range =>
	    Scan_Discrete_Range (The_Choice, The_Context);
	when Asis_Td.An_Others_Choice =>
	    Scan_Others_Choice (The_Choice, The_Context);
	when Asis_Td.A_Simple_Name =>
	    Scan_Simple_Name (The_Choice, The_Context);
	when Asis_Td.An_Exception_Name =>
	    Scan_Exception_Name (The_Choice, The_Context);
	when Asis_Td.Not_A_Choice =>
	    Error.Log (Error.An_Unhandled_Case, Cuid, Puid, The_Choice);
	    if Scan.Raise_Exception_On_Unhandled_Case then
		raise Traversal_Error;
	    else
		if Trace.On then
		    Trace.Log (The_Choice);
		end if;
	    end if;
    end case;

end Scan_Choice;
