-- ============================================================================
-- >>>>>>>>>>>>>>>>>>>>>>>>>> ADA COMPILATION UNIT <<<<<<<<<<<<<<<<<<<<<<<<<<<<
-- ============================================================================
--
-- NAME:        Map_Simple_Noncached_Sequential_Unbounded_Managed_Iterator
--
--              SPECIFICATION
-- 
-- AUTHOR:      Chuck Hobin
--
-- DATE:        16 September 1993
--
--                               CHANGE HISTORY
--
-- MM-DD-YY | Initials | Description
-- ----------------------------------------------------------------------------
-- <include SPR#, if applicable>
-- ============================================================================

with Structure_Exceptions;

generic

    type Domain is private;
    type Ranges is private;
    Number_Of_Buckets : in Positive;
    with function Hash_Of (The_Domain : in Domain) return Positive;

package Map_Simple_Noncached_Sequential_Unbounded_Managed_Iterator is

-- Based on the map structure presented in Booch, "Software Components
-- with Ada", Benjamin-Cummings, 1987, Chapter 9.

    type Map is limited private;

    procedure Copy (From_The_Map : in Map; To_The_Map : in out Map);

    -- Copy the ordered pairs from one map to another map.
    --
    -- Raises : Overflow, if the destination map cannot grow large
    --          enough to hold the source state.

    procedure Clear (The_Map : in out Map);

    -- Remove all the ordered pairs (if any) from the map and make
    -- the map empty.

    procedure Bind (The_Domain : in Domain; 
		    And_The_Range : in Ranges; 
		    In_The_Map : in out Map);

    -- Add an ordered pair consisting of an object of the domain
    -- and an object of the range to the map.
    --
    -- Raises : Overflow, if the destination map cannot grow large
    --          enough to hold the new binding.
    --
    -- Raises : Multiple_Binding, if there already exists a binding
    --          for the given object in the domain.

    procedure Unbind (The_Domain : in Domain; In_The_Map : in out Map);

    -- Remove the ordered pair for a given object of the domain from
    -- the map.
    --
    -- Raises : Domain_Is_Not_Bound, if there does not currently exist
    --          a binding for the given object of the domain.

    function Is_Equal (Left : in Map; Right : in Map) return Boolean;

    -- Return True if the two given maps have the same state.

    function Extent_Of (The_Map : in Map) return Natural;

    -- Return the current number of ordered pairs in the map.

    function Is_Empty (The_Map : in Map) return Boolean;

    -- Return True if the map contains no ordered pairs.

    function Is_Bound 
		(The_Domain : in Domain; In_The_Map : in Map) return Boolean;

    -- Return True if there is an object of the range corresponding
    -- to the given object of the domain in the map.

    function Range_Of 
		(The_Domain : in Domain; In_The_Map : in Map) return Ranges;

    -- Return an object of the range corresponding to the given object
    -- of the domain in the map.
    --
    -- Raises : Domain_Is_Not_Bound, if there does not currently exist
    --          a binding for the given object of the domain.


    Overflow : exception renames Structure_Exceptions.Overflow;
    Domain_Is_Not_Bound : exception;
    Multiple_Binding : exception;


------------------------------------------------------------------------------
-- Passive Iterator

    generic

	with procedure Process (The_Domain : in Domain; 
				The_Range : in Ranges; 
				Continue : out Boolean);

    procedure Iterate (Over_The_Map : in Map);

------------------------------------------------------------------------------
-- Active Iterator

    type Iterator is limited private;

    procedure Initialize (The_Iterator : in out Iterator; 
			  With_The_Map : in Map);

    -- Associate the iterator with the map object and position it
    -- at the first pair in the map.

    function Is_Done (The_Iterator : in Iterator) return Boolean;

    -- Return True if the iterator has visited every pair in the map.

    procedure Value_Of (The_Iterator : in Iterator; 
			The_Domain : out Domain; 
			The_Range : out Ranges);

    -- Return the current pair in the map.
    --
    -- Raises : Iterator_Error, if the iterator has already visited 
    --          every pair in the map.

    procedure Get_Next (The_Iterator : in out Iterator);

    -- Advance the iterator to the next pair in the map.
    --
    -- Raises : Iterator_Error, if the iterator has already visited 
    --          every pair in the map.


    Iterator_Error : exception;

private

    type Node;

    type Structure is access Node;

    type Map is array (Positive range 1 .. Number_Of_Buckets) of Structure;

    type Iterator is
	record
	    The_Map : Map;
	    Map_Index : Positive;
	    Current_Pair : Structure;
	end record;

end Map_Simple_Noncached_Sequential_Unbounded_Managed_Iterator;
