-- ============================================================================
-- >>>>>>>>>>>>>>>>>>>>>>>>>> ADA COMPILATION UNIT <<<<<<<<<<<<<<<<<<<<<<<<<<<<
-- ============================================================================
--
-- NAME:        Stack_Sequential_Unbounded_Managed_Iterator
--
--              SPECIFICATION
-- 
-- AUTHOR:      Chuck Hobin
--
-- DATE:        27 September 1993
--
--                               CHANGE HISTORY
--
-- MM-DD-YY | Initials | Description
-- ----------------------------------------------------------------------------
-- <include SPR#, if applicable>
-- ============================================================================

with Structure_Exceptions;

generic

    type Item is private;

package Stack_Sequential_Unbounded_Managed_Iterator is

-- Based on the stack structure presented in Booch, "Software Components
-- with Ada", Benjamin-Cummings, 1987, Chapter 4.

    type Stack is limited private;

    procedure Copy (From_The_Stack : in Stack; To_The_Stack : in out Stack);

    -- Copy the items from one stack to another stack.
    --
    -- Raises : Overflow, if the destination stack cannot grow large
    --          enough to hold the source state.

    procedure Clear (The_Stack : in out Stack);

    -- Remove all the items (if any) from the top of the stack and make
    -- the stack empty.

    procedure Push (The_Item : in Item; On_The_Stack : in out Stack);

    -- Insert an item on the top of the stack.
    --
    -- Raises : Overflow, if the stack cannot grow large enough to hold
    --          the item.

    procedure Pop (The_Stack : in out Stack);

    -- Remove an item from the top of the stack.
    --
    -- Raises : Underflow, if the stack is already empty.

    function Is_Equal (Left : in Stack; Right : in Stack) return Boolean;

    -- Return True if the two stacks have same state.

    function Depth_Of (The_Stack : in Stack) return Natural;

    -- Return the current number of items on the stack.

    function Is_Empty (The_Stack : in Stack) return Boolean;

    -- Return True if there are no items on the stack.

    function Top_Of (The_Stack : in Stack) return Item;

    -- Return the item on the top of the stack.
    --
    -- Raises : Underflow, if the stack is already empty.


    Underflow : exception;
    Overflow : exception;

------------------------------------------------------------------------------
-- Passive Iterator

    generic

	with procedure Process (The_Item : in Item; Continue : out Boolean);

    procedure Iterate (Over_The_Stack : in Stack);

    -- Visit every item on the stack in order from top to bottom.

------------------------------------------------------------------------------
-- Active Iterator

    type Iterator is limited private;

    procedure Initialize (The_Iterator : in out Iterator; 
			  With_The_Stack : in Stack);

    -- Associate the iterator with the stack object and position it
    -- at the item at the top of the stack.

    function Is_Done (The_Iterator : in Iterator) return Boolean;

    -- Return True if the iterator has visited every item in the stack.

    function Value_Of (The_Iterator : in Iterator) return Item;

    -- Return the item currently being visited by the iterator.
    --
    -- Raises : Iterator_Error, if the iterator has already visited 
    --          every item in the stack.

    procedure Get_Next (The_Iterator : in out Iterator);

    -- Advance the iterator in the top-to-bottom direction to the next 
    -- item on the stack.
    --
    -- Raises : Iterator_Error, if the iterator has already visited 
    --          every item in the stack.


    Iterator_Error : exception renames Structure_Exceptions.Overflow;


private

    type Node;

    type Stack is access Node;

    type Iterator is new Stack;

end Stack_Sequential_Unbounded_Managed_Iterator;
