--| +=========================================================================+
--| |                                                                         |
--| | BUILD_CONTROL_FLOW_VIEW                                                 |
--| |                                                                         |
--| | Usage: build_control_flow_view <library> <secondary-unit>               |
--| |                                                                         |
--| | This program is a simple driver for the Control_Flow_View package.  It  |
--| | accepts two command line arguments, the name of an ASIS library and the |
--| | name of a secondary unit contained therein, and it writes a             |
--| | textual representation of the control flow view for each program        |
--| | unit body in the unit.  The output is directed to the standard output.  |
--| |                                                                         |
--| | Chuck Hobin                                                             |
--| | General Research Corporation                                            |
--| |                                                                         |
--| +=========================================================================+

with Asis;
with Command;
with Msg_Log;
with Control_Flow_Defs;
with Control_Flow_View;
with Text_Io;

procedure Build_Control_Flow_View is

--| Standard Asis renames...

    package Asis_Cu renames Asis.Compilation_Units;
    package Asis_D renames Asis.Declarations;
    package Asis_E renames Asis.Elements;
    package Asis_En renames Asis.Environment;
    package Asis_L renames Asis.Libraries;
    package Asis_Str renames Asis.Strings;
    package Asis_X renames Asis.Expressions;

    package Cli renames Command;

    Argument_Count : Integer := Cli.Argc - 1;
    Arguments : constant Cli.String_Ptr_Array := Cli.Arguments;

    The_Library : Asis.Library;
    The_Unit : Asis.Compilation_Unit;
    The_Element : Asis.Element;

    Find_Error : exception;
    Usage_Error : exception;

    Dummy : Boolean := True;
    Control : Asis.Elements.Traverse_Control := Asis.Elements.Continue;

--------------------------------------------------------------------------------

    procedure Pre_Operation (Element : in Asis.Element; 
			     Control : in out Asis.Elements.Traverse_Control; 
			     Dummy : in out Boolean);

    procedure Post_Operation (Element : in Asis.Element; 
			      Control : in out Asis.Elements.Traverse_Control; 
			      Dummy : in out Boolean);

    -- Instantiate Traverse_Element generic ...

    procedure Visit_Bodies is new Asis.Elements.Traverse_Element 
				     (State_Information => Boolean, -- not used
				      Pre_Operation => Pre_Operation, 
				      Post_Operation => Post_Operation);

    procedure Pre_Operation (Element : in Asis.Element; 
			     Control : in out Asis.Elements.Traverse_Control; 
			     Dummy : in out Boolean) is

	The_View : Control_Flow_Defs.Control_Flow_View;
    begin
	if Asis_E."=" (Asis_E.Element_Kind (Element), Asis_E.A_Declaration) then
	    case Asis.Declarations.Kind (Element) is
		when Asis_D.A_Procedure_Body_Declaration | 
		     Asis_D.A_Function_Body_Declaration | 
		     Asis_D.A_Task_Body_Declaration | 
		     Asis_D.A_Package_Body_Declaration =>

		    Control_Flow_View.Construct (The_View, For_Body => Element);

		    Text_Io.Put_Line 
		       ("----------------------------------------" & 
			"----------------------------------------");
		    Text_Io.Put_Line 
		       ("View of " & Asis_D.Declaration_Kinds'Image 
					(Asis_D.Kind (Element)) & 
			" """ & Asis_X.Name (Asis_D.Names (Element) (1)) & 
			""" at line" & 
			Asis.Line_Number'Image 
			   (Asis.Text.First_Line_Number (Element)));
		    Text_Io.New_Line;

		    Control_Flow_View.Dump (The_View);
		    Control_Flow_View.Free (The_View);

		when others =>
		    null;
	    end case;
	end if;

    end Pre_Operation;

    procedure Post_Operation (Element : in Asis.Element; 
			      Control : in out Asis.Elements.Traverse_Control; 
			      Dummy : in out Boolean) is
    begin
	null;
    end Post_Operation;

--------------------------------------------------------------------------------

begin -- main program

    Msg_Log.Set_Program ("cf");

    if Argument_Count < 2 then
	raise Usage_Error;
    end if;

    Asis_En.Initialize;
    Asis_L.Associate (The_Library, Asis_Str.To_Asis_String (Arguments (1).all));
    Asis_L.Open (The_Library);

    The_Unit := Asis_Cu.Secondary_Unit 
		   (Arguments (Arguments'Last).all, The_Library);
    if Asis_Cu.Is_Nil (The_Unit) then
	raise Find_Error;
    end if;

    The_Element := Asis.Compilation_Units.Unit_Declaration (The_Unit);

    Text_Io.Put_Line ("CONTROL FLOW VIEWS IN UNIT : """ & 
		      Asis_Cu.Name (The_Unit) & '"');
    Text_Io.New_Line;

    Visit_Bodies (The_Element, Control, Dummy);

    Asis_L.Close (The_Library);
    Asis_L.Dissociate (The_Library);
    Asis_En.Finalize;

exception

    when Usage_Error =>
	Msg_Log.Put_Msg 
	   (Msg_Log.Error, 
	    "usage is ""build_control_flow_view <library> <secondary-unit>""");

    when Find_Error =>
	Msg_Log.Put_Msg (Msg_Log.Error, "unit not found");

    when Asis.Asis_Inappropriate_Library =>
	Msg_Log.Put_Msg 
	   (Msg_Log.Error, 
	    "exception Asis_Inappropriate_Library raised; status is " & 
	       Asis_En.Error_Kinds'Image (Asis_En.Status) & 
	       "; diagnosis follows");
	Msg_Log.Put_Msg (Msg_Log.Error, 
			 Asis_Str.To_Standard_String (Asis_En.Diagnosis));

    when Asis.Asis_Failed =>
	Msg_Log.Put_Msg (Msg_Log.Error, 
			 "exception Asis_Failed raised; status is " & 
			    Asis_En.Error_Kinds'Image (Asis_En.Status) & 
			    "; diagnosis follows");
	Msg_Log.Put_Msg (Msg_Log.Error, 
			 Asis_Str.To_Standard_String (Asis_En.Diagnosis));

end Build_Control_Flow_View;


