-- ============================================================================
-- >>>>>>>>>>>>>>>>>>>>>>>>>> ADA COMPILATION UNIT <<<<<<<<<<<<<<<<<<<<<<<<<<<<
-- ============================================================================
--
-- NAME:        Control_Flow_View
--
--              BODY
--
-- AUTHOR:      Pilar Montes, Chuck Hobin
--              General Research Corporation
--
--                            CHANGE 
--
-- MM-DD-YY | Initials | Description
-- ----------------------------------------------------------------------------
-- 
-- ============================================================================

with Control_Flow_Save;
with Control_Flow_Scan;
with Unchecked_Deallocation;

package body Control_Flow_View is

    package Asis_D renames Asis.Declarations;
    package Asis_E renames Asis.Elements;
    package Cf_Defs renames Control_Flow_Defs;
    package Cf_Graph renames Control_Flow_Defs.Control_Graph;


----------------------------------------------------------------------
-- LOCAL SUPROGRAM DECLARATIONS
----------------------------------------------------------------------

    procedure Free is new Unchecked_Deallocation 
			     (Cf_Defs.Edge_Type, Cf_Defs.Edge_Type_Ptr);
    procedure Free is new Unchecked_Deallocation 
			     (Cf_Defs.Item_Type, Cf_Defs.Item_Type_Ptr);
    procedure Free is new Unchecked_Deallocation 
			     (Cf_Defs.Handler_Type, Cf_Defs.Handler_Type_Ptr);  
    procedure Free_View is 
       new Unchecked_Deallocation 
	      (Cf_Defs.View_Type, Cf_Defs.Control_Flow_View);

    procedure Process_Node (The_Vertex : in Cf_Graph.Vertex; 
			    Continue : out Boolean);
    procedure Process_Edge (The_Arc : in Cf_Graph.Arc; Continue : out Boolean);

    procedure Free_Nodes is new Cf_Graph.Iterate_Vertices (Process_Node);
    procedure Free_Edges is new Cf_Graph.Iterate_Arcs (Process_Edge);

------------------------------------------------------------------------
-- EXPORTED SUBPROGRAM BODIES
------------------------------------------------------------------------

    procedure Construct (The_View : in out Control_Flow_Defs.Control_Flow_View; 
			 For_Body : in Asis.Declaration) is
    begin
	if Asis_E."=" (Asis_E.Element_Kind (For_Body), 
		       Asis_E.A_Declaration) then
	    case Asis_D.Kind (For_Body) is
		when Asis_D.A_Procedure_Body_Declaration | 
		     Asis_D.A_Function_Body_Declaration | 
		     Asis_D.A_Task_Body_Declaration | 
		     Asis_D.A_Package_Body_Declaration =>

		    if Cf_Defs."/=" (The_View, null) then
			Free (The_View);
		    end if;

		    The_View := 
		       Control_Flow_Scan.Build_Control_Flow_View (For_Body);

		when others =>
		    raise Inappropriate_Element;
	    end case;
	else
	    raise Inappropriate_Element;
	end if;
    end Construct;

------------------------------------------------------------------------

    procedure Free (The_View : in out Control_Flow_Defs.Control_Flow_View) is

	Puid : constant String := "Free";
	The_Iterator : Cf_Defs.Handler_Set.Iterator;
	The_Handler : Cf_Defs.Handler_Type_Ptr;

    begin
	if Cf_Defs."/=" (The_View, null) then

	    -- Free the Graph structure components.
	    Free_Nodes (Over_The_Graph => The_View.Graph);
	    Free_Edges (Over_The_Graph => The_View.Graph);
	    Cf_Graph.Clear (The_Graph => The_View.Graph);

	    -- Free the set of handlers.
	    Cf_Defs.Handler_Set.Initialize (The_Iterator => The_Iterator, 
					    With_The_Set => The_View.Handlers);
	    while not Cf_Defs.Handler_Set.Is_Done (The_Iterator) loop
		The_Handler := Cf_Defs.Handler_Set.Value_Of (The_Iterator);
		Cf_Defs.Handler_Set.Remove (The_Item => The_Handler, 
					    From_The_Set => The_View.Handlers);
		Free (The_Handler);
		Cf_Defs.Handler_Set.Get_Next (The_Iterator);
	    end loop;

	    Free_View (The_View);
	end if;
    end Free;

------------------------------------------------------------------------

    procedure Dump (The_View : in Control_Flow_Defs.Control_Flow_View; 
		    To_The_File : in Text_Io.File_Type := 
		       Text_Io.Standard_Output) is
    begin
	if Cf_Defs."/=" (The_View, null) then
	    Control_Flow_Save.Save_In_Grl_Format (The_View, To_The_File);
	end if;
    end Dump;

------------------------------------------------------------------------
-- LOCAL SUPROGRAM BODIES
------------------------------------------------------------------------

    procedure Process_Node (The_Vertex : in Cf_Graph.Vertex; 
			    Continue : out Boolean) is

	Puid : constant String := "Process_Node";
	The_Item : Cf_Defs.Item_Type_Ptr;

    begin
	The_Item := Cf_Graph.Item_Of (The_Vertex => The_Vertex);
	Free (The_Item);

	Continue := True;
	return;
    end Process_Node;

------------------------------------------------------------------------

    procedure Process_Edge (The_Arc : in Cf_Graph.Arc; 
			    Continue : out Boolean) is

	Puid : constant String := "Process_Edge";
	The_Attribute : Cf_Defs.Edge_Type_Ptr;

    begin

	The_Attribute := Cf_Graph.Attribute_Of (The_Arc => The_Arc);
	Free (The_Attribute);
	Continue := True;
	return;
    end Process_Edge;

end Control_Flow_View;
