--| +=========================================================================+
--| |                                                                         |
--| | SCAN_BY_REGION                                                          |
--| |                                                                         |
--| | Usage: build_namespace_views <library> [-l|-s] <unit>                   |
--| |                                                                         |
--| | This program accepts two command line arguments, the name of an ASIS    |
--| | library and the name of a compilation unit contained therein, and it    |
--| | writes a dump of each region part's namespace view to standard output.  |
--| |                                                                         |
--| | The unit is looked up primarily as a library unit and secondarily       |
--| | as a secondary unit.  To force looking up a library unit, specify the   |
--| | "-l" option.  To force looking up a secondary unit, specify the "-s"    |
--| | option.                                                                 |
--| |                                                                         |
--| | Greg Janee                                                              |
--| | General Research Corporation                                            |
--| |                                                                         |
--| +=========================================================================+

with Asis;
with Asis_Debug_Support;
with Command;
with Msg_Log;
with Namespace_View;
with Region_Scan;
with Region_Support;
with Text_Io;

procedure Build_Namespace_Views is

--| Standard Asis renames...

    package Asis_Cu renames Asis.Compilation_Units;
    package Asis_E renames Asis.Elements;
    package Asis_En renames Asis.Environment;
    package Asis_L renames Asis.Libraries;
    package Asis_Str renames Asis.Strings;
    package Asis_Txt renames Asis.Text;

    package Cli renames Command;

    package Asis_Int_Io is new Text_Io.Integer_Io (Asis.Asis_Integer);

    Argument_Count : Integer := Cli.Argc - 1;
    Arguments : constant Cli.String_Ptr_Array := Cli.Arguments;
    Examine_Library_Units_Only : Boolean := False;
    Examine_Secondary_Units_Only : Boolean := False;
    The_Library : Asis.Library;
    The_Unit : Asis.Compilation_Unit;

    Find_Error : exception;
    Usage_Error : exception;


    Dummy_State : Boolean;

    procedure Pre_Operation (Region : in Region_Support.Region; 
			     Level : in Natural; 
			     Control : out Asis_E.Traverse_Control; 
			     State : in out Boolean) is
	The_View : Namespace_View.Namespace_View;
	Head_Element : Asis.Element;
	Span : Asis.Span;
    begin
	Namespace_View.Construct (The_View, Region);
	Text_Io.New_Line;
	Text_Io.Put_Line ("----------------------------------------");
	Text_Io.Put_Line ("View for Region Part of Kind : " & 
			  Region_Support.Region_Kinds'Image 
			     (Region_Support.Kind (Region)));
	case Region_Support.Kind (Region) is
	    when Region_Support.A_Compilation_Unit =>
		null;
	    when Region_Support.A_Record_Type_Declaration .. 
		    Region_Support.A_Record_Representation_Clause =>
		Text_Io.Put_Line ("  Region Part Head Element :");
		Head_Element := Region_Support.Head_Element (Region);
		Span := Asis_Txt.Element_Span (Head_Element);
		Text_Io.Put ("   ");
		Asis_Int_Io.Put (Span.First_Line, 3);
		Text_Io.Put (" ");
		Asis_Int_Io.Put (Span.First_Column, 3);
		Text_Io.Put_Line 
		   (" " & Asis_Debug_Support.Element_Image (Head_Element));
	end case;
	Text_Io.New_Line;
	Text_Io.Put_Line ("Entity names defined in region part:");
	Namespace_View.Dump (The_View);
	Namespace_View.Free (The_View);
	Control := Asis_E.Continue;
    end Pre_Operation;

    procedure Post_Operation (Region : in Region_Support.Region; 
			      Level : in Natural; 
			      Control : out Asis_E.Traverse_Control; 
			      State : in out Boolean) is
    begin
	Control := Asis_E.Continue;
    end Post_Operation;

    package The_Region_Scan is new Region_Scan 
				      (Boolean, Pre_Operation, Post_Operation);

begin

    Msg_Log.Set_Program ("ns");

    if Argument_Count = 3 then
	if Arguments (2).all = "-l" then
	    Examine_Library_Units_Only := True;
	elsif Arguments (2).all = "-s" then
	    Examine_Secondary_Units_Only := True;
	else
	    raise Usage_Error;
	end if;
    elsif Argument_Count /= 2 then
	raise Usage_Error;
    end if;

    Asis_En.Initialize;
    Asis_L.Associate (The_Library, Asis_Str.To_Asis_String (Arguments (1).all));
    Asis_L.Open (The_Library);

    if Examine_Library_Units_Only then
	The_Unit := Asis_Cu.Library_Unit 
		       (Arguments (Arguments'Last).all, The_Library);
	if Asis_Cu.Is_Nil (The_Unit) then
	    raise Find_Error;
	end if;
    elsif Examine_Secondary_Units_Only then
	The_Unit := Asis_Cu.Secondary_Unit 
		       (Arguments (Arguments'Last).all, The_Library);
	if Asis_Cu.Is_Nil (The_Unit) then
	    raise Find_Error;
	end if;
    else
	The_Unit := Asis_Cu.Library_Unit 
		       (Arguments (Arguments'Last).all, The_Library);
	if Asis_Cu.Is_Nil (The_Unit) then
	    The_Unit := Asis_Cu.Secondary_Unit 
			   (Arguments (Arguments'Last).all, The_Library);
	    if Asis_Cu.Is_Nil (The_Unit) then
		raise Find_Error;
	    end if;
	end if;
    end if;

    The_Region_Scan.Traverse_Unit (The_Unit, Dummy_State);


    Asis_L.Close (The_Library);
    Asis_L.Dissociate (The_Library);
    Asis_En.Finalize;

exception

    when Usage_Error =>
	Msg_Log.Put_Msg 
	   (Msg_Log.Error, 
	    "usage is ""build_namespace_view <library> [-l|-s] " & "<unit>""");

    when Find_Error =>
	Msg_Log.Put_Msg (Msg_Log.Error, "unit not found");

    when Asis.Asis_Inappropriate_Library =>
	Msg_Log.Put_Msg 
	   (Msg_Log.Error, 
	    "exception Asis_Inappropriate_Library raised; status is " & 
	       Asis_En.Error_Kinds'Image (Asis_En.Status) & 
	       "; diagnosis follows");
	Msg_Log.Put_Msg (Msg_Log.Error, 
			 Asis_Str.To_Standard_String (Asis_En.Diagnosis));

    when Asis.Asis_Failed =>
	Msg_Log.Put_Msg (Msg_Log.Error, 
			 "exception Asis_Failed raised; status is " & 
			    Asis_En.Error_Kinds'Image (Asis_En.Status) & 
			    "; diagnosis follows");
	Msg_Log.Put_Msg (Msg_Log.Error, 
			 Asis_Str.To_Standard_String (Asis_En.Diagnosis));

    when The_Region_Scan.Traversal_Error | Namespace_View.Traversal_Error =>
	Msg_Log.Put_Msg (Msg_Log.Error, "exception Traversal_Error raised");

end Build_Namespace_Views;

