--| +=========================================================================+
--| |                                                                         |
--| | NAMESPACE_SCAN.SCAN_STATEMENT (body)                                    |
--| |                                                                         |
--| | Greg Janee                                                              |
--| | General Research Corporation                                            |
--| |                                                                         |
--| +=========================================================================+

separate (Namespace_Scan)
procedure Scan_Statement (The_Statement : in Asis.Statement; 
			  The_Context : in Context) is

    Cuid : constant String := "Namespace_Scan.Scan_Statement";
    Puid : constant String := "Scan_Statement";

    package Error renames Namespace_Scan.Error_Handling_Support;
    package Trace renames Namespace_Scan.Trace_Support;

--| +-------------------------------------------------------------------------+
--| | SCAN_ABORT_STATEMENT (local)                                            |
--| +-------------------------------------------------------------------------+

    procedure Scan_Abort_Statement (The_Statement : in Asis.Statement; 
				    The_Context : in Context) is

	Puid : constant String := "Scan_Abort_Statement";

    begin

	declare
	    Labels : constant Asis.Entity_Name_Definition_List := 
	       Asis_S.Label_Names (The_Statement);
	    Tasks : constant Asis.Expression_List := 
	       Asis_S.Aborted_Tasks (The_Statement);
	begin

	    if Trace.On then
		Trace.Log (The_Statement);
		Trace.Add_Level (Labels'Length + Tasks'Length);
	    end if;

	    Namespace_Scan.Scan_Entity_Name_Definition_List 
	       (Labels, The_Statement);

	    Namespace_Scan.Scan_Expression_List (Tasks, The_Context);

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Statement);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Statement);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Statement);
	    raise;

    end Scan_Abort_Statement;

--| +-------------------------------------------------------------------------+
--| | SCAN_ACCEPT_STATEMENT (local)                                           |
--| +-------------------------------------------------------------------------+

    procedure Scan_Accept_Statement (The_Statement : in Asis.Statement; 
				     The_Context : in Context) is

	Puid : constant String := "Scan_Accept_Statement";

    begin

	declare
	    Entry_Name : Asis.Simple_Name := 
	       Asis_S.Accept_Entry_Simple_Name (The_Statement);
	    Index : Asis.Expression := Asis_S.Family_Index (The_Statement);
	    Labels : constant Asis.Entity_Name_Definition_List := 
	       Asis_S.Label_Names (The_Statement);
	    Parameters : constant Asis.Parameter_Specification_List := 
	       Asis_S.Accept_Parameters (The_Statement);
	    Statements : constant Asis.Statement_List := 
	       Asis_S.Accept_Body_Statements 
		  (The_Statement, Namespace_Scan.
				     Include_Accept_Statement_Pragmas);
	begin

	    if Trace.On then
		Trace.Log (The_Statement);
		Trace.Add_Level (Labels'Length + 1 + 
				 Namespace_Scan.One_If_Present (Index) + 
				 Parameters'Length + 
				 Statements'Length);
	    end if;

	    Namespace_Scan.Scan_Entity_Name_Definition_List 
	       (Labels, The_Statement);

	    Namespace_Scan.Scan_Expression (Entry_Name, The_Context);

	    if not Asis_E.Is_Nil (Index) then
		Namespace_Scan.Scan_Expression (Index, The_Context);
	    end if;

	    Namespace_Scan.Scan_Declaration_List (Parameters, The_Context);

	    Namespace_Scan.Scan_Statement_List (Statements, The_Context);

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Statement);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Statement);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Statement);
	    raise;

    end Scan_Accept_Statement;

--| +-------------------------------------------------------------------------+
--| | SCAN_ASSIGNMENT_STATEMENT (local)                                       |
--| +-------------------------------------------------------------------------+

    procedure Scan_Assignment_Statement (The_Statement : in Asis.Statement; 
					 The_Context : in Context) is

	Puid : constant String := "Scan_Assignment_Statement";

    begin

	declare
	    Labels : constant Asis.Entity_Name_Definition_List := 
	       Asis_S.Label_Names (The_Statement);
	    Left_Side : Asis.Expression := 
	       Asis_S.Object_Assigned_To (The_Statement);
	    Right_Side : Asis.Expression := 
	       Asis_S.Assignment_Expression (The_Statement);
	begin

	    if Trace.On then
		Trace.Log (The_Statement);
		Trace.Add_Level (Labels'Length + 2);
	    end if;

	    Namespace_Scan.Scan_Entity_Name_Definition_List 
	       (Labels, The_Statement);

	    Namespace_Scan.Scan_Expression (Left_Side, The_Context);

	    Namespace_Scan.Scan_Expression (Right_Side, The_Context);

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Statement);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Statement);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Statement);
	    raise;

    end Scan_Assignment_Statement;

--| +-------------------------------------------------------------------------+
--| | SCAN_BLOCK_STATEMENT (local)                                            |
--| +-------------------------------------------------------------------------+

    procedure Scan_Block_Statement (The_Statement : in Asis.Statement; 
				    The_Context : in Context) is

	Puid : constant String := "Scan_Block_Statement";

    begin

	declare
	    Block_Name : Asis.Entity_Name_Definition := 
	       Asis_S.Block_Simple_Name (The_Statement);
	    Declarations : constant Asis.Declarative_Item_List := 
	       Asis_S.Declarative_Items 
		  (The_Statement, Namespace_Scan.
				     Include_Block_Statement_Pragmas);
	    Handlers : constant Asis.Exception_Handler_List := 
	       Asis_S.Block_Exception_Handlers (The_Statement);
	    Labels : constant Asis.Entity_Name_Definition_List := 
	       Asis_S.Label_Names (The_Statement);
	    Statements : constant Asis.Statement_List := 
	       Asis_S.Block_Body_Statements 
		  (The_Statement, Namespace_Scan.
				     Include_Block_Statement_Pragmas);
	begin

	    if Trace.On then
		Trace.Log (The_Statement);
		Trace.Add_Level (Labels'Length + 
				 Namespace_Scan.One_If_Present (Block_Name) + 
				 Declarations'Length + 
				 Statements'Length + 
				 Handlers'Length);
	    end if;

	    Namespace_Scan.Scan_Entity_Name_Definition_List 
	       (Labels, The_Statement);

	    if not Asis_E.Is_Nil (Block_Name) then
		Namespace_Scan.Scan_Entity_Name_Definition 
		   (Block_Name, The_Statement);
	    end if;

	    Namespace_Scan.Scan_Declarative_Item_List 
	       (Declarations, The_Context);

	    Namespace_Scan.Scan_Statement_List (Statements, The_Context);

	    Namespace_Scan.Scan_Exception_Handler_List (Handlers, The_Context);

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Statement);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Statement);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Statement);
	    raise;

    end Scan_Block_Statement;

--| +-------------------------------------------------------------------------+
--| | SCAN_CASE_STATEMENT (local)                                             |
--| +-------------------------------------------------------------------------+

    procedure Scan_Case_Statement (The_Statement : in Asis.Statement; 
				   The_Context : in Context) is

	Puid : constant String := "Scan_Case_Statement";

    begin

	declare
	    Alternatives : constant Asis.Case_Statement_Alternative_List := 
	       Asis_S.Case_Statement_Alternatives 
		  (The_Statement, Namespace_Scan.
				     Include_Case_Statement_Pragmas);
	    Case_Expression : Asis.Expression := 
	       Asis_S.Case_Expression (The_Statement);
	    Labels : constant Asis.Entity_Name_Definition_List := 
	       Asis_S.Label_Names (The_Statement);
	begin

	    if Trace.On then
		Trace.Log (The_Statement);
		Trace.Add_Level (Labels'Length + 1 + Alternatives'Length);
	    end if;

	    Namespace_Scan.Scan_Entity_Name_Definition_List 
	       (Labels, The_Statement);

	    Namespace_Scan.Scan_Expression (Case_Expression, The_Context);

	    Namespace_Scan.Scan_Case_Statement_Alternative_List 
	       (Alternatives, The_Context);

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Statement);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Statement);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Statement);
	    raise;

    end Scan_Case_Statement;

--| +-------------------------------------------------------------------------+
--| | SCAN_CODE_STATEMENT (local)                                             |
--| +-------------------------------------------------------------------------+

    procedure Scan_Code_Statement (The_Statement : in Asis.Statement; 
				   The_Context : in Context) is

	Puid : constant String := "Scan_Code_Statement";

    begin

	declare
	    Aggregate : Asis.Expression := 
	       Asis_S.Qualified_Expression (The_Statement);
	    Labels : constant Asis.Entity_Name_Definition_List := 
	       Asis_S.Label_Names (The_Statement);
	begin

	    if Trace.On then
		Trace.Log (The_Statement);
		Trace.Add_Level (Labels'Length + 1);
	    end if;

	    Namespace_Scan.Scan_Entity_Name_Definition_List 
	       (Labels, The_Statement);

	    Namespace_Scan.Scan_Expression (Aggregate, The_Context);

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Statement);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Statement);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Statement);
	    raise;

    end Scan_Code_Statement;

--| +-------------------------------------------------------------------------+
--| | SCAN_CONDITIONAL_ENTRY_CALL_STATEMENT (local)                           |
--| +-------------------------------------------------------------------------+

    procedure Scan_Conditional_Entry_Call_Statement 
		 (The_Statement : in Asis.Statement; 
		  The_Context : in Context) is

	Puid : constant String := "Scan_Conditional_Entry_Call_Statement";

    begin

	declare
	    Arms : constant Asis.Select_Statement_Arm_List := 
	       Asis_S.Select_Statement_Arms (The_Statement);
	    Labels : constant Asis.Entity_Name_Definition_List := 
	       Asis_S.Label_Names (The_Statement);
	begin

	    if Trace.On then
		Trace.Log (The_Statement);
		Trace.Add_Level (Labels'Length + Arms'Length);
	    end if;

	    Namespace_Scan.Scan_Entity_Name_Definition_List 
	       (Labels, The_Statement);

	    Namespace_Scan.Scan_Select_Statement_Arm_List (Arms, The_Context);

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Statement);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Statement);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Statement);
	    raise;

    end Scan_Conditional_Entry_Call_Statement;

--| +-------------------------------------------------------------------------+
--| | SCAN_DELAY_STATEMENT (local)                                            |
--| +-------------------------------------------------------------------------+

    procedure Scan_Delay_Statement (The_Statement : in Asis.Statement; 
				    The_Context : in Context) is

	Puid : constant String := "Scan_Delay_Statement";

    begin

	declare
	    Amount : Asis.Expression := Asis_S.Delay_Expression (The_Statement);
	    Labels : constant Asis.Entity_Name_Definition_List := 
	       Asis_S.Label_Names (The_Statement);
	begin

	    if Trace.On then
		Trace.Log (The_Statement);
		Trace.Add_Level (Labels'Length + 1);
	    end if;

	    Namespace_Scan.Scan_Entity_Name_Definition_List 
	       (Labels, The_Statement);

	    Namespace_Scan.Scan_Expression (Amount, The_Context);

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Statement);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Statement);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Statement);
	    raise;

    end Scan_Delay_Statement;

--| +-------------------------------------------------------------------------+
--| | SCAN_ENTRY_CALL_STATEMENT (local)                                       |
--| +-------------------------------------------------------------------------+

    procedure Scan_Entry_Call_Statement (The_Statement : in Asis.Statement; 
					 The_Context : in Context) is

	Puid : constant String := "Scan_Entry_Call_Statement";

    begin

	declare
	    Arguments : constant Asis.Association_List := 
	       Asis_S.Call_Parameters 
		  (The_Statement, Namespace_Scan.
				     Normalize_Entry_Call_Parameters);
	    Labels : constant Asis.Entity_Name_Definition_List := 
	       Asis_S.Label_Names (The_Statement);
	    Name : Asis.Expression := Asis_S.Called_Name (The_Statement);
	begin

	    if Trace.On then
		Trace.Log (The_Statement);
		Trace.Add_Level (Labels'Length + 1 + Arguments'Length);
	    end if;

	    Namespace_Scan.Scan_Entity_Name_Definition_List 
	       (Labels, The_Statement);

	    Namespace_Scan.Scan_Expression (Name, The_Context);

	    Namespace_Scan.Scan_Parameter_Association_List 
	       (Arguments, The_Context);

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Statement);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Statement);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Statement);
	    raise;

    end Scan_Entry_Call_Statement;

--| +-------------------------------------------------------------------------+
--| | SCAN_EXIT_STATEMENT (local)                                             |
--| +-------------------------------------------------------------------------+

    procedure Scan_Exit_Statement (The_Statement : in Asis.Statement; 
				   The_Context : in Context) is

	Puid : constant String := "Scan_Exit_Statement";

    begin

	declare
	    Condition : Asis.Expression := 
	       Asis_S.Exit_Condition (The_Statement);
	    Labels : constant Asis.Entity_Name_Definition_List := 
	       Asis_S.Label_Names (The_Statement);
	    Loop_Name : Asis.Expression := 
	       Asis_S.Exit_Loop_Name (The_Statement);
	begin

	    if Trace.On then
		Trace.Log (The_Statement);
		Trace.Add_Level (Labels'Length + 
				 Namespace_Scan.One_If_Present (Loop_Name) + 
				 Namespace_Scan.One_If_Present (Condition));
	    end if;

	    Namespace_Scan.Scan_Entity_Name_Definition_List 
	       (Labels, The_Statement);

	    if not Asis_E.Is_Nil (Loop_Name) then
		Namespace_Scan.Scan_Expression (Loop_Name, The_Context);
	    end if;

	    if not Asis_E.Is_Nil (Condition) then
		Namespace_Scan.Scan_Expression (Condition, The_Context);
	    end if;

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Statement);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Statement);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Statement);
	    raise;

    end Scan_Exit_Statement;

--| +-------------------------------------------------------------------------+
--| | SCAN_FOR_LOOP_STATEMENT (local)                                         |
--| +-------------------------------------------------------------------------+

    procedure Scan_For_Loop_Statement (The_Statement : in Asis.Statement; 
				       The_Context : in Context) is

	Puid : constant String := "Scan_For_Loop_Statement";

    begin

	declare
	    Labels : constant Asis.Entity_Name_Definition_List := 
	       Asis_S.Label_Names (The_Statement);
	    Loop_Name : Asis.Entity_Name_Definition := 
	       Asis_S.Loop_Simple_Name (The_Statement);
	    Scheme : Asis.Declaration := 
	       Asis_S.For_Loop_Parameter_Specification (The_Statement);
	    Statements : constant Asis.Statement_List := 
	       Asis_S.Loop_Statements 
		  (The_Statement, Namespace_Scan.
				     Include_Loop_Statement_Pragmas);
	begin

	    if Trace.On then
		Trace.Log (The_Statement);
		Trace.Add_Level (Labels'Length + 
				 Namespace_Scan.One_If_Present (Loop_Name) + 1 + 
				 Statements'Length);
	    end if;

	    Namespace_Scan.Scan_Entity_Name_Definition_List 
	       (Labels, The_Statement);

	    if not Asis_E.Is_Nil (Loop_Name) then
		Namespace_Scan.Scan_Entity_Name_Definition 
		   (Loop_Name, The_Statement);
	    end if;

	    Namespace_Scan.Scan_Declaration (Scheme, The_Context);

	    Namespace_Scan.Scan_Statement_List (Statements, The_Context);

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Statement);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Statement);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Statement);
	    raise;

    end Scan_For_Loop_Statement;

--| +-------------------------------------------------------------------------+
--| | SCAN_GOTO_STATEMENT (local)                                             |
--| +-------------------------------------------------------------------------+

    procedure Scan_Goto_Statement (The_Statement : in Asis.Statement; 
				   The_Context : in Context) is

	Puid : constant String := "Scan_Goto_Statement";

    begin

	declare
	    Destination_Label : Asis.Expression := 
	       Asis_S.Goto_Label (The_Statement);
	    Labels : constant Asis.Entity_Name_Definition_List := 
	       Asis_S.Label_Names (The_Statement);
	begin

	    if Trace.On then
		Trace.Log (The_Statement);
		Trace.Add_Level (Labels'Length + 1);
	    end if;

	    Namespace_Scan.Scan_Entity_Name_Definition_List 
	       (Labels, The_Statement);

	    Namespace_Scan.Scan_Expression (Destination_Label, The_Context);

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Statement);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Statement);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Statement);
	    raise;

    end Scan_Goto_Statement;

--| +-------------------------------------------------------------------------+
--| | SCAN_IF_STATEMENT (local)                                               |
--| +-------------------------------------------------------------------------+

    procedure Scan_If_Statement (The_Statement : in Asis.Statement; 
				 The_Context : in Context) is

	Puid : constant String := "Scan_If_Statement";

    begin

	declare
	    Arms : constant Asis.If_Statement_Arm_List := 
	       Asis_S.If_Statement_Arms (The_Statement);
	    Labels : constant Asis.Entity_Name_Definition_List := 
	       Asis_S.Label_Names (The_Statement);
	begin

	    if Trace.On then
		Trace.Log (The_Statement);
		Trace.Add_Level (Labels'Length + Arms'Length);
	    end if;

	    Namespace_Scan.Scan_Entity_Name_Definition_List 
	       (Labels, The_Statement);

	    Namespace_Scan.Scan_If_Statement_Arm_List (Arms, The_Context);

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Statement);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Statement);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Statement);
	    raise;

    end Scan_If_Statement;

--| +-------------------------------------------------------------------------+
--| | SCAN_NULL_STATEMENT (local)                                             |
--| +-------------------------------------------------------------------------+

    procedure Scan_Null_Statement (The_Statement : in Asis.Statement; 
				   The_Context : in Context) is

	Puid : constant String := "Scan_Null_Statement";

    begin

	declare
	    Labels : constant Asis.Entity_Name_Definition_List := 
	       Asis_S.Label_Names (The_Statement);
	begin

	    if Trace.On then
		Trace.Log (The_Statement);
		Trace.Add_Level (Labels'Length);
	    end if;

	    Namespace_Scan.Scan_Entity_Name_Definition_List 
	       (Labels, The_Statement);

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Statement);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Statement);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Statement);
	    raise;

    end Scan_Null_Statement;

--| +-------------------------------------------------------------------------+
--| | SCAN_PROCEDURE_CALL_STATEMENT (local)                                   |
--| +-------------------------------------------------------------------------+

    procedure Scan_Procedure_Call_Statement (The_Statement : in Asis.Statement; 
					     The_Context : in Context) is

	Puid : constant String := "Scan_Procedure_Call_Statement";

    begin

	declare
	    Arguments : constant Asis.Association_List := 
	       Asis_S.Call_Parameters 
		  (The_Statement, Namespace_Scan.
				     Normalize_Procedure_Call_Parameters);
	    Labels : constant Asis.Entity_Name_Definition_List := 
	       Asis_S.Label_Names (The_Statement);
	    Name : Asis.Expression := Asis_S.Called_Name (The_Statement);
	begin

	    if Trace.On then
		Trace.Log (The_Statement);
		Trace.Add_Level (Labels'Length + 1 + Arguments'Length);
	    end if;

	    Namespace_Scan.Scan_Entity_Name_Definition_List 
	       (Labels, The_Statement);

	    Namespace_Scan.Scan_Expression (Name, The_Context);

	    Namespace_Scan.Scan_Parameter_Association_List 
	       (Arguments, The_Context);

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Statement);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Statement);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Statement);
	    raise;

    end Scan_Procedure_Call_Statement;

--| +-------------------------------------------------------------------------+
--| | SCAN_RAISE_STATEMENT (local)                                            |
--| +-------------------------------------------------------------------------+

    procedure Scan_Raise_Statement (The_Statement : in Asis.Statement; 
				    The_Context : in Context) is

	Puid : constant String := "Scan_Raise_Statement";

    begin

	declare
	    Exception_Name : Asis.Expression := 
	       Asis_S.Raised_Exception (The_Statement);
	    Labels : constant Asis.Entity_Name_Definition_List := 
	       Asis_S.Label_Names (The_Statement);
	begin

	    if Trace.On then
		Trace.Log (The_Statement);
		Trace.Add_Level (Labels'Length + Namespace_Scan.One_If_Present 
						    (Exception_Name));
	    end if;

	    Namespace_Scan.Scan_Entity_Name_Definition_List 
	       (Labels, The_Statement);

	    if not Asis_E.Is_Nil (Exception_Name) then
		Namespace_Scan.Scan_Expression (Exception_Name, The_Context);
	    end if;

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Statement);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Statement);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Statement);
	    raise;

    end Scan_Raise_Statement;

--| +-------------------------------------------------------------------------+
--| | SCAN_RETURN_STATEMENT (local)                                           |
--| +-------------------------------------------------------------------------+

    procedure Scan_Return_Statement (The_Statement : in Asis.Statement; 
				     The_Context : in Context) is

	Puid : constant String := "Scan_Return_Statement";

    begin

	declare
	    Labels : constant Asis.Entity_Name_Definition_List := 
	       Asis_S.Label_Names (The_Statement);
	    Return_Expression : Asis.Expression := 
	       Asis_S.Return_Expression (The_Statement);
	begin

	    if Trace.On then
		Trace.Log (The_Statement);
		Trace.Add_Level (Labels'Length + Namespace_Scan.One_If_Present 
						    (Return_Expression));
	    end if;

	    Namespace_Scan.Scan_Entity_Name_Definition_List 
	       (Labels, The_Statement);

	    if not Asis_E.Is_Nil (Return_Expression) then
		Namespace_Scan.Scan_Expression (Return_Expression, The_Context);
	    end if;

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Statement);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Statement);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Statement);
	    raise;

    end Scan_Return_Statement;

--| +-------------------------------------------------------------------------+
--| | SCAN_SELECTIVE_WAIT_STATEMENT (local)                                   |
--| +-------------------------------------------------------------------------+

    procedure Scan_Selective_Wait_Statement (The_Statement : in Asis.Statement; 
					     The_Context : in Context) is

	Puid : constant String := "Scan_Selective_Wait_Statement";

    begin

	declare
	    Arms : constant Asis.Select_Statement_Arm_List := 
	       Asis_S.Select_Statement_Arms (The_Statement);
	    Labels : constant Asis.Entity_Name_Definition_List := 
	       Asis_S.Label_Names (The_Statement);
	begin

	    if Trace.On then
		Trace.Log (The_Statement);
		Trace.Add_Level (Labels'Length + Arms'Length);
	    end if;

	    Namespace_Scan.Scan_Entity_Name_Definition_List 
	       (Labels, The_Statement);

	    Namespace_Scan.Scan_Select_Statement_Arm_List (Arms, The_Context);

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Statement);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Statement);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Statement);
	    raise;

    end Scan_Selective_Wait_Statement;

--| +-------------------------------------------------------------------------+
--| | SCAN_SIMPLE_LOOP_STATEMENT (local)                                      |
--| +-------------------------------------------------------------------------+

    procedure Scan_Simple_Loop_Statement (The_Statement : in Asis.Statement; 
					  The_Context : in Context) is

	Puid : constant String := "Scan_Simple_Loop_Statement";

    begin

	declare
	    Labels : constant Asis.Entity_Name_Definition_List := 
	       Asis_S.Label_Names (The_Statement);
	    Loop_Name : Asis.Entity_Name_Definition := 
	       Asis_S.Loop_Simple_Name (The_Statement);
	    Statements : constant Asis.Statement_List := 
	       Asis_S.Loop_Statements 
		  (The_Statement, Namespace_Scan.
				     Include_Loop_Statement_Pragmas);
	begin

	    if Trace.On then
		Trace.Log (The_Statement);
		Trace.Add_Level (Labels'Length + 
				 Namespace_Scan.One_If_Present (Loop_Name) + 
				 Statements'Length);
	    end if;

	    Namespace_Scan.Scan_Entity_Name_Definition_List 
	       (Labels, The_Statement);

	    if not Asis_E.Is_Nil (Loop_Name) then
		Namespace_Scan.Scan_Entity_Name_Definition 
		   (Loop_Name, The_Statement);
	    end if;

	    Namespace_Scan.Scan_Statement_List (Statements, The_Context);

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Statement);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Statement);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Statement);
	    raise;

    end Scan_Simple_Loop_Statement;

--| +-------------------------------------------------------------------------+
--| | SCAN_TIMED_ENTRY_CALL_STATEMENT (local)                                 |
--| +-------------------------------------------------------------------------+

    procedure Scan_Timed_Entry_Call_Statement 
		 (The_Statement : in Asis.Statement; 
		  The_Context : in Context) is

	Puid : constant String := "Scan_Timed_Entry_Call_Statement";

    begin

	declare
	    Arms : constant Asis.Select_Statement_Arm_List := 
	       Asis_S.Select_Statement_Arms (The_Statement);
	    Labels : constant Asis.Entity_Name_Definition_List := 
	       Asis_S.Label_Names (The_Statement);
	begin

	    if Trace.On then
		Trace.Log (The_Statement);
		Trace.Add_Level (Labels'Length + Arms'Length);
	    end if;

	    Namespace_Scan.Scan_Entity_Name_Definition_List 
	       (Labels, The_Statement);

	    Namespace_Scan.Scan_Select_Statement_Arm_List (Arms, The_Context);

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Statement);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Statement);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Statement);
	    raise;

    end Scan_Timed_Entry_Call_Statement;

--| +-------------------------------------------------------------------------+
--| | SCAN_WHILE_LOOP_STATEMENT (local)                                       |
--| +-------------------------------------------------------------------------+

    procedure Scan_While_Loop_Statement (The_Statement : in Asis.Statement; 
					 The_Context : in Context) is

	Puid : constant String := "Scan_While_Loop_Statement";

    begin

	declare
	    Condition : Asis.Expression := 
	       Asis_S.While_Condition (The_Statement);
	    Labels : constant Asis.Entity_Name_Definition_List := 
	       Asis_S.Label_Names (The_Statement);
	    Loop_Name : Asis.Entity_Name_Definition := 
	       Asis_S.Loop_Simple_Name (The_Statement);
	    Statements : constant Asis.Statement_List := 
	       Asis_S.Loop_Statements 
		  (The_Statement, Namespace_Scan.
				     Include_Loop_Statement_Pragmas);
	begin

	    if Trace.On then
		Trace.Log (The_Statement);
		Trace.Add_Level (Labels'Length + 
				 Namespace_Scan.One_If_Present (Loop_Name) + 1 + 
				 Statements'Length);
	    end if;

	    Namespace_Scan.Scan_Entity_Name_Definition_List 
	       (Labels, The_Statement);

	    if not Asis_E.Is_Nil (Loop_Name) then
		Namespace_Scan.Scan_Entity_Name_Definition 
		   (Loop_Name, The_Statement);
	    end if;

	    Namespace_Scan.Scan_Expression (Condition, The_Context);

	    Namespace_Scan.Scan_Statement_List (Statements, The_Context);

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Statement);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Statement);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Statement);
	    raise;

    end Scan_While_Loop_Statement;

--| +-------------------------------------------------------------------------+
--| | SCAN_STATEMENT (exported)                                               |
--| +-------------------------------------------------------------------------+

begin

    if Namespace_Scan.Obeying_Regions and then 
       Rgn.Denotes_One_Or_More_Regions (The_Statement) then
	declare
	    Remaining_Parent_Elements : constant Asis.Element_List := 
	       Rgn.Non_Region_Subelements 
		  (The_Statement, Include_Pragmas => False);
	begin
	    Namespace_Scan.Scan_Any_List 
	       (Remaining_Parent_Elements, The_Statement);
	end;
	return;
    end if;

    case Asis_S.Kind (The_Statement) is
	when Asis_S.A_Null_Statement =>
	    Scan_Null_Statement (The_Statement, The_Context);
	when Asis_S.An_Assignment_Statement =>
	    Scan_Assignment_Statement (The_Statement, The_Context);
	when Asis_S.A_Procedure_Call_Statement =>
	    Scan_Procedure_Call_Statement (The_Statement, The_Context);
	when Asis_S.An_Exit_Statement =>
	    Scan_Exit_Statement (The_Statement, The_Context);
	when Asis_S.A_Return_Statement =>
	    Scan_Return_Statement (The_Statement, The_Context);
	when Asis_S.A_Goto_Statement =>
	    Scan_Goto_Statement (The_Statement, The_Context);
	when Asis_S.An_Entry_Call_Statement =>
	    Scan_Entry_Call_Statement (The_Statement, The_Context);
	when Asis_S.A_Delay_Statement =>
	    Scan_Delay_Statement (The_Statement, The_Context);
	when Asis_S.An_Abort_Statement =>
	    Scan_Abort_Statement (The_Statement, The_Context);
	when Asis_S.A_Raise_Statement =>
	    Scan_Raise_Statement (The_Statement, The_Context);
	when Asis_S.A_Code_Statement =>
	    Scan_Code_Statement (The_Statement, The_Context);
	when Asis_S.An_If_Statement =>
	    Scan_If_Statement (The_Statement, The_Context);
	when Asis_S.A_Case_Statement =>
	    Scan_Case_Statement (The_Statement, The_Context);
	when Asis_S.A_Loop_Statement =>
	    case Asis_S.Loop_Kind (The_Statement) is
		when Asis_S.A_For_Loop =>
		    Scan_For_Loop_Statement (The_Statement, The_Context);
		when Asis_S.A_While_Loop =>
		    Scan_While_Loop_Statement (The_Statement, The_Context);
		when Asis_S.A_Simple_Loop =>
		    Scan_Simple_Loop_Statement (The_Statement, The_Context);
		when Asis_S.Not_A_Loop =>
		    Error.Log (Error.An_Unhandled_Case, Cuid, 
			       Puid & "/1", The_Statement);
		    if Namespace_Scan.Raise_Exception_On_Unhandled_Case then
			raise Traversal_Error;
		    else
			if Trace.On then
			    Trace.Log (The_Statement);
			end if;
		    end if;
	    end case;
	when Asis_S.A_Block_Statement =>
	    Scan_Block_Statement (The_Statement, The_Context);
	when Asis_S.An_Accept_Statement =>
	    Scan_Accept_Statement (The_Statement, The_Context);
	when Asis_S.A_Selective_Wait_Statement =>
	    Scan_Selective_Wait_Statement (The_Statement, The_Context);
	when Asis_S.A_Conditional_Entry_Call_Statement =>
	    Scan_Conditional_Entry_Call_Statement (The_Statement, The_Context);
	when Asis_S.A_Timed_Entry_Call_Statement =>
	    Scan_Timed_Entry_Call_Statement (The_Statement, The_Context);
	when Asis_S.Not_A_Statement =>
	    Error.Log (Error.An_Unhandled_Case, Cuid, 
		       Puid & "/2", The_Statement);
	    if Namespace_Scan.Raise_Exception_On_Unhandled_Case then
		raise Traversal_Error;
	    else
		if Trace.On then
		    Trace.Log (The_Statement);
		end if;
	    end if;
    end case;

end Scan_Statement;
