--| +=========================================================================+
--| |                                                                         |
--| | REFERENCE_SCAN.SCAN_CHOICE (body)                                       |
--| |                                                                         |
--| | Greg Janee                                                              |
--| | General Research Corporation                                            |
--| |                                                                         |
--| +=========================================================================+

separate (Reference_Scan)
procedure Scan_Choice (The_Choice : in Asis.Choice; The_Context : in Context) is

    Cuid : constant String := "Reference_Scan.Scan_Choice";
    Puid : constant String := "Scan_Choice";

    package Ctx renames Reference_Scan.Context_Support;
    package Error renames Reference_Scan.Error_Handling_Support;
    package Trace renames Reference_Scan.Trace_Support;

--| +-------------------------------------------------------------------------+
--| | SCAN_DISCRETE_RANGE (local)                                             |
--| +-------------------------------------------------------------------------+

    procedure Scan_Discrete_Range 
		 (The_Choice : in Asis.Choice; The_Context : in Context) is

	Puid : constant String := "Scan_Discrete_Range";

    begin

	declare
	    The_Range : Asis.Discrete_Range := 
	       Asis_Td.Choice_Discrete_Range (The_Choice);
	begin

	    if Trace.On then
		Trace.Log (The_Choice, The_Context);
		Trace.Add_Level (1);
	    end if;

	    Reference_Scan.Scan_Discrete_Range 
	       (The_Range, Ctx.Set (The_Context, 
				    Basic_Context => Rvs.A_Read, 
				    Weight => 90, 
				    New_Weight => 5, 
				    Add_Data_Access_Context => Rvs.A_Choice, 
				    Type_Mark_Context => Rvs.A_Choice));

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Choice);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Choice);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Choice);
	    raise;

    end Scan_Discrete_Range;

--| +-------------------------------------------------------------------------+
--| | SCAN_EXCEPTION_NAME (local)                                             |
--| +-------------------------------------------------------------------------+

    procedure Scan_Exception_Name 
		 (The_Choice : in Asis.Choice; The_Context : in Context) is

	Puid : constant String := "Scan_Exception_Name";

    begin

	declare
	    Name : Asis.Expression := Asis_Td.Choice_Name (The_Choice);
	begin

	    if Trace.On then
		Trace.Log (The_Choice, The_Context);
		Trace.Add_Level (1);
	    end if;

	    Reference_Scan.Scan_Expression 
	       (Name, Ctx.Set (The_Context, 
			       Basic_Context => Rvs.A_Handle, 
			       Weight => 90));

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Choice);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Choice);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Choice);
	    raise;

    end Scan_Exception_Name;

--| +-------------------------------------------------------------------------+
--| | SCAN_OTHERS_CHOICE (local)                                              |
--| +-------------------------------------------------------------------------+

    procedure Scan_Others_Choice 
		 (The_Choice : in Asis.Choice; The_Context : in Context) is

	Puid : constant String := "Scan_Others_Choice";

    begin

	if Trace.On then
	    Trace.Log (The_Choice, The_Context);
	end if;

    end Scan_Others_Choice;

--| +-------------------------------------------------------------------------+
--| | SCAN_SIMPLE_EXPRESSION (local)                                          |
--| +-------------------------------------------------------------------------+

    procedure Scan_Simple_Expression 
		 (The_Choice : in Asis.Choice; The_Context : in Context) is

	Puid : constant String := "Scan_Simple_Expression";

    begin

	declare
	    The_Expression : Asis.Expression := 
	       Asis_Td.Choice_Simple_Expression (The_Choice);
	begin

	    if Trace.On then
		Trace.Log (The_Choice, The_Context);
		Trace.Add_Level (1);
	    end if;

	    Reference_Scan.Scan_Expression 
	       (The_Expression, 
		Ctx.Set (The_Context, 
			 Basic_Context => Rvs.A_Read, 
			 Weight => 90, 
			 New_Weight => 5, 
			 Add_Data_Access_Context => Rvs.A_Choice, 
			 Type_Mark_Context => Rvs.A_Choice));

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Choice);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Choice);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Choice);
	    raise;

    end Scan_Simple_Expression;

--| +-------------------------------------------------------------------------+
--| | SCAN_SIMPLE_NAME (local)                                                |
--| +-------------------------------------------------------------------------+

    procedure Scan_Simple_Name (The_Choice : in Asis.Choice; 
				The_Context : in Context) is

	Puid : constant String := "Scan_Simple_Name";

    begin

	declare
	    Name : Asis.Expression := Asis_Td.Choice_Name (The_Choice);
	begin

	    if Trace.On then
		Trace.Log (The_Choice, The_Context);
		Trace.Add_Level (1);
	    end if;

	    Reference_Scan.Scan_Expression 
	       (Name, Ctx.Set (The_Context, 
			       Basic_Context => Rvs.A_Read, 
			       Weight => 90, 
			       New_Weight => 5, 
			       Add_Data_Access_Context => Rvs.A_Choice, 
			       Type_Mark_Context => Rvs.A_Choice));

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Choice);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Choice);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Choice);
	    raise;

    end Scan_Simple_Name;

--| +-------------------------------------------------------------------------+
--| | SCAN_CHOICE (exported)                                                  |
--| +-------------------------------------------------------------------------+

begin

    case Asis_Td.Choice_Kind (The_Choice) is
	when Asis_Td.A_Simple_Expression =>
	    Scan_Simple_Expression (The_Choice, The_Context);
	when Asis_Td.A_Discrete_Range =>
	    Scan_Discrete_Range (The_Choice, The_Context);
	when Asis_Td.An_Others_Choice =>
	    Scan_Others_Choice (The_Choice, The_Context);
	when Asis_Td.A_Simple_Name =>
	    Scan_Simple_Name (The_Choice, The_Context);
	when Asis_Td.An_Exception_Name =>
	    Scan_Exception_Name (The_Choice, The_Context);
	when Asis_Td.Not_A_Choice =>
	    Error.Log (Error.An_Unhandled_Case, Cuid, Puid, The_Choice);
	    if Reference_Scan.Raise_Exception_On_Unhandled_Case then
		raise Traversal_Error;
	    else
		if Trace.On then
		    Trace.Log (The_Choice, The_Context);
		end if;
	    end if;
    end case;

end Scan_Choice;
