--| +=========================================================================+
--| |                                                                         |
--| | REFERENCE_SCAN.SCAN_DECLARATION (body)                                  |
--| |                                                                         |
--| | Greg Janee                                                              |
--| | General Research Corporation                                            |
--| |                                                                         |
--| +=========================================================================+

separate (Reference_Scan)
procedure Scan_Declaration (The_Declaration : in Asis.Declaration; 
			    The_Context : in Context) is

    Cuid : constant String := "Reference_Scan.Scan_Declaration";
    Puid : constant String := "Scan_Declaration";

    package Ctx renames Reference_Scan.Context_Support;
    package Error renames Reference_Scan.Error_Handling_Support;
    package Trace renames Reference_Scan.Trace_Support;

--| +-------------------------------------------------------------------------+
--| | SCAN_COMPONENT_DECLARATION (local)                                      |
--| +-------------------------------------------------------------------------+

    procedure Scan_Component_Declaration (The_Declaration : in Asis.Declaration; 
					  The_Context : in Context) is

	Puid : constant String := "Scan_Component_Declaration";

    begin

	declare
	    Default_Value : Asis.Expression := 
	       Asis_D.Initial_Value (The_Declaration);
	    The_Type : Asis.Type_Definition := 
	       Asis_D.Object_Declaration_Definition (The_Declaration);
	begin

	    if Trace.On then
		Trace.Log (The_Declaration, The_Context);
		Trace.Add_Level (1 + 
				 Reference_Scan.One_If_Present (Default_Value));
	    end if;

	    Reference_Scan.Scan_Type_Definition 
	       (The_Type, Ctx.Set (The_Context, 
				   Type_Mark_Context => 
				      Rvs.A_Component_Subtype_Definition));

	    if not Asis_E.Is_Nil (Default_Value) then
		Reference_Scan.Scan_Expression 
		   (Default_Value, Ctx.Set (The_Context, 
					    Basic_Context => Rvs.A_Read, 
					    Weight => 5, 
					    Add_Data_Access_Context => 
					       Rvs.A_Component_Declaration));
	    end if;

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Declaration);
	    raise;

    end Scan_Component_Declaration;

--| +-------------------------------------------------------------------------+
--| | SCAN_CONSTANT_DECLARATION (local)                                       |
--| +-------------------------------------------------------------------------+

    procedure Scan_Constant_Declaration (The_Declaration : in Asis.Declaration; 
					 The_Context : in Context) is

	Puid : constant String := "Scan_Constant_Declaration";

    begin

	declare
	    The_Type : Asis.Type_Definition := 
	       Asis_D.Object_Declaration_Definition (The_Declaration);
	    Value : Asis.Expression := Asis_D.Initial_Value (The_Declaration);
	begin

	    if Trace.On then
		Trace.Log (The_Declaration, The_Context);
		Trace.Add_Level (2);
	    end if;

	    Reference_Scan.Scan_Type_Definition 
	       (The_Type, Ctx.Set 
			     (The_Context, 
			      Type_Mark_Context => Rvs.An_Object_Declaration));

	    Reference_Scan.Scan_Expression 
	       (Value, Ctx.Set (The_Context, 
				Basic_Context => Rvs.A_Read, 
				Weight => 5, 
				Add_Data_Access_Context => 
				   Rvs.An_Object_Declaration));

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Declaration);
	    raise;

    end Scan_Constant_Declaration;

--| +-------------------------------------------------------------------------+
--| | SCAN_DEFERRED_CONSTANT_DECLARATION (local)                              |
--| +-------------------------------------------------------------------------+

    procedure Scan_Deferred_Constant_Declaration 
		 (The_Declaration : in Asis.Declaration; 
		  The_Context : in Context) is

	Puid : constant String := "Scan_Deferred_Constant_Declaration";

    begin

	declare
	    Type_Mark : Asis.Expression := Asis_D.Type_Mark (The_Declaration);
	begin

	    if Trace.On then
		Trace.Log (The_Declaration, The_Context);
		Trace.Add_Level (1);
	    end if;

	    Reference_Scan.Scan_Expression 
	       (Type_Mark, Ctx.Set (The_Context, 
				    Basic_Context => Rvs.A_Type_Mark, 
				    Weight => 90, 
				    Type_Mark_Context => 
				       Rvs.A_Deferred_Constant_Declaration));

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Declaration);
	    raise;

    end Scan_Deferred_Constant_Declaration;

--| +-------------------------------------------------------------------------+
--| | SCAN_DISCRIMINANT_SPECIFICATION (local)                                 |
--| +-------------------------------------------------------------------------+

    procedure Scan_Discriminant_Specification 
		 (The_Declaration : in Asis.Declaration; 
		  The_Context : in Context) is

	Puid : constant String := "Scan_Discriminant_Specification";

    begin

	declare
	    Default_Value : Asis.Expression := 
	       Asis_D.Initial_Value (The_Declaration);
	    The_Type : Asis.Expression := Asis_D.Type_Mark (The_Declaration);
	begin

	    if Trace.On then
		Trace.Log (The_Declaration, The_Context);
		Trace.Add_Level (1 + 
				 Reference_Scan.One_If_Present (Default_Value));
	    end if;

	    Reference_Scan.Scan_Expression 
	       (The_Type, Ctx.Set (The_Context, 
				   Basic_Context => Rvs.A_Type_Mark, 
				   Weight => 90, 
				   Type_Mark_Context => 
				      Rvs.A_Discriminant_Specification));

	    if not Asis_E.Is_Nil (Default_Value) then
		Reference_Scan.Scan_Expression 
		   (Default_Value, 
		    Ctx.Set (The_Context, 
			     Basic_Context => Rvs.A_Read, 
			     Weight => 5, 
			     Add_Data_Access_Context => 
				Rvs.A_Discriminant_Specification));
	    end if;

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Declaration);
	    raise;

    end Scan_Discriminant_Specification;

--| +-------------------------------------------------------------------------+
--| | SCAN_ENTRY_DECLARATION (local)                                          |
--| +-------------------------------------------------------------------------+

    procedure Scan_Entry_Declaration (The_Declaration : in Asis.Declaration; 
				      The_Context : in Context) is

	Puid : constant String := "Scan_Entry_Declaration";

    begin

	declare
	    Index : Asis.Discrete_Range := 
	       Asis_D.Family_Index (The_Declaration);
	    Parameters : constant Asis.Parameter_Specification_List := 
	       Asis_D.Parameters (The_Declaration);
	begin

	    if Trace.On then
		Trace.Log (The_Declaration, The_Context);
		Trace.Add_Level (Reference_Scan.One_If_Present (Index) + 
				 Reference_Scan.One_If_True 
				    (not Reference_Scan.Obeying_Regions) * 
				 Parameters'Length);
	    end if;

	    if not Asis_E.Is_Nil (Index) then
		Reference_Scan.Scan_Discrete_Range 
		   (Index, Ctx.Set 
			      (The_Context, 
			       Type_Mark_Context => Rvs.An_Entry_Declaration));
	    end if;

	    if not Reference_Scan.Obeying_Regions then
		Reference_Scan.Scan_Declaration_List (Parameters, The_Context);
	    end if;

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Declaration);
	    raise;

    end Scan_Entry_Declaration;

--| +-------------------------------------------------------------------------+
--| | SCAN_ENUMERATION_LITERAL_SPECIFICATION (local)                          |
--| +-------------------------------------------------------------------------+

    procedure Scan_Enumeration_Literal_Specification 
		 (The_Declaration : in Asis.Declaration; 
		  The_Context : in Context) is

	Puid : constant String := "Scan_Enumeration_Literal_Specification";

    begin

	if Trace.On then
	    Trace.Log (The_Declaration, The_Context);
	end if;

    end Scan_Enumeration_Literal_Specification;

--| +-------------------------------------------------------------------------+
--| | SCAN_EXCEPTION_DECLARATION (local)                                      |
--| +-------------------------------------------------------------------------+

    procedure Scan_Exception_Declaration (The_Declaration : in Asis.Declaration; 
					  The_Context : in Context) is

	Puid : constant String := "Scan_Exception_Declaration";

    begin

	if Trace.On then
	    Trace.Log (The_Declaration, The_Context);
	end if;

    end Scan_Exception_Declaration;

--| +-------------------------------------------------------------------------+
--| | SCAN_EXCEPTION_RENAME_DECLARATION (local)                               |
--| +-------------------------------------------------------------------------+

    procedure Scan_Exception_Rename_Declaration 
		 (The_Declaration : in Asis.Declaration; 
		  The_Context : in Context) is

	Puid : constant String := "Scan_Exception_Rename_Declaration";

    begin

	declare
	    Renamed_Exception : Asis.Expression := 
	       Asis_D.Renamed_Entity (The_Declaration);
	begin

	    if Trace.On then
		Trace.Log (The_Declaration, The_Context);
		Trace.Add_Level (1);
	    end if;

	    Reference_Scan.Scan_Expression 
	       (Renamed_Exception, Ctx.Set (The_Context, 
					    Basic_Context => Rvs.A_Rename, 
					    Weight => 90));

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Declaration);
	    raise;

    end Scan_Exception_Rename_Declaration;

--| +-------------------------------------------------------------------------+
--| | SCAN_FULL_TYPE_DECLARATION (local)                                      |
--| +-------------------------------------------------------------------------+

    procedure Scan_Full_Type_Declaration (The_Declaration : in Asis.Declaration; 
					  The_Context : in Context) is

	Puid : constant String := "Scan_Full_Type_Declaration";

    begin

	declare
	    Discriminants : constant Asis.Discriminant_Specification_List := 
	       Asis_D.Discriminants (The_Declaration);
	    The_Type : Asis.Type_Definition := 
	       Asis_D.Type_Declaration_Definition (The_Declaration);
	begin

	    if Trace.On then
		Trace.Log (The_Declaration, The_Context);
	    end if;

	    -- A full type declaration is not always a region. Use Rgn  
	    -- operation to check it.
	    if not (Reference_Scan.Obeying_Regions and then 
		    Rgn.Denotes_One_Or_More_Regions (The_Declaration)) then
		if Trace.On then
		    Trace.Add_Level (Discriminants'Length + 1);
		end if;
		Reference_Scan.Scan_Declaration_List 
		   (Discriminants, The_Context);
		Reference_Scan.Scan_Type_Definition (The_Type, The_Context);
	    end if;

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Rgn.Region_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Declaration);
	    raise;

    end Scan_Full_Type_Declaration;

--| +-------------------------------------------------------------------------+
--| | SCAN_FUNCTION_BODY_DECLARATION (local)                                  |
--| +-------------------------------------------------------------------------+

    procedure Scan_Function_Body_Declaration 
		 (The_Declaration : in Asis.Declaration; 
		  The_Context : in Context) is

	Puid : constant String := "Scan_Function_Body_Declaration";

    begin

	declare
	    Parameters : constant Asis.Parameter_Specification_List := 
	       Asis_D.Parameters (The_Declaration);
	    Return_Type : Asis.Expression := 
	       Asis_D.Return_Type (The_Declaration);
	    The_Body : Asis.Statement := 
	       Asis_D.Subprogram_Body_Block (The_Declaration);
	begin

	    if Trace.On then
		Trace.Log (The_Declaration, The_Context);
	    end if;

	    if not Reference_Scan.Obeying_Regions then
		if Trace.On then
		    Trace.Add_Level (Parameters'Length + 2);
		end if;
		Reference_Scan.Scan_Declaration_List (Parameters, The_Context);
		Reference_Scan.Scan_Expression 
		   (Return_Type, 
		    Ctx.Set (The_Context, 
			     Basic_Context => Rvs.A_Type_Mark, 
			     Weight => 90, 
			     Type_Mark_Context => Rvs.A_Function_Body));
		Reference_Scan.Scan_Statement (The_Body, The_Context);
	    end if;

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Declaration);
	    raise;

    end Scan_Function_Body_Declaration;

--| +-------------------------------------------------------------------------+
--| | SCAN_FUNCTION_BODY_STUB (local)                                         |
--| +-------------------------------------------------------------------------+

    procedure Scan_Function_Body_Stub (The_Declaration : in Asis.Declaration; 
				       The_Context : in Context) is

	Puid : constant String := "Scan_Function_Body_Stub";

    begin

	declare
	    Parameters : constant Asis.Parameter_Specification_List := 
	       Asis_D.Parameters (The_Declaration);
	    Return_Type : Asis.Expression := 
	       Asis_D.Return_Type (The_Declaration);
	begin

	    if Trace.On then
		Trace.Log (The_Declaration, The_Context);
	    end if;

	    if not Reference_Scan.Obeying_Regions then
		if Trace.On then
		    Trace.Add_Level (Parameters'Length + 1);
		end if;
		Reference_Scan.Scan_Declaration_List (Parameters, The_Context);
		Reference_Scan.Scan_Expression 
		   (Return_Type, 
		    Ctx.Set (The_Context, 
			     Basic_Context => Rvs.A_Type_Mark, 
			     Weight => 90, 
			     Type_Mark_Context => Rvs.A_Function_Stub));
	    end if;

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Declaration);
	    raise;

    end Scan_Function_Body_Stub;

--| +-------------------------------------------------------------------------+
--| | SCAN_FUNCTION_DECLARATION (local)                                       |
--| +-------------------------------------------------------------------------+

    procedure Scan_Function_Declaration (The_Declaration : in Asis.Declaration; 
					 The_Context : in Context) is

	Puid : constant String := "Scan_Function_Declaration";

    begin

	declare
	    Parameters : constant Asis.Parameter_Specification_List := 
	       Asis_D.Parameters (The_Declaration);
	    Return_Type : Asis.Expression := 
	       Asis_D.Return_Type (The_Declaration);
	begin

	    if Trace.On then
		Trace.Log (The_Declaration, The_Context);
	    end if;

	    if not Reference_Scan.Obeying_Regions then
		if Trace.On then
		    Trace.Add_Level (Parameters'Length + 1);
		end if;
		Reference_Scan.Scan_Declaration_List (Parameters, The_Context);
		Reference_Scan.Scan_Expression 
		   (Return_Type, 
		    Ctx.Set (The_Context, 
			     Basic_Context => Rvs.A_Type_Mark, 
			     Weight => 90, 
			     Type_Mark_Context => Rvs.A_Function_Declaration));
	    end if;

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Declaration);
	    raise;

    end Scan_Function_Declaration;

--| +-------------------------------------------------------------------------+
--| | SCAN_FUNCTION_INSTANTIATION (local)                                     |
--| +-------------------------------------------------------------------------+

    procedure Scan_Function_Instantiation 
		 (The_Declaration : in Asis.Declaration; 
		  The_Context : in Context) is

	Puid : constant String := "Scan_Function_Instantiation";

    begin

	if Trace.On then
	    Trace.Log (The_Declaration, The_Context);
	end if;

	if Reference_Scan.Expand_Function_Instantiations then

	    if not Reference_Scan.Obeying_Regions then
		declare
		    Specification : Asis.Declaration := 
		       Asis_D.Corresponding_Specification (The_Declaration);
		    The_Body : Asis.Declaration := 
		       Asis_D.Corresponding_Body (The_Declaration);
		begin
		    if Trace.On then
			Trace.Add_Level 
			   (Reference_Scan.One_If_Present (Specification) + 
			    Reference_Scan.One_If_Present (The_Body));
		    end if;
		    if not Asis_E.Is_Nil (Specification) then
			Reference_Scan.Scan_Declaration 
			   (Specification, The_Context);
		    end if;
		    if not Asis_E.Is_Nil (The_Body) then
			Reference_Scan.Scan_Declaration (The_Body, The_Context);
		    end if;
		end;
	    end if;

	else

	    declare
		Arguments : constant Asis.Association_List := 
		   Asis_D.Generic_Parameters 
		      (The_Declaration, Reference_Scan.
					   Normalize_Instantiation_Parameters);
		The_Function : Asis.Expression := 
		   Asis_D.Generic_Unit_Name (The_Declaration);
	    begin
		if Trace.On then
		    Trace.Add_Level (1 + Arguments'Length);
		end if;
		Reference_Scan.Scan_Expression 
		   (The_Function, 
		    Ctx.Set (The_Context, 
			     Basic_Context => Rvs.An_Instantiation, 
			     Weight => 90));
		Reference_Scan.Scan_Parameter_Association_List 
		   (Arguments, Ctx.Set 
				  (The_Context, 
				   Basic_Context => Rvs.A_Generic_Association, 
				   Weight => 90));
	    end;

	end if;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Declaration);
	    raise;

    end Scan_Function_Instantiation;

--| +-------------------------------------------------------------------------+
--| | SCAN_FUNCTION_RENAME_DECLARATION (local)                                |
--| +-------------------------------------------------------------------------+

    procedure Scan_Function_Rename_Declaration 
		 (The_Declaration : in Asis.Declaration; 
		  The_Context : in Context) is

	Puid : constant String := "Scan_Function_Rename_Declaration";

    begin

	declare
	    Parameters : constant Asis.Parameter_Specification_List := 
	       Asis_D.Parameters (The_Declaration);
	    Renamed_Function : Asis.Expression := 
	       Asis_D.Renamed_Entity (The_Declaration);
	    Return_Type : Asis.Expression := 
	       Asis_D.Return_Type (The_Declaration);
	    Obey_Region : Boolean;
	begin
	    -- A function rename is not always a region. Use Rgn operation 
	    -- to check it.
	    Obey_Region := Reference_Scan.Obeying_Regions and then 
			      Rgn.Denotes_One_Or_More_Regions (The_Declaration);

	    if Trace.On then
		Trace.Log (The_Declaration, The_Context);
		Trace.Add_Level (Reference_Scan.One_If_True (not Obey_Region) * 
				 (Parameters'Length + 1) + 1);
	    end if;

	    if not Obey_Region then
		Reference_Scan.Scan_Declaration_List (Parameters, The_Context);
		Reference_Scan.Scan_Expression 
		   (Return_Type, 
		    Ctx.Set (The_Context, 
			     Basic_Context => Rvs.A_Type_Mark, 
			     Weight => 90, 
			     Type_Mark_Context => Rvs.A_Function_Rename));
	    end if;

	    Reference_Scan.Scan_Expression 
	       (Renamed_Function, Ctx.Set (The_Context, 
					   Basic_Context => Rvs.A_Rename, 
					   Weight => 90));

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Rgn.Region_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Declaration);
	    raise;

    end Scan_Function_Rename_Declaration;

--| +-------------------------------------------------------------------------+
--| | SCAN_GENERIC_FORMAL_FUNCTION_DECLARATION (local)                        |
--| +-------------------------------------------------------------------------+

    procedure Scan_Generic_Formal_Function_Declaration 
		 (The_Declaration : in Asis.Declaration; 
		  The_Context : in Context) is

	Puid : constant String := "Scan_Generic_Formal_Function_Declaration";

    begin

	declare
	    Default_Kind : Asis_D.Generic_Formal_Subprogram_Default_Kinds := 
	       Asis_D.Generic_Formal_Subprogram_Default_Kind (The_Declaration);
	    Parameters : constant Asis.Parameter_Specification_List := 
	       Asis_D.Parameters (The_Declaration);
	    Return_Type : Asis.Expression := 
	       Asis_D.Return_Type (The_Declaration);
	    Obey_Region : Boolean;
	begin
	    -- A generic formal function is not always a region. Use Rgn operation 
	    -- to check it.
	    Obey_Region := Reference_Scan.Obeying_Regions and then 
			      Rgn.Denotes_One_Or_More_Regions (The_Declaration);

	    if Trace.On then
		Trace.Log (The_Declaration, The_Context);
		Trace.Add_Level (Reference_Scan.One_If_True (not Obey_Region) * 
				 (Parameters'Length + 1));
		if Asis_D."=" (Default_Kind, Asis_D.A_Name) then
		    Trace.Add_Children (1);
		end if;
	    end if;

	    if not Obey_Region then
		Reference_Scan.Scan_Declaration_List (Parameters, The_Context);
		Reference_Scan.Scan_Expression 
		   (Return_Type, Ctx.Set (The_Context, 
					  Basic_Context => Rvs.A_Type_Mark, 
					  Weight => 90, 
					  Type_Mark_Context => 
					     Rvs.A_Generic_Formal_Function));
	    end if;

	    if Asis_D."=" (Default_Kind, Asis_D.A_Name) then
		Reference_Scan.Scan_Expression 
		   (Asis_D.Generic_Formal_Subprogram_Default (The_Declaration), 
		    Ctx.Set (The_Context, 
			     Basic_Context => 
				Rvs.A_Generic_Formal_Subprogram_Default, 
			     Weight => 90));
	    end if;

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Rgn.Region_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Declaration);
	    raise;

    end Scan_Generic_Formal_Function_Declaration;

--| +-------------------------------------------------------------------------+
--| | SCAN_GENERIC_FORMAL_OBJECT_DECLARATION (local)                          |
--| +-------------------------------------------------------------------------+

    procedure Scan_Generic_Formal_Object_Declaration 
		 (The_Declaration : in Asis.Declaration; 
		  The_Context : in Context) is

	Puid : constant String := "Scan_Generic_Formal_Object_Declaration";

    begin

	declare
	    Default_Value : Asis.Expression := 
	       Asis_D.Initial_Value (The_Declaration);
	    The_Type : Asis.Expression := Asis_D.Type_Mark (The_Declaration);
	begin

	    if Trace.On then
		Trace.Log (The_Declaration, The_Context);
		Trace.Add_Level (1 + 
				 Reference_Scan.One_If_Present (Default_Value));
	    end if;

	    Reference_Scan.Scan_Expression 
	       (The_Type, Ctx.Set (The_Context, 
				   Basic_Context => Rvs.A_Type_Mark, 
				   Weight => 90, 
				   Type_Mark_Context => 
				      Rvs.A_Generic_Parameter_Declaration));

	    if not Asis_E.Is_Nil (Default_Value) then
		Reference_Scan.Scan_Expression 
		   (Default_Value, 
		    Ctx.Set (The_Context, 
			     Basic_Context => Rvs.A_Read, 
			     Weight => 5, 
			     Add_Data_Access_Context => 
				Rvs.A_Generic_Parameter_Declaration));
	    end if;

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Declaration);
	    raise;

    end Scan_Generic_Formal_Object_Declaration;

--| +-------------------------------------------------------------------------+
--| | SCAN_GENERIC_FORMAL_PRIVATE_TYPE_DECLARATION (local)                    |
--| +-------------------------------------------------------------------------+

    procedure Scan_Generic_Formal_Private_Type_Declaration 
		 (The_Declaration : in Asis.Declaration; 
		  The_Context : in Context) is

	Puid : constant String := 
	   "Scan_Generic_Formal_Private_Type_Declaration";

    begin

	declare
	    Discriminants : constant Asis.Discriminant_Specification_List := 
	       Asis_D.Discriminants (The_Declaration);
	    Obey_Region : Boolean;
	begin
	    -- A generic formal private type is not always a region. Use Rgn  
	    -- operation to check it.
	    Obey_Region := Reference_Scan.Obeying_Regions and then 
			      Rgn.Denotes_One_Or_More_Regions (The_Declaration);

	    if Trace.On then
		Trace.Log (The_Declaration, The_Context);
	    end if;

	    if not Obey_Region then
		if Trace.On then
		    Trace.Add_Level (Discriminants'Length);
		end if;
		Reference_Scan.Scan_Declaration_List 
		   (Discriminants, The_Context);
	    end if;

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Rgn.Region_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Declaration);
	    raise;

    end Scan_Generic_Formal_Private_Type_Declaration;

--| +-------------------------------------------------------------------------+
--| | SCAN_GENERIC_FORMAL_PROCEDURE_DECLARATION (local)                       |
--| +-------------------------------------------------------------------------+

    procedure Scan_Generic_Formal_Procedure_Declaration 
		 (The_Declaration : in Asis.Declaration; 
		  The_Context : in Context) is

	Puid : constant String := "Scan_Generic_Formal_Procedure_Declaration";

    begin

	declare
	    Default_Kind : Asis_D.Generic_Formal_Subprogram_Default_Kinds := 
	       Asis_D.Generic_Formal_Subprogram_Default_Kind (The_Declaration);
	    Parameters : constant Asis.Parameter_Specification_List := 
	       Asis_D.Parameters (The_Declaration);
	    Obey_Region : Boolean;
	begin
	    -- A generic formal procedure is not always a region. Use Rgn operation 
	    -- to check it.
	    Obey_Region := Reference_Scan.Obeying_Regions and then 
			      Rgn.Denotes_One_Or_More_Regions (The_Declaration);


	    if Trace.On then
		Trace.Log (The_Declaration, The_Context);
		Trace.Add_Level (Reference_Scan.One_If_True (not Obey_Region) * 
				 Parameters'Length);
		if Asis_D."=" (Default_Kind, Asis_D.A_Name) then
		    Trace.Add_Children (1);
		end if;
	    end if;

	    if not Obey_Region then
		Reference_Scan.Scan_Declaration_List (Parameters, The_Context);
	    end if;

	    if Asis_D."=" (Default_Kind, Asis_D.A_Name) then
		Reference_Scan.Scan_Expression 
		   (Asis_D.Generic_Formal_Subprogram_Default (The_Declaration), 
		    Ctx.Set (The_Context, 
			     Basic_Context => 
				Rvs.A_Generic_Formal_Subprogram_Default, 
			     Weight => 90));
	    end if;

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Rgn.Region_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Declaration);
	    raise;

    end Scan_Generic_Formal_Procedure_Declaration;

--| +-------------------------------------------------------------------------+
--| | SCAN_GENERIC_FORMAL_TYPE_DECLARATION (local)                            |
--| +-------------------------------------------------------------------------+

    procedure Scan_Generic_Formal_Type_Declaration 
		 (The_Declaration : in Asis.Declaration; 
		  The_Context : in Context) is

	Puid : constant String := "Scan_Generic_Formal_Type_Declaration";

    begin

	declare
	    The_Type : Asis.Type_Definition := 
	       Asis_D.Type_Declaration_Definition (The_Declaration);
	begin

	    if Trace.On then
		Trace.Log (The_Declaration, The_Context);
		Trace.Add_Level (1);
	    end if;

	    Reference_Scan.Scan_Type_Definition (The_Type, The_Context);

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Declaration);
	    raise;

    end Scan_Generic_Formal_Type_Declaration;

--| +-------------------------------------------------------------------------+
--| | SCAN_GENERIC_FUNCTION_DECLARATION (local)                               |
--| +-------------------------------------------------------------------------+

    procedure Scan_Generic_Function_Declaration 
		 (The_Declaration : in Asis.Declaration; 
		  The_Context : in Context) is

	Puid : constant String := "Scan_Generic_Function_Declaration";

    begin

	declare
	    Formal_Parameters : constant Asis.Generic_Formal_Parameter_List := 
	       Asis_D.Generic_Formal_Parameters 
		  (The_Declaration, Reference_Scan.
				       Include_Generic_Formal_Part_Pragmas);
	    Parameters : constant Asis.Parameter_Specification_List := 
	       Asis_D.Parameters (The_Declaration);
	    Return_Type : Asis.Expression := 
	       Asis_D.Return_Type (The_Declaration);
	begin

	    if Trace.On then
		Trace.Log (The_Declaration, The_Context);
	    end if;

	    if not Reference_Scan.Obeying_Regions then
		if Trace.On then
		    Trace.Add_Level (Formal_Parameters'Length + 
				     Parameters'Length + 1);
		end if;
		Reference_Scan.Scan_Declarative_Item_List 
		   (Formal_Parameters, The_Context);
		Reference_Scan.Scan_Declaration_List (Parameters, The_Context);
		Reference_Scan.Scan_Expression 
		   (Return_Type, 
		    Ctx.Set (The_Context, 
			     Basic_Context => Rvs.A_Type_Mark, 
			     Weight => 90, 
			     Type_Mark_Context => Rvs.A_Generic_Function));
	    end if;

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Declaration);
	    raise;

    end Scan_Generic_Function_Declaration;

--| +-------------------------------------------------------------------------+
--| | SCAN_GENERIC_PACKAGE_DECLARATION (local)                                |
--| +-------------------------------------------------------------------------+

    procedure Scan_Generic_Package_Declaration 
		 (The_Declaration : in Asis.Declaration; 
		  The_Context : in Context) is

	Puid : constant String := "Scan_Generic_Package_Declaration";

    begin

	declare
	    Formal_Parameters : constant Asis.Generic_Formal_Parameter_List := 
	       Asis_D.Generic_Formal_Parameters 
		  (The_Declaration, Reference_Scan.
				       Include_Generic_Formal_Part_Pragmas);
	    Private_Declarations : constant Asis.Declarative_Item_List := 
	       Asis_D.Private_Part_Declarative_Items 
		  (The_Declaration, Reference_Scan.Include_Package_Pragmas);
	    Public_Declarations : constant Asis.Declarative_Item_List := 
	       Asis_D.Visible_Part_Declarative_Items 
		  (The_Declaration, Reference_Scan.Include_Package_Pragmas);
	begin

	    if Trace.On then
		Trace.Log (The_Declaration, The_Context);
	    end if;

	    if not Reference_Scan.Obeying_Regions then
		if Trace.On then
		    Trace.Add_Level (Formal_Parameters'Length + 
				     Public_Declarations'Length + 
				     Private_Declarations'Length);
		end if;
		Reference_Scan.Scan_Declarative_Item_List 
		   (Formal_Parameters, The_Context);
		Reference_Scan.Scan_Declarative_Item_List 
		   (Public_Declarations, The_Context);
		Reference_Scan.Scan_Declarative_Item_List 
		   (Private_Declarations, The_Context);
	    end if;

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Declaration);
	    raise;

    end Scan_Generic_Package_Declaration;

--| +-------------------------------------------------------------------------+
--| | SCAN_GENERIC_PROCEDURE_DECLARATION (local)                              |
--| +-------------------------------------------------------------------------+

    procedure Scan_Generic_Procedure_Declaration 
		 (The_Declaration : in Asis.Declaration; 
		  The_Context : in Context) is

	Puid : constant String := "Scan_Generic_Procedure_Declaration";

    begin

	declare
	    Formal_Parameters : constant Asis.Generic_Formal_Parameter_List := 
	       Asis_D.Generic_Formal_Parameters 
		  (The_Declaration, Reference_Scan.
				       Include_Generic_Formal_Part_Pragmas);
	    Parameters : constant Asis.Parameter_Specification_List := 
	       Asis_D.Parameters (The_Declaration);
	begin

	    if Trace.On then
		Trace.Log (The_Declaration, The_Context);
	    end if;

	    if not Reference_Scan.Obeying_Regions then
		if Trace.On then
		    Trace.Add_Level (Formal_Parameters'Length + 
				     Parameters'Length);
		end if;
		Reference_Scan.Scan_Declarative_Item_List 
		   (Formal_Parameters, The_Context);
		Reference_Scan.Scan_Declaration_List (Parameters, The_Context);
	    end if;

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Declaration);
	    raise;

    end Scan_Generic_Procedure_Declaration;

--| +-------------------------------------------------------------------------+
--| | SCAN_INCOMPLETE_TYPE_DECLARATION (local)                                |
--| +-------------------------------------------------------------------------+

    procedure Scan_Incomplete_Type_Declaration 
		 (The_Declaration : in Asis.Declaration; 
		  The_Context : in Context) is

	Puid : constant String := "Scan_Incomplete_Type_Declaration";

    begin

	declare
	    Discriminants : constant Asis.Discriminant_Specification_List := 
	       Asis_D.Discriminants (The_Declaration);
	    Obey_Region : Boolean;
	begin
	    -- A incomplete type is not always a region. Use Rgn operation 
	    -- to check it.
	    Obey_Region := Reference_Scan.Obeying_Regions and then 
			      Rgn.Denotes_One_Or_More_Regions (The_Declaration);

	    if Trace.On then
		Trace.Log (The_Declaration, The_Context);
	    end if;

	    if not Obey_Region then
		if Trace.On then
		    Trace.Add_Level (Discriminants'Length);
		end if;
		Reference_Scan.Scan_Declaration_List 
		   (Discriminants, The_Context);
	    end if;

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Rgn.Region_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Declaration);
	    raise;

    end Scan_Incomplete_Type_Declaration;

--| +-------------------------------------------------------------------------+
--| | SCAN_INTEGER_NUMBER_DECLARATION (local)                                 |
--| +-------------------------------------------------------------------------+

    procedure Scan_Integer_Number_Declaration 
		 (The_Declaration : in Asis.Declaration; 
		  The_Context : in Context) is

	Puid : constant String := "Scan_Integer_Number_Declaration";

    begin

	declare
	    Value : Asis.Expression := Asis_D.Initial_Value (The_Declaration);
	begin

	    if Trace.On then
		Trace.Log (The_Declaration, The_Context);
		Trace.Add_Level (1);
	    end if;

	    Reference_Scan.Scan_Expression 
	       (Value, Ctx.Set (The_Context, 
				Basic_Context => Rvs.A_Read, 
				Weight => 5, 
				Add_Data_Access_Context => 
				   Rvs.A_Number_Declaration));

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Declaration);
	    raise;

    end Scan_Integer_Number_Declaration;

--| +-------------------------------------------------------------------------+
--| | SCAN_LOOP_PARAMETER_SPECIFICATION (local)                               |
--| +-------------------------------------------------------------------------+

    procedure Scan_Loop_Parameter_Specification 
		 (The_Declaration : in Asis.Declaration; 
		  The_Context : in Context) is

	Puid : constant String := "Scan_Loop_Parameter_Specification";

    begin

	declare
	    The_Range : Asis.Discrete_Range := 
	       Asis_S.Loop_Parameter_Range (The_Declaration);
	begin

	    if Trace.On then
		Trace.Log (The_Declaration, The_Context);
		Trace.Add_Level (1);
	    end if;

	    Reference_Scan.Scan_Discrete_Range 
	       (The_Range, Ctx.Set (The_Context, 
				    Type_Mark_Context => 
				       Rvs.A_Loop_Parameter_Specification));

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Declaration);
	    raise;

    end Scan_Loop_Parameter_Specification;

--| +-------------------------------------------------------------------------+
--| | SCAN_OBJECT_RENAME_DECLARATION (local)                                  |
--| +-------------------------------------------------------------------------+

    procedure Scan_Object_Rename_Declaration 
		 (The_Declaration : in Asis.Declaration; 
		  The_Context : in Context) is

	Puid : constant String := "Scan_Object_Rename_Declaration";

    begin

	declare
	    Renamed_Object : Asis.Expression := 
	       Asis_D.Renamed_Entity (The_Declaration);
	    The_Type : Asis.Expression := Asis_D.Type_Mark (The_Declaration);
	begin

	    if Trace.On then
		Trace.Log (The_Declaration, The_Context);
		Trace.Add_Level (2);
	    end if;

	    Reference_Scan.Scan_Expression 
	       (The_Type, Ctx.Set 
			     (The_Context, 
			      Basic_Context => Rvs.A_Type_Mark, 
			      Weight => 90, 
			      Type_Mark_Context => Rvs.A_Renaming_Declaration));

	    Reference_Scan.Scan_Expression 
	       (Renamed_Object, Ctx.Set (The_Context, 
					 Basic_Context => Rvs.A_Rename, 
					 Weight => 90));

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Declaration);
	    raise;

    end Scan_Object_Rename_Declaration;

--| +-------------------------------------------------------------------------+
--| | SCAN_PACKAGE_BODY_DECLARATION (local)                                   |
--| +-------------------------------------------------------------------------+

    procedure Scan_Package_Body_Declaration 
		 (The_Declaration : in Asis.Declaration; 
		  The_Context : in Context) is

	Puid : constant String := "Scan_Package_Body_Declaration";

    begin

	declare
	    The_Body : Asis.Statement := 
	       Asis_D.Package_Body_Block (The_Declaration);
	begin

	    if Trace.On then
		Trace.Log (The_Declaration, The_Context);
	    end if;

	    if not Reference_Scan.Obeying_Regions then
		if Trace.On then
		    Trace.Add_Level (1);
		end if;
		Reference_Scan.Scan_Statement (The_Body, The_Context);
	    end if;

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Declaration);
	    raise;

    end Scan_Package_Body_Declaration;

--| +-------------------------------------------------------------------------+
--| | SCAN_PACKAGE_BODY_STUB (local)                                          |
--| +-------------------------------------------------------------------------+

    procedure Scan_Package_Body_Stub (The_Declaration : in Asis.Declaration; 
				      The_Context : in Context) is

	Puid : constant String := "Scan_Package_Body_Stub";

    begin

	if Trace.On then
	    Trace.Log (The_Declaration, The_Context);
	end if;

    end Scan_Package_Body_Stub;

--| +-------------------------------------------------------------------------+
--| | SCAN_PACKAGE_DECLARATION (local)                                        |
--| +-------------------------------------------------------------------------+

    procedure Scan_Package_Declaration (The_Declaration : in Asis.Declaration; 
					The_Context : in Context) is

	Puid : constant String := "Scan_Package_Declaration";

    begin

	declare
	    Private_Declarations : constant Asis.Declarative_Item_List := 
	       Asis_D.Private_Part_Declarative_Items 
		  (The_Declaration, Reference_Scan.Include_Package_Pragmas);
	    Public_Declarations : constant Asis.Declarative_Item_List := 
	       Asis_D.Visible_Part_Declarative_Items 
		  (The_Declaration, Reference_Scan.Include_Package_Pragmas);
	begin

	    if Trace.On then
		Trace.Log (The_Declaration, The_Context);
	    end if;

	    if not Reference_Scan.Obeying_Regions then
		if Trace.On then
		    Trace.Add_Level (Public_Declarations'Length + 
				     Private_Declarations'Length);
		end if;
		Reference_Scan.Scan_Declarative_Item_List 
		   (Public_Declarations, The_Context);
		Reference_Scan.Scan_Declarative_Item_List 
		   (Private_Declarations, The_Context);
	    end if;

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Declaration);
	    raise;

    end Scan_Package_Declaration;

--| +-------------------------------------------------------------------------+
--| | SCAN_PACKAGE_INSTANTIATION (local)                                      |
--| +-------------------------------------------------------------------------+

    procedure Scan_Package_Instantiation (The_Declaration : in Asis.Declaration; 
					  The_Context : in Context) is

	Puid : constant String := "Scan_Package_Instantiation";

    begin

	if Trace.On then
	    Trace.Log (The_Declaration, The_Context);
	end if;

	if Reference_Scan.Expand_Package_Instantiations then

	    if not Reference_Scan.Obeying_Regions then
		declare
		    Specification : Asis.Declaration := 
		       Asis_D.Corresponding_Specification (The_Declaration);
		    The_Body : Asis.Declaration := 
		       Asis_D.Corresponding_Body (The_Declaration);
		begin
		    if Trace.On then
			Trace.Add_Level 
			   (Reference_Scan.One_If_Present (Specification) + 
			    Reference_Scan.One_If_Present (The_Body));
		    end if;
		    if not Asis_E.Is_Nil (Specification) then
			Reference_Scan.Scan_Declaration 
			   (Specification, The_Context);
		    end if;
		    if not Asis_E.Is_Nil (The_Body) then
			Reference_Scan.Scan_Declaration (The_Body, The_Context);
		    end if;
		end;
	    end if;

	else

	    declare
		Arguments : constant Asis.Association_List := 
		   Asis_D.Generic_Parameters 
		      (The_Declaration, Reference_Scan.
					   Normalize_Instantiation_Parameters);
		The_Package : Asis.Expression := 
		   Asis_D.Generic_Unit_Name (The_Declaration);
	    begin
		if Trace.On then
		    Trace.Add_Level (1 + Arguments'Length);
		end if;
		Reference_Scan.Scan_Expression 
		   (The_Package, Ctx.Set (The_Context, 
					  Basic_Context => Rvs.An_Instantiation, 
					  Weight => 90));
		Reference_Scan.Scan_Parameter_Association_List 
		   (Arguments, Ctx.Set 
				  (The_Context, 
				   Basic_Context => Rvs.A_Generic_Association, 
				   Weight => 90));
	    end;

	end if;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Declaration);
	    raise;

    end Scan_Package_Instantiation;

--| +-------------------------------------------------------------------------+
--| | SCAN_PACKAGE_RENAME_DECLARATION (local)                                 |
--| +-------------------------------------------------------------------------+

    procedure Scan_Package_Rename_Declaration 
		 (The_Declaration : in Asis.Declaration; 
		  The_Context : in Context) is

	Puid : constant String := "Scan_Package_Rename_Declaration";

    begin

	declare
	    Renamed_Package : Asis.Expression := 
	       Asis_D.Renamed_Entity (The_Declaration);
	begin

	    if Trace.On then
		Trace.Log (The_Declaration, The_Context);
		Trace.Add_Level (1);
	    end if;

	    Reference_Scan.Scan_Expression 
	       (Renamed_Package, Ctx.Set (The_Context, 
					  Basic_Context => Rvs.A_Rename, 
					  Weight => 90));

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Declaration);
	    raise;

    end Scan_Package_Rename_Declaration;

--| +-------------------------------------------------------------------------+
--| | SCAN_PARAMETER_SPECIFICATION (local)                                    |
--| +-------------------------------------------------------------------------+

    procedure Scan_Parameter_Specification 
		 (The_Declaration : in Asis.Declaration; 
		  The_Context : in Context) is

	Puid : constant String := "Scan_Parameter_Specification";

    begin

	declare
	    Default_Value : Asis.Expression := 
	       Asis_D.Initial_Value (The_Declaration);
	    The_Type : Asis.Expression := Asis_D.Type_Mark (The_Declaration);
	begin

	    if Trace.On then
		Trace.Log (The_Declaration, The_Context);
		Trace.Add_Level (1 + 
				 Reference_Scan.One_If_Present (Default_Value));
	    end if;

	    Reference_Scan.Scan_Expression 
	       (The_Type, Ctx.Set (The_Context, 
				   Basic_Context => Rvs.A_Type_Mark, 
				   Weight => 90, 
				   Type_Mark_Context => 
				      Rvs.A_Parameter_Specification));

	    if not Asis_E.Is_Nil (Default_Value) then
		Reference_Scan.Scan_Expression 
		   (Default_Value, Ctx.Set (The_Context, 
					    Basic_Context => Rvs.A_Read, 
					    Weight => 5, 
					    Add_Data_Access_Context => 
					       Rvs.A_Parameter_Specification));
	    end if;

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Declaration);
	    raise;

    end Scan_Parameter_Specification;

--| +-------------------------------------------------------------------------+
--| | SCAN_PRIVATE_TYPE_DECLARATION (local)                                   |
--| +-------------------------------------------------------------------------+

    procedure Scan_Private_Type_Declaration 
		 (The_Declaration : in Asis.Declaration; 
		  The_Context : in Context) is

	Puid : constant String := "Scan_Private_Type_Declaration";

    begin

	declare
	    Discriminants : constant Asis.Discriminant_Specification_List := 
	       Asis_D.Discriminants (The_Declaration);
	    Obey_Region : Boolean;
	begin
	    -- A private type is not always a region. Use Rgn operation 
	    -- to check it.
	    Obey_Region := Reference_Scan.Obeying_Regions and then 
			      Rgn.Denotes_One_Or_More_Regions (The_Declaration);

	    if Trace.On then
		Trace.Log (The_Declaration, The_Context);
	    end if;

	    if not Obey_Region then
		if Trace.On then
		    Trace.Add_Level (Discriminants'Length);
		end if;
		Reference_Scan.Scan_Declaration_List 
		   (Discriminants, The_Context);
	    end if;

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Rgn.Region_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Declaration);
	    raise;

    end Scan_Private_Type_Declaration;

--| +-------------------------------------------------------------------------+
--| | SCAN_PROCEDURE_BODY_DECLARATION (local)                                 |
--| +-------------------------------------------------------------------------+

    procedure Scan_Procedure_Body_Declaration 
		 (The_Declaration : in Asis.Declaration; 
		  The_Context : in Context) is

	Puid : constant String := "Scan_Procedure_Body_Declaration";

    begin

	declare
	    Parameters : constant Asis.Parameter_Specification_List := 
	       Asis_D.Parameters (The_Declaration);
	    The_Body : Asis.Statement := 
	       Asis_D.Subprogram_Body_Block (The_Declaration);
	begin

	    if Trace.On then
		Trace.Log (The_Declaration, The_Context);
	    end if;

	    if not Reference_Scan.Obeying_Regions then
		if Trace.On then
		    Trace.Add_Level (Parameters'Length + 1);
		end if;
		Reference_Scan.Scan_Declaration_List (Parameters, The_Context);
		Reference_Scan.Scan_Statement (The_Body, The_Context);
	    end if;

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Declaration);
	    raise;

    end Scan_Procedure_Body_Declaration;

--| +-------------------------------------------------------------------------+
--| | SCAN_PROCEDURE_BODY_STUB (local)                                        |
--| +-------------------------------------------------------------------------+

    procedure Scan_Procedure_Body_Stub (The_Declaration : in Asis.Declaration; 
					The_Context : in Context) is

	Puid : constant String := "Scan_Procedure_Body_Stub";

    begin

	declare
	    Parameters : constant Asis.Parameter_Specification_List := 
	       Asis_D.Parameters (The_Declaration);
	begin

	    if Trace.On then
		Trace.Log (The_Declaration, The_Context);
	    end if;

	    if not Reference_Scan.Obeying_Regions then
		if Trace.On then
		    Trace.Add_Level (Parameters'Length);
		end if;
		Reference_Scan.Scan_Declaration_List (Parameters, The_Context);
	    end if;

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Declaration);
	    raise;

    end Scan_Procedure_Body_Stub;

--| +-------------------------------------------------------------------------+
--| | SCAN_PROCEDURE_DECLARATION (local)                                      |
--| +-------------------------------------------------------------------------+

    procedure Scan_Procedure_Declaration (The_Declaration : in Asis.Declaration; 
					  The_Context : in Context) is

	Puid : constant String := "Scan_Procedure_Declaration";

    begin

	declare
	    Parameters : constant Asis.Parameter_Specification_List := 
	       Asis_D.Parameters (The_Declaration);
	begin

	    if Trace.On then
		Trace.Log (The_Declaration, The_Context);
	    end if;

	    if not Reference_Scan.Obeying_Regions then
		if Trace.On then
		    Trace.Add_Level (Parameters'Length);
		end if;
		Reference_Scan.Scan_Declaration_List (Parameters, The_Context);
	    end if;

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Declaration);
	    raise;

    end Scan_Procedure_Declaration;

--| +-------------------------------------------------------------------------+
--| | SCAN_PROCEDURE_INSTANTIATION (local)                                    |
--| +-------------------------------------------------------------------------+

    procedure Scan_Procedure_Instantiation 
		 (The_Declaration : in Asis.Declaration; 
		  The_Context : in Context) is

	Puid : constant String := "Scan_Procedure_Instantiation";

    begin

	if Trace.On then
	    Trace.Log (The_Declaration, The_Context);
	end if;

	if Reference_Scan.Expand_Procedure_Instantiations then

	    if not Reference_Scan.Obeying_Regions then
		declare
		    Specification : Asis.Declaration := 
		       Asis_D.Corresponding_Specification (The_Declaration);
		    The_Body : Asis.Declaration := 
		       Asis_D.Corresponding_Body (The_Declaration);
		begin
		    if Trace.On then
			Trace.Add_Level 
			   (Reference_Scan.One_If_Present (Specification) + 
			    Reference_Scan.One_If_Present (The_Body));
		    end if;
		    if not Asis_E.Is_Nil (Specification) then
			Reference_Scan.Scan_Declaration 
			   (Specification, The_Context);
		    end if;
		    if not Asis_E.Is_Nil (The_Body) then
			Reference_Scan.Scan_Declaration (The_Body, The_Context);
		    end if;
		end;
	    end if;

	else

	    declare
		Arguments : constant Asis.Association_List := 
		   Asis_D.Generic_Parameters 
		      (The_Declaration, Reference_Scan.
					   Normalize_Instantiation_Parameters);
		The_Procedure : Asis.Expression := 
		   Asis_D.Generic_Unit_Name (The_Declaration);
	    begin
		if Trace.On then
		    Trace.Add_Level (1 + Arguments'Length);
		end if;
		Reference_Scan.Scan_Expression 
		   (The_Procedure, 
		    Ctx.Set (The_Context, 
			     Basic_Context => Rvs.An_Instantiation, 
			     Weight => 90));
		Reference_Scan.Scan_Parameter_Association_List 
		   (Arguments, Ctx.Set 
				  (The_Context, 
				   Basic_Context => Rvs.A_Generic_Association, 
				   Weight => 90));
	    end;

	end if;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Declaration);
	    raise;

    end Scan_Procedure_Instantiation;

--| +-------------------------------------------------------------------------+
--| | SCAN_PROCEDURE_RENAME_DECLARATION (local)                               |
--| +-------------------------------------------------------------------------+

    procedure Scan_Procedure_Rename_Declaration 
		 (The_Declaration : in Asis.Declaration; 
		  The_Context : in Context) is

	Puid : constant String := "Scan_Procedure_Rename_Declaration";

    begin

	declare
	    Parameters : constant Asis.Parameter_Specification_List := 
	       Asis_D.Parameters (The_Declaration);
	    Renamed_Procedure : Asis.Expression := 
	       Asis_D.Renamed_Entity (The_Declaration);
	    Obey_Region : Boolean;
	begin
	    -- A procedure rename is not always a region. Use Rgn operation 
	    -- to check it.
	    Obey_Region := Reference_Scan.Obeying_Regions and then 
			      Rgn.Denotes_One_Or_More_Regions (The_Declaration);

	    if Trace.On then
		Trace.Log (The_Declaration, The_Context);
		Trace.Add_Level (Reference_Scan.One_If_True (not Obey_Region) * 
				 Parameters'Length + 1);
	    end if;

	    if not Obey_Region then
		Reference_Scan.Scan_Declaration_List (Parameters, The_Context);
	    end if;

	    Reference_Scan.Scan_Expression 
	       (Renamed_Procedure, Ctx.Set (The_Context, 
					    Basic_Context => Rvs.A_Rename, 
					    Weight => 90));

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Rgn.Region_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Declaration);
	    raise;

    end Scan_Procedure_Rename_Declaration;

--| +-------------------------------------------------------------------------+
--| | SCAN_REAL_NUMBER_DECLARATION (local)                                    |
--| +-------------------------------------------------------------------------+

    procedure Scan_Real_Number_Declaration 
		 (The_Declaration : in Asis.Declaration; 
		  The_Context : in Context) is

	Puid : constant String := "Scan_Real_Number_Declaration";

    begin

	declare
	    Value : Asis.Expression := Asis_D.Initial_Value (The_Declaration);
	begin

	    if Trace.On then
		Trace.Log (The_Declaration, The_Context);
		Trace.Add_Level (1);
	    end if;

	    Reference_Scan.Scan_Expression 
	       (Value, Ctx.Set (The_Context, 
				Basic_Context => Rvs.A_Read, 
				Weight => 5, 
				Add_Data_Access_Context => 
				   Rvs.A_Number_Declaration));

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Declaration);
	    raise;

    end Scan_Real_Number_Declaration;

--| +-------------------------------------------------------------------------+
--| | SCAN_SUBTYPE_DECLARATION (local)                                        |
--| +-------------------------------------------------------------------------+

    procedure Scan_Subtype_Declaration (The_Declaration : in Asis.Declaration; 
					The_Context : in Context) is

	Puid : constant String := "Scan_Subtype_Declaration";

    begin

	declare
	    The_Type : Asis.Type_Definition := 
	       Asis_D.Type_Declaration_Definition (The_Declaration);
	begin

	    if Trace.On then
		Trace.Log (The_Declaration, The_Context);
		Trace.Add_Level (1);
	    end if;

	    Reference_Scan.Scan_Type_Definition 
	       (The_Type, Ctx.Set 
			     (The_Context, 
			      Type_Mark_Context => Rvs.A_Subtype_Declaration));

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Declaration);
	    raise;

    end Scan_Subtype_Declaration;

--| +-------------------------------------------------------------------------+
--| | SCAN_TASK_BODY_DECLARATION (local)                                      |
--| +-------------------------------------------------------------------------+

    procedure Scan_Task_Body_Declaration (The_Declaration : in Asis.Declaration; 
					  The_Context : in Context) is

	Puid : constant String := "Scan_Task_Body_Declaration";

    begin

	declare
	    The_Body : Asis.Statement := 
	       Asis_D.Task_Body_Block (The_Declaration);
	begin

	    if Trace.On then
		Trace.Log (The_Declaration, The_Context);
	    end if;

	    if not Reference_Scan.Obeying_Regions then
		if Trace.On then
		    Trace.Add_Level (1);
		end if;
		Reference_Scan.Scan_Statement (The_Body, The_Context);
	    end if;

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Declaration);
	    raise;

    end Scan_Task_Body_Declaration;

--| +-------------------------------------------------------------------------+
--| | SCAN_TASK_BODY_STUB (local)                                             |
--| +-------------------------------------------------------------------------+

    procedure Scan_Task_Body_Stub (The_Declaration : in Asis.Declaration; 
				   The_Context : in Context) is

	Puid : constant String := "Scan_Task_Body_Stub";

    begin

	if Trace.On then
	    Trace.Log (The_Declaration, The_Context);
	end if;

    end Scan_Task_Body_Stub;

--| +-------------------------------------------------------------------------+
--| | SCAN_TASK_DECLARATION (local)                                           |
--| +-------------------------------------------------------------------------+

    procedure Scan_Task_Declaration (The_Declaration : in Asis.Declaration; 
				     The_Context : in Context) is

	Puid : constant String := "Scan_Task_Declaration";

    begin

	declare
	    Declarations : constant Asis.Declarative_Item_List := 
	       Asis_D.Task_Declaration_Declarative_Items 
		  (The_Declaration, Reference_Scan.
				       Include_Task_Declaration_Pragmas);
	begin

	    if Trace.On then
		Trace.Log (The_Declaration, The_Context);
	    end if;

	    if not Reference_Scan.Obeying_Regions then
		if Trace.On then
		    Trace.Add_Level (Declarations'Length);
		end if;
		Reference_Scan.Scan_Declarative_Item_List 
		   (Declarations, The_Context);
	    end if;

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Declaration);
	    raise;

    end Scan_Task_Declaration;

--| +-------------------------------------------------------------------------+
--| | SCAN_TASK_TYPE_DECLARATION (local)                                      |
--| +-------------------------------------------------------------------------+

    procedure Scan_Task_Type_Declaration (The_Declaration : in Asis.Declaration; 
					  The_Context : in Context) is

	Puid : constant String := "Scan_Task_Type_Declaration";

    begin

	declare
	    The_Type : Asis.Type_Definition := 
	       Asis_D.Type_Declaration_Definition (The_Declaration);
	begin

	    if Trace.On then
		Trace.Log (The_Declaration, The_Context);
	    end if;

	    if not Reference_Scan.Obeying_Regions then
		if Trace.On then
		    Trace.Add_Level (1);
		end if;
		Reference_Scan.Scan_Type_Definition (The_Type, The_Context);
	    end if;

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Declaration);
	    raise;

    end Scan_Task_Type_Declaration;

--| +-------------------------------------------------------------------------+
--| | SCAN_VARIABLE_DECLARATION (local)                                       |
--| +-------------------------------------------------------------------------+

    procedure Scan_Variable_Declaration (The_Declaration : in Asis.Declaration; 
					 The_Context : in Context) is

	Puid : constant String := "Scan_Variable_Declaration";

    begin

	declare
	    Initial_Value : Asis.Expression := 
	       Asis_D.Initial_Value (The_Declaration);
	    The_Type : Asis.Type_Definition := 
	       Asis_D.Object_Declaration_Definition (The_Declaration);
	begin

	    if Trace.On then
		Trace.Log (The_Declaration, The_Context);
		Trace.Add_Level (1 + 
				 Reference_Scan.One_If_Present (Initial_Value));
	    end if;

	    Reference_Scan.Scan_Type_Definition 
	       (The_Type, Ctx.Set 
			     (The_Context, 
			      Type_Mark_Context => Rvs.An_Object_Declaration));

	    if not Asis_E.Is_Nil (Initial_Value) then
		Reference_Scan.Scan_Expression 
		   (Initial_Value, Ctx.Set (The_Context, 
					    Basic_Context => Rvs.A_Read, 
					    Weight => 5, 
					    Add_Data_Access_Context => 
					       Rvs.An_Object_Declaration));
	    end if;

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Declaration);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Declaration);
	    raise;

    end Scan_Variable_Declaration;

--| +-------------------------------------------------------------------------+
--| | SCAN_DECLARATION (exported)                                             |
--| +-------------------------------------------------------------------------+

begin

    case Asis_D.Kind (The_Declaration) is
	when Asis_D.A_Variable_Declaration =>
	    Scan_Variable_Declaration (The_Declaration, The_Context);
	when Asis_D.A_Component_Declaration =>
	    Scan_Component_Declaration (The_Declaration, The_Context);
	when Asis_D.A_Constant_Declaration =>
	    Scan_Constant_Declaration (The_Declaration, The_Context);
	when Asis_D.A_Deferred_Constant_Declaration =>
	    Scan_Deferred_Constant_Declaration (The_Declaration, The_Context);
	when Asis_D.A_Generic_Formal_Object_Declaration =>
	    Scan_Generic_Formal_Object_Declaration 
	       (The_Declaration, The_Context);
	when Asis_D.A_Discriminant_Specification =>
	    Scan_Discriminant_Specification (The_Declaration, The_Context);
	when Asis_D.A_Parameter_Specification =>
	    Scan_Parameter_Specification (The_Declaration, The_Context);
	when Asis_D.An_Integer_Number_Declaration =>
	    Scan_Integer_Number_Declaration (The_Declaration, The_Context);
	when Asis_D.A_Real_Number_Declaration =>
	    Scan_Real_Number_Declaration (The_Declaration, The_Context);
	when Asis_D.An_Exception_Declaration =>
	    Scan_Exception_Declaration (The_Declaration, The_Context);
	when Asis_D.An_Enumeration_Literal_Specification =>
	    Scan_Enumeration_Literal_Specification 
	       (The_Declaration, The_Context);
	when Asis_D.A_Loop_Parameter_Specification =>
	    Scan_Loop_Parameter_Specification (The_Declaration, The_Context);
	when Asis_D.A_Full_Type_Declaration =>
	    Scan_Full_Type_Declaration (The_Declaration, The_Context);
	when Asis_D.An_Incomplete_Type_Declaration =>
	    Scan_Incomplete_Type_Declaration (The_Declaration, The_Context);
	when Asis_D.A_Private_Type_Declaration =>
	    Scan_Private_Type_Declaration (The_Declaration, The_Context);
	when Asis_D.A_Subtype_Declaration =>
	    Scan_Subtype_Declaration (The_Declaration, The_Context);
	when Asis_D.A_Package_Declaration =>
	    Scan_Package_Declaration (The_Declaration, The_Context);
	when Asis_D.A_Package_Body_Declaration =>
	    Scan_Package_Body_Declaration (The_Declaration, The_Context);
	when Asis_D.A_Procedure_Declaration =>
	    Scan_Procedure_Declaration (The_Declaration, The_Context);
	when Asis_D.A_Procedure_Body_Declaration =>
	    Scan_Procedure_Body_Declaration (The_Declaration, The_Context);
	when Asis_D.A_Function_Declaration =>
	    Scan_Function_Declaration (The_Declaration, The_Context);
	when Asis_D.A_Function_Body_Declaration =>
	    Scan_Function_Body_Declaration (The_Declaration, The_Context);
	when Asis_D.An_Object_Rename_Declaration =>
	    Scan_Object_Rename_Declaration (The_Declaration, The_Context);
	when Asis_D.An_Exception_Rename_Declaration =>
	    Scan_Exception_Rename_Declaration (The_Declaration, The_Context);
	when Asis_D.A_Package_Rename_Declaration =>
	    Scan_Package_Rename_Declaration (The_Declaration, The_Context);
	when Asis_D.A_Procedure_Rename_Declaration =>
	    Scan_Procedure_Rename_Declaration (The_Declaration, The_Context);
	when Asis_D.A_Function_Rename_Declaration =>
	    Scan_Function_Rename_Declaration (The_Declaration, The_Context);
	when Asis_D.A_Generic_Package_Declaration =>
	    Scan_Generic_Package_Declaration (The_Declaration, The_Context);
	when Asis_D.A_Generic_Procedure_Declaration =>
	    Scan_Generic_Procedure_Declaration (The_Declaration, The_Context);
	when Asis_D.A_Generic_Function_Declaration =>
	    Scan_Generic_Function_Declaration (The_Declaration, The_Context);
	when Asis_D.A_Package_Instantiation =>
	    Scan_Package_Instantiation (The_Declaration, The_Context);
	when Asis_D.A_Procedure_Instantiation =>
	    Scan_Procedure_Instantiation (The_Declaration, The_Context);
	when Asis_D.A_Function_Instantiation =>
	    Scan_Function_Instantiation (The_Declaration, The_Context);
	when Asis_D.A_Task_Declaration =>
	    Scan_Task_Declaration (The_Declaration, The_Context);
	when Asis_D.A_Task_Body_Declaration =>
	    Scan_Task_Body_Declaration (The_Declaration, The_Context);
	when Asis_D.A_Task_Type_Declaration =>
	    Scan_Task_Type_Declaration (The_Declaration, The_Context);
	when Asis_D.An_Entry_Declaration =>
	    Scan_Entry_Declaration (The_Declaration, The_Context);
	when Asis_D.A_Procedure_Body_Stub =>
	    Scan_Procedure_Body_Stub (The_Declaration, The_Context);
	when Asis_D.A_Function_Body_Stub =>
	    Scan_Function_Body_Stub (The_Declaration, The_Context);
	when Asis_D.A_Package_Body_Stub =>
	    Scan_Package_Body_Stub (The_Declaration, The_Context);
	when Asis_D.A_Task_Body_Stub =>
	    Scan_Task_Body_Stub (The_Declaration, The_Context);
	when Asis_D.A_Generic_Formal_Type_Declaration =>
	    Scan_Generic_Formal_Type_Declaration (The_Declaration, The_Context);
	when Asis_D.A_Generic_Formal_Private_Type_Declaration =>
	    Scan_Generic_Formal_Private_Type_Declaration 
	       (The_Declaration, The_Context);
	when Asis_D.A_Generic_Formal_Procedure_Declaration =>
	    Scan_Generic_Formal_Procedure_Declaration 
	       (The_Declaration, The_Context);
	when Asis_D.A_Generic_Formal_Function_Declaration =>
	    Scan_Generic_Formal_Function_Declaration 
	       (The_Declaration, The_Context);
	when Asis_D.Not_A_Declaration =>
	    Error.Log (Error.An_Unhandled_Case, Cuid, Puid, The_Declaration);
	    if Reference_Scan.Raise_Exception_On_Unhandled_Case then
		raise Traversal_Error;
	    else
		if Trace.On then
		    Trace.Log (The_Declaration, The_Context);
		end if;
	    end if;
    end case;

end Scan_Declaration;
