-- ============================================================================
-- >>>>>>>>>>>>>>>>>>>>>>>>>> ADA COMPILATION UNIT <<<<<<<<<<<<<<<<<<<<<<<<<<<<
-- ============================================================================
--
-- NAME:        Region_View_Structures
--
--              SPECIFICATION
-- 
-- AUTHOR:      Chuck Hobin
--              General Research Corporation
--
--                               CHANGE HISTORY
--
-- MM-DD-YY | Initials | Description
-- ----------------------------------------------------------------------------
-- <include SPR#, if applicable>
-- ============================================================================

with Asis;
with Hash_Asis_Element_To_Positive;
with Map_Simple_Noncached_Sequential_Unbounded_Managed_Iterator_Semilimited_Domain;
with Region_Support;
with Stack_Sequential_Unbounded_Managed_Iterator;

package Region_View_Structures is

--| This package declares the data structures used to represent a Region view.
--| An object of the type Region_View (see end of unit) is created by the 
--| operations of package Region_View.
--| 
--| A Region view identifies the set of declarative regions occurring in a 
--| Library Unit Group (LUG), i.e, a library unit together with is corresponding
--| body and subunits, if any.  The view organizes the regions as a tree, 
--| reflecting the nesting relationships of the regions in the source code.
--| 
--| A declarative region consists of one or more region parts.  Each region part
--| is a physical, contiguous text fragment residing in a single compilation unit.
--| The region parts within a compilation unit exhibit their own nesting 
--| relationships; these are represented here as a second tree structure.
--| 
--| See the package Region_Support for more background on declarative regions
--| and region parts.


    type Region_Part;

    type Region_Part_Ptr is access Region_Part;

    type Region_Part_List_Element;

    type Region_Part_List is access Region_Part_List_Element;

    type Region_Part is
	record
	    Region : Region_Support.Region;
	    --| The Region object which identifies this region part.
	    --| The head element of this object is the declaration, 
	    --| statement, or rep clause element that denotes the part.

	    Parent : Region_Part_Ptr;
	    --| If non-null, Parent is the region part that immediately
	    --| encloses this region part.  Regions of kind A_Compilation_Unit
	    --| always have a null Parent; all others have a non-null Parent.

	    Children : Region_Part_List;
	    --| If non-null, Children is the set of region parts that are
	    --| immediately enclosed by this region part.  The children
	    --| appear in source code order.
	end record;

    type Region_Part_List_Element is
	record
	    Region_Part : Region_Part_Ptr;
	    Next : Region_Part_List;
	end record;

    --| The kinds of declarative regions, per LRM 8.1:

    type Declarative_Region_Kinds is 
       (A_Package, A_Generic_Package, A_Subprogram, A_Generic_Subprogram, 
	A_Task, An_Entry, A_Record_Type, A_Rename, A_Generic_Formal_Subprogram, 
	A_Generic_Formal_Private_Type, A_Block_Statement, A_Loop_Statement);

    type Declarative_Region (Kind : Declarative_Region_Kinds);

    type Declarative_Region_Ptr is access Declarative_Region;

    type Declarative_Region_List_Element;

    type Declarative_Region_List is access Declarative_Region_List_Element;

    type Declarative_Region (Kind : Declarative_Region_Kinds) is
	record
	    Defining_Element : Asis.Element;
	    --| The declaration or statement element with which
	    --| this declarative region is associated.  This is also
	    --| the head element of one or more region parts belonging 
	    --| to the region.

	    Parent : Declarative_Region_Ptr;
	    --| If non-null, Parent is the region that immediately
	    --| encloses this region.

	    Children : Declarative_Region_List;
	    --| If non-null, Children is the set of regions that are
	    --| immediately enclosed by this region.

	    --| Each declarative region consists of one or more region parts.
	    --| The following variant components identify the possible 
	    --| region parts that may be present for each kind of region.
	    --| Note that some of these components may be null.
	    case Kind is
		when A_Package | A_Generic_Package =>

		    Package_Visible_Region_Part : Region_Part_Ptr;
		    Package_Private_Region_Part : Region_Part_Ptr;
		    Package_Stub_Region_Part : Region_Part_Ptr;
		    Package_Body_Region_Part : Region_Part_Ptr;
		    case Kind is
			when A_Generic_Package =>
			    Generic_Package_Formal_Part : Region_Part_Ptr;
			when others =>
			    null;
		    end case;

		when A_Subprogram | A_Generic_Subprogram =>

		    Subprogram_Spec_Region_Part : Region_Part_Ptr;
		    Subprogram_Stub_Region_Part : Region_Part_Ptr;
		    Subprogram_Body_Region_Part : Region_Part_Ptr;
		    case Kind is
			when A_Generic_Subprogram =>
			    Generic_Subprogram_Formal_Part : Region_Part_Ptr;
			when others =>
			    null;
		    end case;

		when A_Task =>

		    Task_Spec_Region_Part : Region_Part_Ptr;
		    Task_Stub_Region_Part : Region_Part_Ptr;
		    Task_Body_Region_Part : Region_Part_Ptr;

		when An_Entry =>

		    Entry_Declaration_Region_Part : Region_Part_Ptr;
		    Accept_Statement_Region_Parts : Region_Part_List;
		    --| There can be multiple accept statements associated
		    --| with an entry; they appear in the list in source
		    --| code order.

		when A_Record_Type =>

		    Private_Or_Incomplete_Region_Part : Region_Part_Ptr;
		    --| A record type declaration can have a corresponding
		    --| private type declaration or incomplete type 
		    --| declaration, but not both.  
		    Record_Type_Region_Part : Region_Part_Ptr;
		    Record_Rep_Clause_Region_Part : Region_Part_Ptr;

		when A_Rename =>

		    Rename_Region_Part : Region_Part_Ptr;

		when A_Generic_Formal_Subprogram =>

		    Generic_Formal_Subprogram_Region_Part : Region_Part_Ptr;

		when A_Generic_Formal_Private_Type =>

		    Generic_Formal_Private_Type_Region_Part : Region_Part_Ptr;

		when A_Block_Statement =>

		    Block_Statement_Region_Part : Region_Part_Ptr;

		when A_Loop_Statement =>

		    Loop_Statement_Region_Part : Region_Part_Ptr;

	    end case;
	end record;

    type Declarative_Region_List_Element is
	record
	    Declarative_Region : Declarative_Region_Ptr;
	    Next : Declarative_Region_List;
	end record;

    --| An object of type Comp_Unit is created for each compilation unit
    --| that is scanned.  

    type Comp_Unit is
	record
	    Region_Part_Tree : Region_Part_Ptr;
	    --| The tree of region parts residing in the comp unit.
	    --| The root of this tree always has the kind
	    --| A_Compilation_Unit.

	    Asis_Comp_Unit : Asis.Compilation_Unit;
	    --| The ASIS unit that was scanned.
	end record;

    type Comp_Unit_Ptr is access Comp_Unit;

    type Comp_Unit_List_Element;

    type Comp_Unit_List is access Comp_Unit_List_Element;

    type Comp_Unit_List_Element is
	record
	    The_Unit : Comp_Unit_Ptr;
	    Next : Comp_Unit_List;
	end record;

    package Element_Region_Map is 
       new Map_Simple_Noncached_Sequential_Unbounded_Managed_Iterator_Semilimited_Domain 
	      (Domain => Asis.Element, 
	       Is_Equal => Asis.Elements.Is_Equal, 
	       Ranges => Declarative_Region_Ptr, 
	       Number_Of_Buckets => 1009,   -- chosen arbitrarily!
	       Hash_Of => Hash_Asis_Element_To_Positive);

    package Rp_Stack is new Stack_Sequential_Unbounded_Managed_Iterator 
			       (Item => Region_Part_Ptr);

    type Region_View_Record is
	record
	    Declarative_Region_Tree : Declarative_Region_Ptr;
	    --| The root of this tree is the outermost region of the
	    --| library unit.

	    Comp_Units : Comp_Unit_List;
	    --| The set of compilation units that have been scanned 
	    --| to create the view.

	    Head_Element_Map : Element_Region_Map.Map;
	    --| This map contains an entry for the head element
	    --| of each region part that has been identified.  
	    --| It associates a region part with the 
	    --| declarative region it is part of.  
	    --|
	    --| The map is used during view construction. Since
	    --| it may also be useful to the view user, it is
	    --| left as an artifact. 

	    Current_Comp_Unit : Comp_Unit_Ptr;
	    --| The compilation unit currently being scanned.  Has
	    --| a null value when view construction is completed.

	    Region_Part_Stack : Rp_Stack.Stack;
	    --| Stack containing the region parts currently being
	    --| scanned in Current_Comp_Unit.  The part on top
	    --| of the stack is the innermost part being scanned.
	    --| The stack is empty when view construction is
	    --| completed.
	end record;

    type Region_View is access Region_View_Record;

end Region_View_Structures;

