--| +=========================================================================+
--| |                                                                         |
--| | REGION_SCAN.REGION_SCAN_INTERNAL.SCAN_REPRESENTATION_CLAUSE (body)      |
--| |                                                                         |
--| | Greg Janee                                                              |
--| | General Research Corporation                                            |
--| |                                                                         |
--| +=========================================================================+

separate (Region_Scan.Region_Scan_Internal)
procedure Scan_Representation_Clause 
	     (The_Clause : in Asis.Representation_Clause; 
	      The_Context : in out Context) is

    Cuid : constant String := 
       "Region_Scan.Region_Scan_Internal.Scan_Representation_Clause";
    Puid : constant String := "Scan_Representation_Clause";

    package Error renames Region_Scan_Internal.Error_Handling_Support;

--| +-------------------------------------------------------------------------+
--| | SCAN_ADDRESS_CLAUSE (local)                                             |
--| +-------------------------------------------------------------------------+

    procedure Scan_Address_Clause (The_Clause : in Asis.Representation_Clause; 
				   The_Context : in out Context) is

	Puid : constant String := "Scan_Address_Clause";

    begin

	declare
	    Address : Asis.Expression := 
	       Asis_Rc.Address_Clause_Simple_Expression (The_Clause);
	    Object_Name : Asis.Simple_Name := 
	       Asis_Rc.Address_Clause_Simple_Name (The_Clause);
	begin

	    Region_Scan_Internal.Scan_Expression (Object_Name, The_Context);

	    Region_Scan_Internal.Scan_Expression (Address, The_Context);

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Clause);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Clause);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Clause);
	    raise;

    end Scan_Address_Clause;

--| +-------------------------------------------------------------------------+
--| | SCAN_ENUMERATION_REPRESENTATION_CLAUSE (local)                          |
--| +-------------------------------------------------------------------------+

    procedure Scan_Enumeration_Representation_Clause 
		 (The_Clause : in Asis.Representation_Clause; 
		  The_Context : in out Context) is

	Puid : constant String := "Scan_Enumeration_Representation_Clause";

    begin

	declare
	    Aggregate : Asis.Expression := 
	       Asis_Rc.Enumeration_Representation_Clause_Aggregate (The_Clause);
	    Type_Name : Asis.Simple_Name := 
	       Asis_Rc.Enumeration_Representation_Clause_Type_Simple_Name 
		  (The_Clause);
	begin

	    Region_Scan_Internal.Scan_Expression (Type_Name, The_Context);

	    Region_Scan_Internal.Scan_Expression (Aggregate, The_Context);

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Clause);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Clause);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Clause);
	    raise;

    end Scan_Enumeration_Representation_Clause;

--| +-------------------------------------------------------------------------+
--| | SCAN_LENGTH_CLAUSE (local)                                              |
--| +-------------------------------------------------------------------------+

    procedure Scan_Length_Clause (The_Clause : in Asis.Representation_Clause; 
				  The_Context : in out Context) is

	Puid : constant String := "Scan_Length_Clause";

    begin

	declare
	    Attribute : Asis.Expression := 
	       Asis_Rc.Length_Clause_Attribute (The_Clause);
	    Length : Asis.Expression := 
	       Asis_Rc.Length_Clause_Simple_Expression (The_Clause);
	begin

	    Region_Scan_Internal.Scan_Expression (Attribute, The_Context);

	    Region_Scan_Internal.Scan_Expression (Length, The_Context);

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Clause);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Clause);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Clause);
	    raise;

    end Scan_Length_Clause;

--| +-------------------------------------------------------------------------+
--| | SCAN_RECORD_REPRESENTATION_CLAUSE (local)                               |
--| +-------------------------------------------------------------------------+

    procedure Scan_Record_Representation_Clause 
		 (The_Clause : in Asis.Representation_Clause; 
		  The_Context : in out Context) is

	Puid : constant String := "Scan_Record_Representation_Clause";

    begin

	declare
	    Alignment : Asis.Expression := 
	       Asis_Rc.Record_Representation_Clause_Alignment_Clause_Expression 
		  (The_Clause);
	    Component_Clauses : constant Asis.Component_Clause_List := 
	       Asis_Rc.Component_Clauses 
		  (The_Clause, Region_Scan_Internal.
				  Include_Record_Representation_Clause_Pragmas);
	    Pragmas : constant Asis.Element_List := 
	       Asis_Rc.Record_Representation_Clause_Alignment_Clause_Pragmas 
		  (The_Clause);
	    Type_Name : Asis.Simple_Name := 
	       Asis_Rc.Record_Representation_Clause_Type_Simple_Name 
		  (The_Clause);
	begin

	    Region_Scan_Internal.Scan_Expression (Type_Name, The_Context);

	    if Region_Scan_Internal.
	       Include_Record_Representation_Clause_Pragmas then
		Region_Scan_Internal.Scan_Pragma_List (Pragmas, The_Context);
	    end if;

	    if not Asis_E.Is_Nil (Alignment) then
		Region_Scan_Internal.Scan_Expression (Alignment, The_Context);
	    end if;

	    Region_Scan_Internal.Scan_Component_Clause_List 
	       (Component_Clauses, The_Context);

	end;

    exception

	when Asis.Asis_Inappropriate_Element =>
	    Error.Log (Error.A_Bad_Element, Cuid, Puid, The_Clause);
	    raise Traversal_Error;

	when Asis.Asis_Failed =>
	    Error.Log (Error.An_Asis_Failure, Cuid, Puid, The_Clause);
	    raise Traversal_Error;

	when Traversal_Error =>
	    Error.Log (Error.A_Previous_Error, Cuid, Puid, The_Clause);
	    raise;

    end Scan_Record_Representation_Clause;

--| +-------------------------------------------------------------------------+
--| | SCAN_REPRESENTATION_CLAUSE (exported)                                   |
--| +-------------------------------------------------------------------------+

begin

    if Rgn.Denotes_One_Or_More_Regions (The_Clause, 
					Region_Scan.Expand_Instantiations) then
	declare
	    Remaining_Parent_Elements : constant Asis.Element_List := 
	       Rgn.Non_Region_Subelements (The_Clause, 
					   Region_Scan.Expand_Instantiations);
	    Regions : constant Rgn.Region_List := 
	       Rgn.Denoted_Regions (The_Clause, 
				    Region_Scan.Expand_Instantiations, 
				    Region_Scan.Include_Instance_Bodies);
	begin
	    Region_Scan_Internal.Scan_Any_List 
	       (Remaining_Parent_Elements, The_Context);
	    for I in Regions'Range loop
		Region_Scan.Process_Region (Regions (I), The_Context);
	    end loop;
	end;
	return;
    end if;

    case Asis_Rc.Kind (The_Clause) is
	when Asis_Rc.A_Length_Clause =>
	    Scan_Length_Clause (The_Clause, The_Context);
	when Asis_Rc.An_Enumeration_Representation_Clause =>
	    Scan_Enumeration_Representation_Clause (The_Clause, The_Context);
	when Asis_Rc.A_Record_Representation_Clause =>
	    Scan_Record_Representation_Clause (The_Clause, The_Context);
	when Asis_Rc.An_Address_Clause =>
	    Scan_Address_Clause (The_Clause, The_Context);
	when Asis_Rc.Not_A_Representation_Clause =>
	    Error.Log (Error.An_Unhandled_Case, Cuid, Puid, The_Clause);
	    if Region_Scan_Internal.Raise_Exception_On_Unhandled_Case then
		raise Traversal_Error;
	    end if;
    end case;

end Scan_Representation_Clause;
